# svs_simulation.ai_lib.logicrules

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Generic logic for AI.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities.lib import Constants


#######################
# CONSTANTS
#######################
logic_const = Constants()

# operands
logic_const.NO_OP = 200
logic_const.AND = 201
logic_const.OR = 202

# symbols
logic_const.WEIGHT_OPEN = "("
logic_const.WEIGHT_CLOSE = ")"


#######################
# EXCEPTIONS
#######################
class LogicRuleParsingException(Exception):
    pass
    

#######################
# RULE CLASS
#######################
class LogicRule:
    def __init__(self, weighting=1.0):
        self.conditions = []
        self.consequents = []
        self.weighting = weighting
        
    def addCondition(self, property, value, operand=logic_const.NO_OP):
        self.conditions.append((property, value, operand))
    
    def addConsequent(self, property, value, operand=logic_const.NO_OP, weighting=1.0):
        print "addConsequent: %s, %s" % (property, value)
        self.consequents.append((property, value, operand, weighting))
        
    def evaluate(self):
        pass
    
#######################
# INFERENCE CLASS
#######################
class Inference:
    def __init__(self, input=None, value=None):
        self.input = input
        self.value = value
        self.rules = []
        
    def addRule(self, newRule):
        """
        Adds rule to inference.
        
        """
        self.rules.append(newRule)
        
        
#######################
# UTILITY FUNCTIONS
#######################
def parseRuleString(rStr, rule):
    """
    Parses string and return L{LogicRule} object.
    """
    import string
    
    tokens = string.split(rStr)
    if not tokens:
        raise LogicRuleParsingException("Invalid string.")
        
    totalTokens = len(tokens)
    
    if totalTokens < 4:
        raise LogicRuleParsingException("Insufficient tokens in string.")
    
    if string.lower(tokens[0]) != "if":
        raise LogicRuleParsingException("Malformed string.")
        
    i = 0
    
    tmp = []
    
    # conditions
    while(i < totalTokens):
        tkn = tokens[i]
        i += 1
        if string.lower(tkn) == "then":
            condition = formatExpressionTokens(tmp)
            rule.addCondition(condition[0], condition[1], logic_const.NO_OP)
            tmp = []
            break
        elif string.lower(tkn) == "and":
            condition = formatExpressionTokens(tmp)
            rule.addCondition(condition[0], condition[1], logic_const.AND)
            tmp = []
        elif string.lower(tkn) == "or":
            condition = formatExpressionTokens(tmp)
            rule.addCondition(condition[0], condition[1], logic_const.OR)
            tmp = []
        elif string.lower(tkn) != "if":
            tmp.append(tkn)
        
    tmp = []
    
    # consequents
    while(i < totalTokens):
        tkn = tokens[i]
        i += 1
        if string.lower(tkn) == "and":
            if len(tmp) > 1:
                weighting = parseWeighting(tmp[-1])
                if weighting:
                    consequent = formatExpressionTokens(tmp[0:-1])
                else:
                    consequent = formatExpressionTokens(tmp)
            else:
                weighting = 1.0
                consequent = tmp[0]
            rule.addConsequent(consequent[0], consequent[1], logic_const.AND, weighting)
            tmp = []
        elif string.lower(tkn) == "or":
            if len(tmp) > 1:
                weighting = parseWeighting(tmp[-1])
                if weighting:
                    consequent = formatExpressionTokens(tmp[0:-1])
                else:
                    consequent = formatExpressionTokens(tmp)
            else:
                weighting = 1.0
                consequent = tmp[0]
            rule.addConsequent(consequent[0], consequent[1], logic_const.OR, weighting)
            tmp = []
        elif string.lower(tkn) != "if":
            print "tkn: <%s>" % tkn
            tmp.append(tkn)
        
    if len(tmp) == 0:
        return rule
    
    if len(tmp) > 1:
        weighting = parseWeighting(tmp[-1])
        if weighting:
            consequent = formatExpressionTokens(tmp[0:-1])
            rule.addConsequent(consequent[0], consequent[1], logic_const.NO_OP, weighting)
        else:
            consequent = formatExpressionTokens(tmp)
            rule.addConsequent(consequent[0], consequent[1], logic_const.NO_OP, 1.0)
    else:
        weighting = 1.0
        consequent = tmp[0]
        rule.addConsequent(consequent, "", logic_const.NO_OP, weighting)
        
    return rule
    
    
def parseWeighting(wStr):
    """
    Chaecks if string has correct format for
    weighting then returns number as float.  If
    not properly formatted, or unable to convert to
    number, returns C{None}.
    """
    # validate
    if len(wStr) < 3:
        return None
    if wStr[0] != logic_const.WEIGHT_OPEN:
        return None
    if wStr[-1] != logic_const.WEIGHT_CLOSE:
        return None
    
    wNum = None
    try:
        wNum = float(wStr[1:-1])
    except:
        pass
    return wNum
    
def formatExpressionTokens(tokens):
    """
    Checks tokens and separates into property
    and value.  Returns as tuple.
    """
    import string
    
    totalTokens = len(tokens)
    if totalTokens < 3:
        raise LogicRuleParsingException("Insufficient tokens for expression.")
    
    property = []
    value = []
    formatState = 1 # flag for property
    i = 0
    
    while(i < totalTokens):
        tkn = tokens[i]
        i += 1
        if string.lower(tkn) == "is":
            formatState = 2
        elif formatState == 1:
            property.append(tkn)
        elif formatState == 2:
            value.append(tkn)
        else:
            continue
    expr = (string.join(property), string.join(value))
    return expr
