# svs_simulation.ai_lib.steering

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Steering behaviours for simulation entities, such as agents.

This module is based on the C{SteeringBehaviours} class outlined in
Mat Buckland, 2005, Programming Game AI by Example, Wordware:Plano, 
see U{http://www.wordware.com/files/ai}, and U{http://www.ai-junkie.com}.
Mat's examples are based on Craig Reynold's work and the OpenSteer project:
U{http://www.red3d.com/cwr/steer/}, U{http://opensteer.sourceforge.net/}.

This version does not support dithered summing on steering force, space 
partitioning, wall avoidance or path following.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
import random

# internal imports
from svs_simulation import settings
from svs_core.utilities.lib import Constants, Enumeration
from svs_simulation.numdata.vectors import Vector2D, vec2DNormalize
from svs_simulation.numdata.geomlib import geom_const
from svs_simulation.numdata.transformations import vec2DRotateAroundOrigin


#############################
# CONSTANTS
#############################
steering_const = Constants()
steering_const.WANDER_RADIUS = 1.2
steering_const.WANDER_DISTANCE = 2.0
steering_const.WANDER_JITTER_PER_SECOND = 80.0
steering_const.WAYPOINT_SEEK_DISTANCE = 20

behavior_type = Constants()
behavior_type.none               = 0x00000
behavior_type.seek               = 0x00002
behavior_type.flee               = 0x00004
behavior_type.arrive             = 0x00008
behavior_type.wander             = 0x00010
behavior_type.cohesion           = 0x00020
behavior_type.separation         = 0x00040
behavior_type.allignment         = 0x00080
behavior_type.obstacle_avoidance = 0x00100
behavior_type.wall_avoidance     = 0x00200
behavior_type.follow_path        = 0x00400
behavior_type.pursuit            = 0x00800
behavior_type.evade              = 0x01000
behavior_type.interpose          = 0x02000
behavior_type.hide               = 0x04000
behavior_type.flock              = 0x08000
behavior_type.offset_pursuit     = 0x10000

summing_method = Enumeration('summing_method',['weighted_average', 'prioritized','dithered'])

deceleration_rate = Constants()
deceleration_rate.slow = 3
deceleration_rate.normal = 2
deceleration_rate.fast = 1


class SimpleSteeringBehaviour:
	"""
	Encapsulates steering behaviours for moving entities.

	@ivar self.entity:the owner of this instance
	@type self.entity: L{svs_demogame.gamebase.MovingEntity}
	@ivar self.steeringForce:the steering force created by the combined effect of all the selected behaviors
	@type self.steeringForce: L{svs_simulation.numdata.vectors.Vector2D}
	@ivar self.target:the current target, location in space
	@type self.target: L{svs_simulation.numdata.vectors.Vector2D}
	@ivar self.targetAgent_1:agent that may affect behaviour, such as friends, pursuers, or prey
	@type self.targetAgent_1: L{svs_simulation.agents.basic_agents.SimAgent}
	@ivar self.targetAgent_2:agent that may affect behaviour, such as friends, pursuers, or prey
	@type self.targetAgent_2: L{svs_simulation.agents.basic_agents.SimAgent}
	@ivar self.detectionBoxLength:length of the 'detection box' utilized in obstacle avoidance
	@type self.detectionBoxLength: float
	@ivar self.feelers:list of feelers required for wall avoidance
	@type self.feelers: Array
	@ivar self.wallDetectionFeelerLength:the length of the 'feeler/s' used in wall detection
	@type self.wallDetectionFeelerLength: float

	@ivar self.wanderTarget:the current position on the wander circle the agent is attempting to steer towards
	@type self.wanderTarget: L{svs_simulation.numdata.vectors.Vector2D}
	@ivar self.wanderJitter:the maximum amount of displacement along the circle each frame
	@type self.wanderJitter: float
	@ivar self.wanderRadius:the radius of the constraining circle for the wander behavior
	@type self.wanderRadius: float
	@ivar self.wanderDistance:distance the wander circle is projected in front of the agent
	@type self.wanderDistance: float

	@ivar self.weightSeparation:multiplier for adjusting behaviour
	@type self.weightSeparation: float
	@ivar self.weightCohesion:multiplier for adjusting behaviour
	@type self.weightCohesion: float
	@ivar self.weightAlignment:multiplier for adjusting behaviour
	@type self.weightAlignment: float
	@ivar self.weightWander:multiplier for adjusting behaviour
	@type self.weightWander: float
	@ivar self.weightObstacleAvoidance:multiplier for adjusting behaviour
	@type self.weightObstacleAvoidance: float
	@ivar self.weightWallAvoidance:multiplier for adjusting behaviour
	@type self.weightWallAvoidance: float
	@ivar self.weightSeek:multiplier for adjusting behaviour
	@type self.weightSeek: float
	@ivar self.weightFlee:multiplier for adjusting behaviour
	@type self.weightFlee: float
	@ivar self.weightArrive:multiplier for adjusting behaviour
	@type self.weightArrive: float
	@ivar self.weightPursuit:multiplier for adjusting behaviour
	@type self.weightPursuit: float
	@ivar self.weightOffsetPursuit:multiplier for adjusting behaviour
	@type self.weightOffsetPursuit: float
	@ivar self.weightInterpose:multiplier for adjusting behaviour
	@type self.weightInterpose: float
	@ivar self.weightHide:multiplier for adjusting behaviour
	@type self.weightHide: float
	@ivar self.weightEvade:multiplier for adjusting behaviour
	@type self.weightEvade: float
	@ivar self.weightFollowPath:multiplier for adjusting behaviour
	@type self.weightFollowPath: float

	@ivar self.viewDistance:how far the agent can 'see'
	@type self.viewDistance: float

	@ivar self.path:pointer to any current path
	@type self.path: Path !tmp

	@ivar self.waypointSeekDistSq:the distance (squared) a entity has to be from a path waypoint before it starts seeking to the next waypoint
	@type self.waypointSeekDistSq: float

	@ivar self.offset:any offset used for formations or offset pursuit
	@type self.offset: L{svs_simulation.numdata.vectors.Vector2D}

	@ivar self.flags:binary flags to indicate whether or not a behavior should be active
	@type self.flags: integer

	@ivar self.deceleration:C{Arrive} behaviour makes use of these to determine how quickly a entity should decelerate to its target
	@type self.deceleration: L{deceleration_rate} value

	@ivar self.summingMethod:what type of method is used to sum any active behavior
	@type self.summingMethod: L{summing_method} value
	"""
	def __init__(self, entity):
		self.entity = entity
		self.steeringForce = Vector2D()
		self.target = Vector2D()
		self.targetAgent_1 = None
		self.targetAgent_2 = None
		# arrive
		self.arriveProximity = 0.2
		# detection
		self.detectionBoxLength = 0.0
		self.feelers = []
		self.wallDetectionFeelerLength = 0.0
		# wandering
		self.wanderTarget = Vector2D()
		self.wanderJitter = 0.0
		self.wanderRadius = 0.0
		self.wanderDistance = 0.0
		# multipliers
		self.weightSeparation = 0.0
		self.weightCohesion = 0.0
		self.weightAlignment = 0.0
		self.weightWander = 0.0
		self.weightObstacleAvoidance = 0.0
		self.weightWallAvoidance = 0.0
		self.weightSeek = 0.0
		self.weightFlee = 0.0
		self.weightArrive = 0.0
		self.weightPursuit = 0.0
		self.weightOffsetPursuit = 0.0
		self.weightInterpose = 0.0
		self.weightHide = 0.0
		self.weightEvade = 0.0
		self.weightFollowPath = 0.0
		# vision
		self.viewDistance = 0.0
		# paths
		self.path = None
		# util
		self.waypointSeekDistSq = 0.0
		self.offset = Vector2D()
		self.flags = 0
		self.deceleration = deceleration_rate.normal
		self.summingMethod = summing_method.weighted_average

	def _behaviourOn(self, behaviorType):
		"""
		Tests if a specific bit of self.flags is set.
		"""
		return (self.flags & behaviorType) == behaviorType

	def accumulateForce(self, runningTotal, forceToAdd):
		"""
		This function calculates how much of its max steering force the 
		entity has left to apply and then applies that amount of the
		force to add.

		@type runningTotal:L{svs_simulation.numdata.vectors.Vector2D}
		@type forceToAdd:L{svs_simulation.numdata.vectors.Vector2D}
		@rtype: boolean
		"""
		# calculate how much steering force the entity has used so far
		magnitudeSoFar = runningTotal.length()
		# calculate how much steering force remains to be used by this entity
		magnitudeRemaining = self.entity.maxForce - magnitudeSoFar
		# return false if there is no more force left to use
		if magnitudeRemaining <= 0.0:return False
		# calculate the magnitude of the force we want to add
		magnitudeToAdd = forceToAdd.length()
		# if the magnitude of the sum of ForceToAdd and the running total
		# does not exceed the maximum force available to this entity, just
		# add together. Otherwise add as much of the ForceToAdd vector is
		# possible without going over the max.
		if magnitudeToAdd < magnitudeRemaining:runningTotal += forceToAdd
		else:
			# add it to the steering force
			runningTotal += vec2DNormalize(forceToAdd) * magnitudeRemaining 
		return True


	def createFeelers(self):
		"""
		Creates the antenna utilized by the wall avoidance behavior.
		"""
		self.feelers = []
		self.feelers.append(self.entity.location + self.wallDetectionFeelerLength * self.entity.heading)
		# feeler to left
		temp = self.entity.heading
		vec2DRotateAroundOrigin(temp, geom_const.half_pi * 3.5)
		self.feelers.append(self.entity.location + self.wallDetectionFeelerLength/2.0 * temp)
		# feeler to right
		temp = self.entity.heading
		vec2DRotateAroundOrigin(temp, geom_const.half_pi * 0.5)
		self.feelers.append(self.entity.location + self.wallDetectionFeelerLength/2.0 * temp)


	###############################
	# BEHAVIOR METHODS
	###############################
	def seek(self, targetPos):
		"""
		Given a target, this behavior returns a steering force which will
		direct the agent towards the target.

		@type targetPos: L{svs_simulation.numdata.vectors.Vector2D}
		@rtpe: L{svs_simulation.numdata.vectors.Vector2D}
		"""
		desiredVelocity = vec2DNormalize(targetPos - self.entity.location) * self.entity.maxSpeed
		return desiredVelocity - self.entity.velocity

	def arrive(self, targetPos, deceleration):
		"""
		Similar to seek but it attempts to arrive at the target position with a zero velocity.

		@type targetPos: L{svs_core.geometry.vectors.Vector2D}
		@type deceleration: integer
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		toTarget = targetPos - self.entity.location
		# calculate the distance to the target
		dist = toTarget.length()
		#print "arrive, dist:", dist
		if dist > self.arriveProximity:
			print "moving"
			# because Deceleration is enumerated as an int, this value is required
			# to provide fine tweaking of the deceleration..
			decelerationTweaker = 0.3
			# calculate the speed required to reach the target given the desired
			# deceleration
			speed =  dist / (deceleration * decelerationTweaker)    
			# make sure the velocity does not exceed the max
			speed = min(speed, self.entity.maxSpeed)
			# from here proceed just like Seek except we don't need to normalize 
			# the ToTarget vector because we have already gone to the trouble
			# of calculating its length: dist. 
			desiredVelocity =  toTarget * speed / dist
			return desiredVelocity - self.entity.velocity
		else:
			print "__________________there_______________"
			self.arriveOff()
			self.entity.halt()
			return Vector2D(0.0,0.0)



	def followPath(self):
		"""
		Given a series of L{svs_simulation.numdata.vectors.Vector2D}s, this method produces a force that will move the agent along the waypoints in order.

		NOTE: not currently implemented.
		
		@rtpe: L{svs_simulation.numdata.vectors.Vector2D}
		"""
		raise Exception("followPath method not implemented.")



	###############################
	# STEERING FORCE METHODS
	###############################
	def calculateWeightedSum(self):
		"""
		Sums up all the active behaviors X their weights and 
		truncates the result to the max available steering force before 
		returning.
		
		@rtpe: L{svs_simulation.numdata.vectors.Vector2D}
		"""
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.wall_avoidance):
			#self.steeringForce += self.wallAvoidance(self.entity.terrain.getWalls()) * self.weightWallAvoidance
		## END NOT IMPLEMENTED ##
		if self._behaviourOn(behavior_type.obstacle_avoidance):
			self.steeringForce += self.obstacleAvoidance(self.entity.terrain.getObstacles()) * self.weightObstacleAvoidance
		if self._behaviourOn(behavior_type.evade):
			assert(self.targetAgent_1)
			self.steeringForce += self.evade(self.targetAgent_1) * self.weightEvade

		# these next three can be combined for flocking behavior (wander is
		# also a good behavior to add into this mix)	
		# NOTE: update for space partitioning
		if self._behaviourOn(behavior_type.separation):
			self.steeringForce += self.separation(self.entity.terrain.Agents()) * self.weightSeparation
		if self._behaviourOn(behavior_type.allignment):
			self.steeringForce += self.alignment(self.entity.terrain.Agents()) * self.weightAlignment
		if self._behaviourOn(behavior_type.cohesion):
			self.steeringForce += self.cohesion(self.entity.terrain.Agents()) * self.weightCohesion
		
		if self._behaviourOn(behavior_type.wander):
			self.steeringForce += self.wander() * self.weightWander
		if self._behaviourOn(behavior_type.seek):
			#print "calc seek:", self.weightSeek
			self.steeringForce += self.seek(self.entity.targetPoint) * self.weightSeek
		if self._behaviourOn(behavior_type.flee):
			self.steeringForce += self.flee(self.entity.targetPoint) * self.weightFlee
		if self._behaviourOn(behavior_type.arrive):
			self.steeringForce += self.arrive(self.entity.targetPoint, self.deceleration) * self.weightArrive
		if self._behaviourOn(behavior_type.pursuit):
			try:
				assert(self.targetAgent_1)
				self.steeringForce += self.pursuit(self.targetAgent_1) * self.weightPursuit
			except:pass #ignore
		if self._behaviourOn(behavior_type.offset_pursuit):
			try:
				assert (self.targetAgent_1)
				assert (not self.offset.isZero())
				self.steeringForce += self.offsetPursuit(self.targetAgent_1, self.offset) * self.weightOffsetPursuit
			except:pass #ignore
		if self._behaviourOn(behavior_type.interpose):
			try:
				assert (self.targetAgent_1 and self.targetAgent_2)
				self.steeringForce += self.interpose(self.targetAgent_1, self.targetAgent_2) * self.weightInterpose
			except:pass #ignore
		if self._behaviourOn(behavior_type.hide):
			try:
				assert(self.targetAgent_1)
				self.steeringForce += self.hide(self.targetAgent_1, self.entity.terrain.getObstacles()) * self.weightHide
			except:pass #ignore
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.follow_path):
			#self.steeringForce += self.followPath() * self.weightFollowPath
		## END NOT IMPLEMENTED ##
		# truncate
		self.steeringForce.truncate(self.entity.maxForce)
		return self.steeringForce



	def calculatePrioritized(self):
		"""
		Calls each active steering behavior in order of priority
		and acumulates their forces until the max steering force magnitude
		is reached, at which time the function returns the steering force 
		accumulated to that  point.
		
		@rtpe: L{svs_simulation.numdata.vectors.Vector2D}
		"""
		force = Vector2D()
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.wall_avoidance):
			#force = self.wallAvoidance(self.entity.terrain.Walls()) * self.weightWallAvoidance
			#if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		## END NOT IMPLEMENTED ##
		if self._behaviourOn(behavior_type.obstacle_avoidance):
			force = self.obstacleAvoidance(self.entity.terrain.Obstacles()) * self.weightObstacleAvoidance
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.evade):
			try:
				assert(self.targetAgent_1 and "Evade target not assigned")
				force = self.evade(self.targetAgent_1) * self.weightEvade
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.flee):
			force = self.flee(self.entity.targetPoint) * self.weightFlee
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		# these next three can be combined for flocking behavior (wander is
		# also a good behavior to add into this mix)
		if self._behaviourOn(behavior_type.separation):
			force = self.separation(self.entity.terrain.Agents()) * self.weightSeparation
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.allignment):
			force = self.alignment(self.entity.terrain.Agents()) * self.weightAlignment
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.cohesion):
			force = self.cohesion(self.entity.terrain.Agents()) * self.weightCohesion
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.seek):
			force = self.seek(self.entity.targetPoint) * self.weightSeek
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.arrive):
			force = self.arrive(self.entity.targetPoint, self.deceleration) * self.weightArrive
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.wander):
			force = self.wander() * self.weightWander
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.pursuit):
			try:
				assert(self.targetAgent_1)
				force = self.pursuit(self.targetAgent_1) * self.weightPursuit
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.offset_pursuit):
			try:
				assert (self.targetAgent_1)
				assert (not self.offset.isZero())
				force = self.offsetPursuit(self.targetAgent_1, self.offset)
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.interpose):
			try:
				assert (self.targetAgent_1 and self.targetAgent_2)
				force = self.interpose(self.targetAgent_1, self.targetAgent_2) * self.weightInterpose
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.hide):
			try:
				assert(self.targetAgent_1)
				force = self.hide(self.targetAgent_1, self.entity.terrain.Obstacles()) * self.weightHide
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.follow_path):
			#force = self.followPath() * self.weightFollowPath
			#if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		## END NOT IMPLEMENTED ##
		return self.steeringForce



	def calculateDithered(self):
		"""
		Sums up the active behaviors by assigning a probabilty
		of being calculated to each behavior. It then tests the first priority
		to see if it should be calcukated this simulation-step. If so, it
		calculates the steering force resulting from this behavior. If it is
		more than zero it returns the force. If zero, or if the behavior is
		skipped it continues onto the next priority, and so on.

		NOTE: not implemented
		
		@rtpe: L{svs_simulation.numdata.vectors.Vector2D}
		"""
		raise Exception("calculateDithered not implemented")


	###############################
	# HIDE METHODS
	###############################
	def getHidingPosition(self, posOb, radiusOb, posHunter):
		"""
		Given the position of a hunter, and the position and radius of
		an obstacle, this method calculates a position DistanceFromBoundary 
		away from its bounding radius and directly opposite the hunter.
		
		@type posOb: L{svs_simulation.numdata.vectors.Vector2D}
		@type radiusOb: float
		@type posHunter: L{svs_simulation.numdata.vectors.Vector2D}
		@rtpe: L{svs_simulation.numdata.vectors.Vector2D}
		"""
		# calculate how far away the agent is to be from the chosen obstacle's bounding radius
		distanceFromBoundary = 30.0
		distAway = radiusOb + distanceFromBoundary
		# calculate the heading toward the object from the hunter
		toOb = vec2DNormalize(posOb - posHunter)
		# scale it to size and add to the obstacles position to get the hiding spot.
		return (toOb * distAway) + posOb


	###############################
	# PUBLIC METHODS
	###############################
	def calculate(self):
		"""
		Calculates and sums the steering forces from any active behaviors,
		according to the method set in L{SteeringBehaviours.summingMethod}.
		
		@todo: add in space-partitioning.
		@rtpe: L{svs_simulation.numdata.vectors.Vector2D}
		"""
		# reset the steering force
		self.steeringForce.zero()
		# NOTE: add in space-aprtitioning
		# tag neighbors if any of the following 3 group behaviors are switched on
		if self._behaviourOn(behavior_type.separation) or self._behaviourOn(behavior_type.allignment) or self._behaviourOn(behavior_type.cohesion):
			self.entity.terrain.tagVehiclesWithinViewRange(self.entity.location, self.viewDistance)

		if self.summingMethod == summing_method.weighted_average:
			self.steeringForce = self.calculateWeightedSum()
		elif self.summingMethod == summing_method.prioritized:
			self.steeringForce = self.calculatePrioritized()
		elif self.summingMethod == summing_method.dithered:
			self.steeringForce = self.calculateDithered()
		else:self.steeringForce = Vector2D(0,0)

		return self.steeringForce


	def forwardComponent(self):
		"""
		Calculates the component of the steering force that is parallel with the entity heading.
		
		@rtpe: float
		"""
		return self.entity.heading.dot(self.steeringForce)

	def sideComponent(self):
		"""
		Calculates the component of the steering force that is perpendicuar with the entity heading.
		
		@rtpe: float
		"""
		return self.entity.side.dot(self.steeringForce)

	def setTarget(self, target):
		"""
		Sets new target for steering behaviour.
		"""
		self.target = target

	def addTargetAgent(self, targetAgent):
		"""
		Adds new agent to list of target agents.
		"""
		if targetAgent not in self.targetAgents: self.targetAgents.append(targetAgent)

	def setOffset(self, offset):
		"""
		Sets offset to steering behaviour.
	
		@type offset:L{svs_simulation.numdata.vectors.Vector2D}
		"""
		self.offset = offset


	def setpath(self, path):
		"""
		Sets path for entity to follow. The path is defined as an array of L{svs_simulation.numdata.vectors.Vector2D}s
		"""
		self.path = path



	def setSummingMethod(self, summingMethod):
		"""
		Sets summing method, this should be a value defined in L{summing_method}.
		"""
		self.summingMethod = summingMethod



	###############################
	# BEHAVIOUR FLAGS
	###############################
	def fleeOn(self):
		"""
		Turns flee behaviour on.
		"""
		self.flags |= behavior_type.flee

	def seekOn(self):
		"""
		Turns seek behaviour on.
		"""
		self.flags |= behavior_type.seek

	def arriveOn(self):
		"""
		Turns arrive behaviour on.
		"""
		self.flags |= behavior_type.arrive

	def wanderOn(self):
		"""
		Turns wander behaviour on.
		"""
		self.flags |= behavior_type.wander

	def pursuitOn(self, targetAgent):
		"""
		Turns pursuit behaviour on and defines target of pursuit.
		"""
		self.flags |= behavior_type.pursuit
		self.targetAgent_1 = targetAgent

	def evadeOn(self, targetAgent):
		"""
		Turns evade behaviour on and defines agent to evade.
		"""
		self.flags |= behavior_type.evade
		self.targetAgent_1 = targetAgent

	def cohesionOn(self):
		"""
		Turns cohesion behaviour on.
		"""
		self.flags |= behavior_type.cohesion

	def separationOn(self):
		"""
		Turns separation behaviour on.
		"""
		self.flags |= behavior_type.separation

	def alignmentOn(self):
		"""
		Turns alignment behaviour on.
		"""
		self.flags |= behavior_type.allignment

	def obstacleAvoidanceOn(self):
		"""
		Turns obstacle avoidance behaviour on.
		"""
		self.flags |= behavior_type.obstacle_avoidance

	def wallAvoidanceOn(self):
		"""
		Turns wall avoidance behaviour on.
		"""
		self.flags |= behavior_type.wall_avoidance

	def followPathOn(self):
		"""
		Turns path following behaviour on.
		"""
		self.flags |= behavior_type.follow_path

	def interposeOn(self, targetAgent_1, targetAgent_2):
		"""
		Turns interpose behaviour on and defines two agents to interpose between.
		"""
		self.flags |= behavior_type.interpose
		self.targetAgent_1 = targetAgent_1
		self.targetAgent_2 = targetAgent_2

	def hideOn(self, targetAgent):
		"""
		Turns hide behaviour on and define sagent to hide from.
		"""
		self.flags |= behavior_type.hide
		self.targetAgent_1 = targetAgent

	def offsetPursuitOn(self, targetAgent, offset):
		"""
		Turns pursuit behaviour on, defines target of pursuit and set offset.
		"""
		self.flags |= behavior_type.offset_pursuit
		self.offset = offset
		self.targetAgent_1 = targetAgent  

	def flockingOn(self):
		"""
		Turns flocking behaviour on.
		"""
		self.cohesionOn()
		self.alignmentOn()
		self.separationOn()
		self.wanderOn()

	def fleeOff(self):
		if self._behaviourOn(behavior_type.flee):self.flags ^= behavior_type.flee

	def seekOff(self):
		if self._behaviourOn(behavior_type.seek):self.flags ^= behavior_type.seek

	def arriveOff(self):
		if self._behaviourOn(behavior_type.arrive):self.flags ^= behavior_type.arrive

	def wanderOff(self):
		if self._behaviourOn(behavior_type.wander):self.flags ^= behavior_type.wander

	def pursuitOff(self):
		if self._behaviourOn(behavior_type.pursuit):self.flags ^= behavior_type.pursuit

	def evadeOff(self):
		if self._behaviourOn(behavior_type.evade):self.flags ^= behavior_type.evade

	def cohesionOff(self):
		if self._behaviourOn(behavior_type.cohesion):self.flags ^= behavior_type.cohesion

	def separationOff(self):
		if self._behaviourOn(behavior_type.separation):self.flags ^= behavior_type.separation

	def alignmentOff(self):
		if self._behaviourOn(behavior_type.allignment):self.flags ^= behavior_type.allignment

	def obstacleAvoidanceOff(self):
		if self._behaviourOn(behavior_type.obstacle_avoidance):self.flags ^= behavior_type.obstacle_avoidance

	def wallAvoidanceOff(self):
		if self._behaviourOn(behavior_type.wall_avoidance):self.flags ^= behavior_type.wall_avoidance

	def followPathOff(self):
		if self._behaviourOn(behavior_type.follow_path):self.flags ^= behavior_type.follow_path

	def interposeOff(self):
		if self._behaviourOn(behavior_type.interpose):self.flags ^= behavior_type.interpose

	def hideOff(self):
		if self._behaviourOn(behavior_type.hide):self.flags ^= behavior_type.hide

	def offsetPursuitOff(self):
		if self._behaviourOn(behavior_type.offset_pursuit):self.flags ^= behavior_type.offset_pursuit

	def flockingOff(self):
		self.cohesionOff()
		self.alignmentOff()
		self.separationOff()
		self.wanderOff()

	def isFleeOn(self):
		return self._behaviourOn(behavior_type.flee)

	def isSeekOn(self):
		return self._behaviourOn(behavior_type.seek)

	def isArriveOn(self):
		return self._behaviourOn(behavior_type.arrive)

	def isWanderOn(self):
		return self._behaviourOn(behavior_type.wander)

	def isPursuitOn(self):
		return self._behaviourOn(behavior_type.pursuit)

	def isEvadeOn(self):
		return self._behaviourOn(behavior_type.evade)

	def isCohesionOn(self):
		return self._behaviourOn(behavior_type.cohesion)

	def isSeparationOn(self):
		return self._behaviourOn(behavior_type.separation)

	def isAlignmentOn(self):
		return self._behaviourOn(behavior_type.allignment)

	def isObstacleAvoidanceOn(self):
		return self._behaviourOn(behavior_type.obstacle_avoidance)

	def isWallAvoidanceOn(self):
		return self._behaviourOn(behavior_type.wall_avoidance)

	def isFollowPathOn(self):
		return self._behaviourOn(behavior_type.follow_path)

	def isInterposeOn(self):
		return self._behaviourOn(behavior_type.interpose)

	def isHideOn(self):
		return self._behaviourOn(behavior_type.hide)

	def isOffsetPursuitOn(self):
		return self._behaviourOn(behavior_type.offset_pursuit)


	###############################
	# ENTITY METHODS
	###############################
	def seekToPoint(self, x, y):
		"""
		Sets a target point for entity to move towards and start seek behaviour.
		"""
		#print "seekToPoint: %f, %f" % (x, y)
		self.entity.setTargetPoint(x, y)
		self.weightSeek = 1.0
		self.seekOn()

