# svs_simulation.steering

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Steering behaviours for simulation entities, such as agents.

This module is based on the C{SteeringBehaviours} class outlined in
Mat Buckland, 2005, Programming Game AI by Example, Wordware:Plano, 
see U{http://www.wordware.com/files/ai}, and U{http://www.ai-junkie.com}.
Mat's examples are based on Craig Reynold's work and the OpenSteer project:
U{http://www.red3d.com/cwr/steer/}, U{http://opensteer.sourceforge.net/}.

This version does not support dithered summing on steering force, space 
partitioning, wall avoidabce or path following.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
import random

# internal imports
from svs_simulation import settings
from svs_core.utilities.lib import Constants, Enumeration
from svs_core.geometry.vectors import Vector2D, vec2DNormalize
from svs_core.geometry.geomlib import geom_const
from svs_core.geometry.transformations import vec2DRotateAroundOrigin


#############################
# CONSTANTS
#############################
steering_const = Constants()
steering_const.WANDER_RADIUS = 1.2
steering_const.WANDER_DISTANCE = 2.0
steering_const.WANDER_JITTER_PER_SECOND = 80.0
steering_const.WAYPOINT_SEEK_DISTANCE = 20

behavior_type = Constants()
behavior_type.none               = 0x00000
behavior_type.seek               = 0x00002
behavior_type.flee               = 0x00004
behavior_type.arrive             = 0x00008
behavior_type.wander             = 0x00010
behavior_type.cohesion           = 0x00020
behavior_type.separation         = 0x00040
behavior_type.allignment         = 0x00080
behavior_type.obstacle_avoidance = 0x00100
behavior_type.wall_avoidance     = 0x00200
behavior_type.follow_path        = 0x00400
behavior_type.pursuit            = 0x00800
behavior_type.evade              = 0x01000
behavior_type.interpose          = 0x02000
behavior_type.hide               = 0x04000
behavior_type.flock              = 0x08000
behavior_type.offset_pursuit     = 0x10000

summing_method = Enumeration('summing_method',['weighted_average', 'prioritized','dithered'])

deceleration_rate = Constants()
deceleration_rate.slow = 3
deceleration_rate.normal = 2
deceleration_rate.fast = 1


class SteeringBehaviours:
	"""
	Encapsulates steering behaviours for moving entities.

	@ivar self.vehicle:the owner of this instance
	@type self.vehicle: L{svs_demogame.gamebase.MovingEntity}
	@ivar self.steeringForce:the steering force created by the combined effect of all the selected behaviors
	@type self.steeringForce: L{svs_core.geometry.vectors.Vector2D}
	@ivar self.target:the current target, location in space
	@type self.target: L{svs_core.geometry.vectors.Vector2D}
	@ivar self.targetAgent_1:agent that may affect behaviour, such as friends, pursuers, or prey
	@type self.targetAgent_1: L{svs_demogame.Agent}
	@ivar self.targetAgent_2:agent that may affect behaviour, such as friends, pursuers, or prey
	@type self.targetAgent_2: L{svs_demogame.Agent}
	@ivar self.detectionBoxLength:length of the 'detection box' utilized in obstacle avoidance
	@type self.detectionBoxLength: float
	@ivar self.feelers:list of feelers required for wall avoidance
	@type self.feelers: Array
	@ivar self.wallDetectionFeelerLength:the length of the 'feeler/s' used in wall detection
	@type self.wallDetectionFeelerLength: float

	@ivar self.wanderTarget:the current position on the wander circle the agent is attempting to steer towards
	@type self.wanderTarget: L{svs_core.geometry.vectors.Vector2D}
	@ivar self.wanderJitter:the maximum amount of displacement along the circle each frame
	@type self.wanderJitter: float
	@ivar self.wanderRadius:the radius of the constraining circle for the wander behavior
	@type self.wanderRadius: float
	@ivar self.wanderDistance:distance the wander circle is projected in front of the agent
	@type self.wanderDistance: float

	@ivar self.weightSeparation:multiplier for adjusting behaviour
	@type self.weightSeparation: float
	@ivar self.weightCohesion:multiplier for adjusting behaviour
	@type self.weightCohesion: float
	@ivar self.weightAlignment:multiplier for adjusting behaviour
	@type self.weightAlignment: float
	@ivar self.weightWander:multiplier for adjusting behaviour
	@type self.weightWander: float
	@ivar self.weightObstacleAvoidance:multiplier for adjusting behaviour
	@type self.weightObstacleAvoidance: float
	@ivar self.weightWallAvoidance:multiplier for adjusting behaviour
	@type self.weightWallAvoidance: float
	@ivar self.weightSeek:multiplier for adjusting behaviour
	@type self.weightSeek: float
	@ivar self.weightFlee:multiplier for adjusting behaviour
	@type self.weightFlee: float
	@ivar self.weightArrive:multiplier for adjusting behaviour
	@type self.weightArrive: float
	@ivar self.weightPursuit:multiplier for adjusting behaviour
	@type self.weightPursuit: float
	@ivar self.weightOffsetPursuit:multiplier for adjusting behaviour
	@type self.weightOffsetPursuit: float
	@ivar self.weightInterpose:multiplier for adjusting behaviour
	@type self.weightInterpose: float
	@ivar self.weightHide:multiplier for adjusting behaviour
	@type self.weightHide: float
	@ivar self.weightEvade:multiplier for adjusting behaviour
	@type self.weightEvade: float
	@ivar self.weightFollowPath:multiplier for adjusting behaviour
	@type self.weightFollowPath: float

	@ivar self.viewDistance:how far the agent can 'see'
	@type self.viewDistance: float

	@ivar self.path:pointer to any current path
	@type self.path: Path !tmp

	@ivar self.waypointSeekDistSq:the distance (squared) a vehicle has to be from a path waypoint before it starts seeking to the next waypoint
	@type self.waypointSeekDistSq: float

	@ivar self.offset:any offset used for formations or offset pursuit
	@type self.offset: L{svs_core.geometry.vectors.Vector2D}

	@ivar self.flags:binary flags to indicate whether or not a behavior should be active
	@type self.flags: integer

	@ivar self.deceleration:C{Arrive} behaviour makes use of these to determine how quickly a vehicle should decelerate to its target
	@type self.deceleration: L{deceleration_rate} value

	@ivar self.summingMethod:what type of method is used to sum any active behavior
	@type self.summingMethod: L{summing_method} value
	"""
	def __init__(self, vehicle):
		self.vehicle = vehicle
		self.steeringForce = Vector2D()
		self.target = Vector2D()
		self.targetAgent_1 = None
		self.targetAgent_2 = None
		# detection
		self.detectionBoxLength = 0.0
		self.feelers = []
		self.wallDetectionFeelerLength = 0.0
		# wandering
		self.wanderTarget = Vector2D()
		self.wanderJitter = 0.0
		self.wanderRadius = 0.0
		self.wanderDistance = 0.0
		# multipliers
		self.weightSeparation = 0.0
		self.weightCohesion = 0.0
		self.weightAlignment = 0.0
		self.weightWander = 0.0
		self.weightObstacleAvoidance = 0.0
		self.weightWallAvoidance = 0.0
		self.weightSeek = 0.0
		self.weightFlee = 0.0
		self.weightArrive = 0.0
		self.weightPursuit = 0.0
		self.weightOffsetPursuit = 0.0
		self.weightInterpose = 0.0
		self.weightHide = 0.0
		self.weightEvade = 0.0
		self.weightFollowPath = 0.0
		# vision
		self.viewDistance = 0.0
		# paths
		self.path = None
		# util
		self.waypointSeekDistSq = 0.0
		self.offset = Vector2D()
		self.flags = 0
		self.deceleration = deceleration_rate.normal
		self.summingMethod = summing_method.weighted_average

	def _behaviourOn(self, behaviorType):
		"""
		Tests if a specific bit of self.flags is set.
		"""
		return (self.flags & behaviorType) == behaviorType

	def accumulateForce(self, runningTotal, forceToAdd):
		"""
		This function calculates how much of its max steering force the 
		vehicle has left to apply and then applies that amount of the
		force to add.

		@type runningTotal:L{svs_core.geometry.vectors.Vector2D}
		@type forceToAdd:L{svs_core.geometry.vectors.Vector2D}
		@rtype: boolean
		"""
		# calculate how much steering force the vehicle has used so far
		magnitudeSoFar = runningTotal.length()
		# calculate how much steering force remains to be used by this vehicle
		magnitudeRemaining = self.vehicle.maxForce - magnitudeSoFar
		# return false if there is no more force left to use
		if magnitudeRemaining <= 0.0:return False
		# calculate the magnitude of the force we want to add
		magnitudeToAdd = forceToAdd.length()
		# if the magnitude of the sum of ForceToAdd and the running total
		# does not exceed the maximum force available to this vehicle, just
		# add together. Otherwise add as much of the ForceToAdd vector is
		# possible without going over the max.
		if magnitudeToAdd < magnitudeRemaining:runningTotal += forceToAdd
		else:
			# add it to the steering force
			runningTotal += vec2DNormalize(forceToAdd) * magnitudeRemaining 
		return True


	def createFeelers(self):
		"""
		Creates the antenna utilized by the wall avoidance behavior.
		"""
		self.feelers = []
		self.feelers.append(self.vehicle.location + self.wallDetectionFeelerLength * self.vehicle.heading)
		# feeler to left
		temp = self.vehicle.heading
		vec2DRotateAroundOrigin(temp, geom_const.half_pi * 3.5)
		self.feelers.append(self.vehicle.location + self.wallDetectionFeelerLength/2.0 * temp)
		# feeler to right
		temp = self.vehicle.heading
		vec2DRotateAroundOrigin(temp, geom_const.half_pi * 0.5)
		self.feelers.append(self.vehicle.location + self.wallDetectionFeelerLength/2.0 * temp)


	###############################
	# BEHAVIOR METHODS
	###############################
	def seek(self, targetPos):
		"""
		Given a target, this behavior returns a steering force which will
		direct the agent towards the target.

		@type targetPos: L{svs_core.geometry.vectors.Vector2D}
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		desiredVelocity = vec2DNormalize(targetPos - self.vehicle.location) * self.vehicle.maxSpeed
		return desiredVelocity - self.vehicle.velocity


	def flee(self, targetPos):
		"""
		Returns a vector that moves the agent away from a target position.
		
		@type targetPos: L{svs_core.geometry.vectors.Vector2D}
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# only flee if the target is within 'panic distance'.
		# work in distance squared space.
		panicDistanceSq = 100.0 * 100.0 # !! should be defined by vehicle
		if self.vehicle.location.distanceSq(target) > panicDistanceSq:return Vector2D(0,0)
		desiredVelocity = vec2DNormalize(self.vehicle.location - targetPos) * self.vehicle.maxSpeed
		return desiredVelocity - self.vehicle.velocity


	def arrive(self, targetPos, deceleration):
		"""
		Similar to seek but it attempts to arrive at the target position with a zero velocity.

		@type targetPos: L{svs_core.geometry.vectors.Vector2D}
		@type deceleration: integer
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		toTarget = targetPos - self.vehicle.location
		# calculate the distance to the target
		dist = toTarget.length()
		if dist > 0.0:
			# because Deceleration is enumerated as an int, this value is required
			# to provide fine tweaking of the deceleration..
			decelerationTweaker = 0.3
			# calculate the speed required to reach the target given the desired
			# deceleration
			speed =  dist / (deceleration * decelerationTweaker)    
			# make sure the velocity does not exceed the max
			speed = min(speed, self.vehicle.maxSpeed)
			# from here proceed just like Seek except we don't need to normalize 
			# the ToTarget vector because we have already gone to the trouble
			# of calculating its length: dist. 
			desiredVelocity =  toTarget * speed / dist
			return desiredVelocity - self.vehicle.velocity
		return Vector2D()


	def pursuit(self, evader):
		"""
		Predicts where an agent will be in time T and seeks towards that point to intercept it.
		
		@type targetAgent: L{svs_simulation.base_entities.MovingEntity}
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# if the evader is ahead and facing the agent then we can just seek
		# for the evader's current position.
		toEvader = evader.location - self.vehicle.location
		relativeHeading = self.vehicle.heading.dot(evader.heading)
		if toEvader.dot(self.vehicle.heading) > 0.0 and relativeHeading < -0.95:  return self.seek(evader.location) #acos(0.95)=18 degs
		# Not considered ahead so we predict where the evader will be.
		# the lookahead time is propotional to the distance between the evader
		# and the pursuer and is inversely proportional to the sum of the
		# agent's velocities
		lookAheadTime = evader.length() / (self.vehicle.maxSpeed + evader.speed())
		# now seek to the predicted future position of the evader
		return self.seek(evader.location + evader.velocity * lookAheadTime)

	def offsetPursuit(self, leader, offset):
		"""
		Produces a steering force that keeps a vehicle at a specified offset
		from a leader vehicle.
		
		@type leader: L{svs_simulation.base_entities.MovingEntity}
		@type offset: L{svs_core.geometry.vectors.Vector2D}
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# calculate the offset's position in world space
		worldOffsetPos = pointToWorldSpace(offset, leader.heading,leader.side, leader.location)
		toOffset = worldOffsetPos - self.vehicle.location
		# the lookahead time is propotional to the distance between the leader
		# and the pursuer and is inversely proportional to the sum of both
		# agent's velocities
		lookAheadTime = toOffset.length()/(self.vehicle.maxSpeed + leader.speed())
		# now Arrive at the predicted future position of the offset
		return self.arrive(worldOffsetPos + leader.velocity * lookAheadTime, deceleration_rate.fast)


	def evade(self, pursuer):
		"""
		Attempts to evade a pursuer.

		@type pursuer: L{svs_simulation.base_entities.MovingEntity}
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# Not necessary to include the check for facing direction this time
		toPursuer = pursuer.location - self.vehicle.location
		# Evade only consider pursuers within a 'threat range'
		if toPursuer.lengthSq() > self.vehicle.threatRange * self.vehicle.threatRange: return Vector2D()
		# the lookahead time is propotional to the distance between the pursuer
		# and the pursuer and is inversely proportional to the sum of the
		# agents' velocities
		lookAheadTime = toPursuer.length() / (self.vehicle.maxSpeed + pursuer.speed())
		# now flee away from predicted future position of the pursuer
		return self.flee(pursuer.location + pursuer.velocity * lookAheadTime)


	def wander(self):
		"""
		Makes the agent wander about randomly.

		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# this behavior is dependent on the update rate, so this line must
		# be included when using time independent framerate.
		jitterThisTimeSlice = self.wanderJitter * self.vehicle.timeElapsed
		# first, add a small random vector to the target's position
		self.wanderTarget += Vector2D(random.uniform(-1.0, 1.0) * jitterThisTimeSlice, random.uniform(-1.0, 1.0) * jitterThisTimeSlice)
		# reproject this new vector back on to a unit circle
		self.wanderTarget.normalize()
		# increase the length of the vector to the same as the radius
		# of the wander circle
		self.wanderTarget *= self.wanderRadius
		# move the target into a position WanderDist in front of the agent
		target = self.wanderTarget + Vector2D(self.wanderDistance, 0)
		# project the target into world space
		target = pointToWorldSpace(target, self.vehicle.heading, self.vehicle.side, self.vehicle.location)
		# and steer towards it
		return target - self.vehicle.location 

	def obstacleAvoidance(self, obstacles):
		"""
		Returns a steering force which will attempt to keep the agent away from any obstacles it may encounter.
		
		@type obstacles: Array
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		from svs_simulation.settings import minDetectionBoxLength
		# the detection box length is proportional to the agent's velocity
		self.detectionBoxLength = minDetectionBoxLength + (self.vehicle.speed()/self.vehicle.maxSpeed) * minDetectionBoxLength
		# tag all obstacles within range of the box for processing
		self.vehicle.terrain.tagObstaclesWithinViewRange(self.vehicle.location, self.detectionBoxLength)
		# this will keep track of the closest intersecting obstacle (CIB)
		closestIntersectingObstacle = None
		# this will be used to track the distance to the CIB
		distToClosestIP = math_const.max_float
		# this will record the transformed local coordinates of the CIB
		localPosOfClosestObstacle = Vector2D()
		# iterate through obstacles
		for curOb in obstacles:
			# if the obstacle has been tagged within range proceed
			if curOb.isTagged():
				# calculate this obstacle's position in local space
				localPos = pointToLocalSpace(curOb.location, self.vehicle.heading, self.vehicle.side, self.vehicle.location)
				# if the local position has a negative x value then it must lay
				# behind the agent. (in which case it can be ignored)
				if localPos.x >= 0.0:
					# if the distance from the x axis to the object's position is less
					# than its radius + half the width of the detection box then there
					# is a potential intersection.
					expandedRadius = curOb.boundingRadius + self.vehicle.boundingRadius
					if math.fabs(localPos.y) < expandedRadius:
						# now to do a line/circle intersection test. The center of the 
						# circle is represented by (cX, cY). The intersection points are 
						# given by the formula x = cX +/-sqrt(r^2-cY^2) for y=0. 
						# We only need to look at the smallest positive value of x because
						# that will be the closest point of intersection.
						cX = localPos.x
						cY = localPos.y
          
						# we only need to calculate the sqrt part of the above equation once
						sqrtPart = math.sqrt(expandedRadius*expandedRadius - cY*cY)
						ip = cX - sqrtPart
						if ip <= 0.0: ip = cX + sqrtPart
						# test to see if this is the closest so far. If it is keep a
						# record of the obstacle and its local coordinates
						if ip < distToClosestIP:
							distToClosestIP = ip
							closestIntersectingObstacle = curOb
							localPosOfClosestObstacle = localPos
						## END IF ##
		# if we have found an intersecting obstacle, calculate a steering 
		# force away from it
		steeringForce = Vector2D()
		if closestIntersectingObstacle:
			# the closer the agent is to an object, the stronger the 
			# steering force should be
			multiplier = 1.0 + (self.detectionBoxLength - localPosOfClosestObstacle.x)/self.detectionBoxLength
			# calculate the lateral force
			steeringForce.y = (closestIntersectingObstacle.boundingRadius-localPosOfClosestObstacle.y) * multiplier   
			# apply a braking force proportional to the obstacles distance from
			# the vehicle. 
			brakingWeight = 0.2 # ?? pos make changeable ??
			steeringForce.x = (closestIntersectingObstacle.boundingRadius - localPosOfClosestObstacle.x) * brakingWeight
			## END IF ##
		# finally, convert the steering vector from local to world space
		return vectorToWorldSpace(steeringForce, self.vehicle.heading, self.vehicle.side)


	def wallAvoidance(self, walls):
		"""
		Returns a steering force which will keep the agent away from any walls it may encounter.
		
		NOTE: not currently implemented.

		@type walls: Array
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		raise Exception("wallAvoidance method not implemented.")

	def followPath(self):
		"""
		Given a series of L{svs_core.geometry.vectors.Vector2D}s, this method produces a force that will move the agent along the waypoints in order.

		NOTE: not currently implemented.
		
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		raise Exception("followPath method not implemented.")


	def interpose(self, vehicleA, vehicleB):
		"""
		Given two agents, this method returns a force that attempts to 
		position the vehicle between them.
		
		@type vehicleA: L{svs_simulation.base_entities.MovingEntity}
		@type vehicleB: L{svs_simulation.base_entities.MovingEntity}
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# first we need to figure out where the two agents are going to be at 
		# time T in the future. This is approximated by determining the time
		# taken to reach the mid way point at the current time at at max speed.
		midPoint = (vehicleA.location + vehicleB.location) / 2.0
		timeToReachMidPoint = vec2DDistance(self.vehicle.location, midPoint) / self.vehicle.maxSpeed
		# now we have T, we assume that agent A and agent B will continue on a
		# straight trajectory and extrapolate to get their future positions
		aPos = vehicleA.location + vehicleA.velocity * timeToReachMidPoint # check multiplication on vector!!!!
		bPos = vehicleB.location + vehicleB.velocity * timeToReachMidPoint
		# calculate the mid point of these predicted positions
		midPoint = (aPos + bPos) / 2.0
		# then steer to Arrive at it
		return self.arrive(midPoint, deceleration_rate.fast)



	def hide(self, hunter, obstacles):
		"""
		Returns a steering force that attempts to steer the vehicle to the center of the vector connecting two moving agents.
		
		@type hunter: L{svs_demogame.agents.Agent}
		@type obstacles: Array
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		distToClosest = math_const.max_float
		bestHidingSpot = None
		for curOb in obstacles:
			# calculate the position of the hiding spot for this obstacle
			hidingSpot = self.getHidingPosition(curOb.location, curOb.boundingRadius,hunter.location)
			# work in distance-squared space to find the closest hiding
			# spot to the agent
			dist = vec2DDistanceSq(hidingSpot, self.vehicle.location)
			if dist < distToClosest:
				distToClosest = dist
				bestHidingSpot = hidingSpot
				closest = curOb
				## END IF ##
			## END LOOP ##
		# if no suitable obstacles found then Evade the hunter
		#if distToClosest == math_const.max_float:return self.evade(hunter) - original
		if not bestHidingSpot:return self.evade(hunter)
		# else use Arrive on the hiding spot
		return self.arrive(bestHidingSpot, deceleration_rate.fast)


	###############################
	# GROUP BEHAVIOR METHODS
	###############################
	def cohesion(self, neighbors):
		"""
		Returns a steering force that attempts to move the agent towards the
		center of mass of the agents in its immediate area.
		
		@type neighbors: Array
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# first find the center of mass of all the agents
		centerOfMass = Vector2D()
		steeringForce = Vector2D()
		neighborCount = 0
		# iterate through the neighbors and sum up all the position vectors
		for neighbor in neighbors:
			# make sure *this* agent isn't included in the calculations and that
			# the agent being examined is close enough ***also make sure it doesn't
			# include the evade target ***
			if (neighbors is not self.vehicle) and neighbor.isTagged() and (neighbor is not self.targetAgent_1):
				centerOfMass += neighbor.location
				neighborCount += 1
			## END LOOP ##
		if neighborCount > 0:
			# the center of mass is the average of the sum of positions
			centerOfMass /= neighborCount * 1.0
			# now seek towards that position
			steeringForce = self.seek(centerOfMass)
		# the magnitude of cohesion is usually much larger than separation or
		# allignment so it usually helps to normalize it.
		return vec2DNormalize(steeringForce)


	def separation(self, neighbors):
		"""
		Returns a steering force that maintains distance between agents.
		
		@type neighbors: Array
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		steeringForce = Vector2D()
		for neighbor in neighbors:
			# make sure this agent isn't included in the calculations and that
			# the agent being examined is close enough. ***also make sure it doesn't
			# include the evade target ***
			if (neighbors is not self.vehicle) and neighbor.isTagged() and (neighbor is not self.targetAgent_1):
				toAgent = self.vehicle.location - neighbor.location
				# scale the force inversely proportional to the agents distance  
				# from its neighbor.
				steeringForce += vec2DNormalize(ToAgent)/ToAgent.length()
			## END LOOP ##
		return steeringForce


	def alignment(self, neighbors):
		"""
		Returns a force that attempts to align this agents heading with that
		of its neighbors.
		
		@type neighbors: Array
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# used to record the average heading of the neighbors
		averageHeading = Vector2D()
		# used to count the number of vehicles in the neighborhood
		neighborCount = 0
		# iterate through all the tagged vehicles and sum their heading vectors  
		for neighbor in neighbors:
			# make sure this agent isn't included in the calculations and that
			# the agent being examined is close enough. ***also make sure it doesn't
			# include the evade target ***
			if (neighbors is not self.vehicle) and neighbor.isTagged() and (neighbor is not self.targetAgent_1):
				averageHeading += neighbors[a].heading
				neighborCount += 1
			## END LOOP ##
		# if the neighborhood contained one or more vehicles, average their
		# heading vectors.
		if neighborCount > 0:
			averageHeading /= neighborCount * 1.0
			averageHeading -= self.vehicle.heading
		return averageHeading


	###############################
	# STEERING FORCE METHODS
	###############################
	def calculateWeightedSum(self):
		"""
		Sums up all the active behaviors X their weights and 
		truncates the result to the max available steering force before 
		returning.
		
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.wall_avoidance):
			#self.steeringForce += self.wallAvoidance(self.vehicle.terrain.getWalls()) * self.weightWallAvoidance
		## END NOT IMPLEMENTED ##
		if self._behaviourOn(behavior_type.obstacle_avoidance):
			self.steeringForce += self.obstacleAvoidance(self.vehicle.terrain.getObstacles()) * self.weightObstacleAvoidance
		if self._behaviourOn(behavior_type.evade):
			assert(self.targetAgent_1)
			self.steeringForce += self.evade(self.targetAgent_1) * self.weightEvade

		# these next three can be combined for flocking behavior (wander is
		# also a good behavior to add into this mix)	
		# NOTE: update for space partitioning
		if self._behaviourOn(behavior_type.separation):
			self.steeringForce += self.separation(self.vehicle.terrain.Agents()) * self.weightSeparation
		if self._behaviourOn(behavior_type.allignment):
			self.steeringForce += self.alignment(self.vehicle.terrain.Agents()) * self.weightAlignment
		if self._behaviourOn(behavior_type.cohesion):
			self.steeringForce += self.cohesion(self.vehicle.terrain.Agents()) * self.weightCohesion
		
		if self._behaviourOn(behavior_type.wander):
			self.steeringForce += self.wander() * self.weightWander
		if self._behaviourOn(behavior_type.seek):
			self.steeringForce += self.seek(self.vehicle.targetPoint) * self.weightSeek
		if self._behaviourOn(behavior_type.flee):
			self.steeringForce += self.flee(self.vehicle.targetPoint) * self.weightFlee
		if self._behaviourOn(behavior_type.arrive):
			self.steeringForce += self.arrive(self.vehicle.targetPoint, self.deceleration) * self.weightArrive
		if self._behaviourOn(behavior_type.pursuit):
			try:
				assert(self.targetAgent_1)
				self.steeringForce += self.pursuit(self.targetAgent_1) * self.weightPursuit
			except:pass #ignore
		if self._behaviourOn(behavior_type.offset_pursuit):
			try:
				assert (self.targetAgent_1)
				assert (not self.offset.isZero())
				self.steeringForce += self.offsetPursuit(self.targetAgent_1, self.offset) * self.weightOffsetPursuit
			except:pass #ignore
		if self._behaviourOn(behavior_type.interpose):
			try:
				assert (self.targetAgent_1 and self.targetAgent_2)
				self.steeringForce += self.interpose(self.targetAgent_1, self.targetAgent_2) * self.weightInterpose
			except:pass #ignore
		if self._behaviourOn(behavior_type.hide):
			try:
				assert(self.targetAgent_1)
				self.steeringForce += self.hide(self.targetAgent_1, self.vehicle.terrain.getObstacles()) * self.weightHide
			except:pass #ignore
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.follow_path):
			#self.steeringForce += self.followPath() * self.weightFollowPath
		## END NOT IMPLEMENTED ##
		# truncate
		self.steeringForce.truncate(self.vehicle.maxForce)
		return self.steeringForce



	def calculatePrioritized(self):
		"""
		Calls each active steering behavior in order of priority
		and acumulates their forces until the max steering force magnitude
		is reached, at which time the function returns the steering force 
		accumulated to that  point.
		
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		force = Vector2D()
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.wall_avoidance):
			#force = self.wallAvoidance(self.vehicle.terrain.Walls()) * self.weightWallAvoidance
			#if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		## END NOT IMPLEMENTED ##
		if self._behaviourOn(behavior_type.obstacle_avoidance):
			force = self.obstacleAvoidance(self.vehicle.terrain.Obstacles()) * self.weightObstacleAvoidance
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.evade):
			try:
				assert(self.targetAgent_1 and "Evade target not assigned")
				force = self.evade(self.targetAgent_1) * self.weightEvade
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.flee):
			force = self.flee(self.vehicle.targetPoint) * self.weightFlee
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		# these next three can be combined for flocking behavior (wander is
		# also a good behavior to add into this mix)
		if self._behaviourOn(behavior_type.separation):
			force = self.separation(self.vehicle.terrain.Agents()) * self.weightSeparation
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.allignment):
			force = self.alignment(self.vehicle.terrain.Agents()) * self.weightAlignment
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.cohesion):
			force = self.cohesion(self.vehicle.terrain.Agents()) * self.weightCohesion
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.seek):
			force = self.seek(self.vehicle.targetPoint) * self.weightSeek
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.arrive):
			force = self.arrive(self.vehicle.targetPoint, self.deceleration) * self.weightArrive
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.wander):
			force = self.wander() * self.weightWander
			if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		if self._behaviourOn(behavior_type.pursuit):
			try:
				assert(self.targetAgent_1)
				force = self.pursuit(self.targetAgent_1) * self.weightPursuit
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.offset_pursuit):
			try:
				assert (self.targetAgent_1)
				assert (not self.offset.isZero())
				force = self.offsetPursuit(self.targetAgent_1, self.offset)
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.interpose):
			try:
				assert (self.targetAgent_1 and self.targetAgent_2)
				force = self.interpose(self.targetAgent_1, self.targetAgent_2) * self.weightInterpose
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		if self._behaviourOn(behavior_type.hide):
			try:
				assert(self.targetAgent_1)
				force = self.hide(self.targetAgent_1, self.vehicle.terrain.Obstacles()) * self.weightHide
				if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
			except:pass #ignore
		## NOT IMPLEMENTED ##
		#if self._behaviourOn(behavior_type.follow_path):
			#force = self.followPath() * self.weightFollowPath
			#if not self.accumulateForce(self.steeringForce, force): return self.steeringForce
		## END NOT IMPLEMENTED ##
		return self.steeringForce



	def calculateDithered(self):
		"""
		Sums up the active behaviors by assigning a probabilty
		of being calculated to each behavior. It then tests the first priority
		to see if it should be calcukated this simulation-step. If so, it
		calculates the steering force resulting from this behavior. If it is
		more than zero it returns the force. If zero, or if the behavior is
		skipped it continues onto the next priority, and so on.

		NOTE: not implemented
		
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		raise Exception("calculateDithered not implemented")


	###############################
	# HIDE METHODS
	###############################
	def getHidingPosition(self, posOb, radiusOb, posHunter):
		"""
		Given the position of a hunter, and the position and radius of
		an obstacle, this method calculates a position DistanceFromBoundary 
		away from its bounding radius and directly opposite the hunter.
		
		@type posOb: L{svs_core.geometry.vectors.Vector2D}
		@type radiusOb: float
		@type posHunter: L{svs_core.geometry.vectors.Vector2D}
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# calculate how far away the agent is to be from the chosen obstacle's bounding radius
		distanceFromBoundary = 30.0
		distAway = radiusOb + distanceFromBoundary
		# calculate the heading toward the object from the hunter
		toOb = vec2DNormalize(posOb - posHunter)
		# scale it to size and add to the obstacles position to get the hiding spot.
		return (toOb * distAway) + posOb


	###############################
	# PUBLIC METHODS
	###############################
	def calculate(self):
		"""
		Calculates and sums the steering forces from any active behaviors,
		according to the method set in L{SteeringBehaviours.summingMethod}.
		
		@todo: add in space-partitioning.
		@rtpe: L{svs_core.geometry.vectors.Vector2D}
		"""
		# reset the steering force
		self.steeringForce.zero()
		# NOTE: add in space-aprtitioning
		# tag neighbors if any of the following 3 group behaviors are switched on
		if self._behaviourOn(behavior_type.separation) or self._behaviourOn(behavior_type.allignment) or self._behaviourOn(behavior_type.cohesion):
			self.vehicle.terrain.tagVehiclesWithinViewRange(self.vehicle.location, self.viewDistance)

		if self.summingMethod == summing_method.weighted_average:
			self.steeringForce = self.calculateWeightedSum()
		elif self.summingMethod == summing_method.prioritized:
			self.steeringForce = self.calculatePrioritized()
		elif self.summingMethod == summing_method.dithered:
			self.steeringForce = self.calculateDithered()
		else:self.steeringForce = Vector2D(0,0)

		return self.steeringForce


	def forwardComponent(self):
		"""
		Calculates the component of the steering force that is parallel with the vehicle heading.
		
		@rtpe: float
		"""
		return self.vehicle.heading.dot(self.steeringForce)

	def sideComponent(self):
		"""
		Calculates the component of the steering force that is perpendicuar with the vehicle heading.
		
		@rtpe: float
		"""
		return self.vehicle.side.dot(self.steeringForce)

	def setTarget(self, target):
		"""
		Sets new target for steering behaviour.
		"""
		self.target = target

	def addTargetAgent(self, targetAgent):
		"""
		Adds new agent to list of target agents.
		"""
		if targetAgent not in self.targetAgents: self.targetAgents.append(targetAgent)

	def setOffset(self, offset):
		"""
		Sets offset to steering behaviour.
	
		@type offset:L{svs_core.geometry.vectors.Vector2D}
		"""
		self.offset = offset


	def setpath(self, path):
		"""
		Sets path for vehicle to follow. The path is defined as an array of L{svs_core.geometry.vectors.Vector2D}s
		"""
		self.path = path



	def setSummingMethod(self, summingMethod):
		"""
		Sets summing method, this should be a value defined in L{summing_method}.
		"""
		self.summingMethod = summingMethod



	###############################
	# BEHAVIOUR FLAGS
	###############################
	def fleeOn(self):
		"""
		Turns flee behaviour on.
		"""
		self.flags |= behavior_type.flee

	def seekOn(self):
		"""
		Turns seek behaviour on.
		"""
		self.flags |= behavior_type.seek

	def arriveOn(self):
		"""
		Turns arrive behaviour on.
		"""
		self.flags |= behavior_type.arrive

	def wanderOn(self):
		"""
		Turns wander behaviour on.
		"""
		self.flags |= behavior_type.wander

	def pursuitOn(self, targetAgent):
		"""
		Turns pursuit behaviour on and defines target of pursuit.
		"""
		self.flags |= behavior_type.pursuit
		self.targetAgent_1 = targetAgent

	def evadeOn(self, targetAgent):
		"""
		Turns evade behaviour on and defines agent to evade.
		"""
		self.flags |= behavior_type.evade
		self.targetAgent_1 = targetAgent

	def cohesionOn(self):
		"""
		Turns cohesion behaviour on.
		"""
		self.flags |= behavior_type.cohesion

	def separationOn(self):
		"""
		Turns separation behaviour on.
		"""
		self.flags |= behavior_type.separation

	def alignmentOn(self):
		"""
		Turns alignment behaviour on.
		"""
		self.flags |= behavior_type.allignment

	def obstacleAvoidanceOn(self):
		"""
		Turns obstacle avoidance behaviour on.
		"""
		self.flags |= behavior_type.obstacle_avoidance

	def wallAvoidanceOn(self):
		"""
		Turns wall avoidance behaviour on.
		"""
		self.flags |= behavior_type.wall_avoidance

	def followPathOn(self):
		"""
		Turns path following behaviour on.
		"""
		self.flags |= behavior_type.follow_path

	def interposeOn(self, targetAgent_1, targetAgent_2):
		"""
		Turns interpose behaviour on and defines two agents to interpose between.
		"""
		self.flags |= behavior_type.interpose
		self.targetAgent_1 = targetAgent_1
		self.targetAgent_2 = targetAgent_2

	def hideOn(self, targetAgent):
		"""
		Turns hide behaviour on and define sagent to hide from.
		"""
		self.flags |= behavior_type.hide
		self.targetAgent_1 = targetAgent

	def offsetPursuitOn(self, targetAgent, offset):
		"""
		Turns pursuit behaviour on, defines target of pursuit and set offset.
		"""
		self.flags |= behavior_type.offset_pursuit
		self.offset = offset
		self.targetAgent_1 = targetAgent  

	def flockingOn(self):
		"""
		Turns flocking behaviour on.
		"""
		self.cohesionOn()
		self.alignmentOn()
		self.separationOn()
		self.wanderOn()

	def fleeOff(self):
		if self._behaviourOn(behavior_type.behavior_type.flee):self.flags ^= behavior_type.flee

	def seekOff(self):
		if self._behaviourOn(behavior_type.behavior_type.seek):self.flags ^= behavior_type.seek

	def arriveOff(self):
		if self._behaviourOn(behavior_type.behavior_type.arrive):self.flags ^= behavior_type.arrive

	def wanderOff(self):
		if self._behaviourOn(behavior_type.behavior_type.wander):self.flags ^= behavior_type.wander

	def pursuitOff(self):
		if self._behaviourOn(behavior_type.behavior_type.pursuit):self.flags ^= behavior_type.pursuit

	def evadeOff(self):
		if self._behaviourOn(behavior_type.behavior_type.evade):self.flags ^= behavior_type.evade

	def cohesionOff(self):
		if self._behaviourOn(behavior_type.behavior_type.cohesion):self.flags ^= behavior_type.cohesion

	def separationOff(self):
		if self._behaviourOn(behavior_type.behavior_type.separation):self.flags ^= behavior_type.separation

	def alignmentOff(self):
		if self._behaviourOn(behavior_type.behavior_type.allignment):self.flags ^= behavior_type.allignment

	def obstacleAvoidanceOff(self):
		if self._behaviourOn(behavior_type.behavior_type.obstacle_avoidance):self.flags ^= behavior_type.obstacle_avoidance

	def wallAvoidanceOff(self):
		if self._behaviourOn(behavior_type.behavior_type.wall_avoidance):self.flags ^= behavior_type.wall_avoidance

	def followPathOff(self):
		if self._behaviourOn(behavior_type.behavior_type.follow_path):self.flags ^= behavior_type.follow_path

	def interposeOff(self):
		if self._behaviourOn(behavior_type.behavior_type.interpose):self.flags ^= behavior_type.interpose

	def hideOff(self):
		if self._behaviourOn(behavior_type.behavior_type.hide):self.flags ^= behavior_type.hide

	def offsetPursuitOff(self):
		if self._behaviourOn(behavior_type.behavior_type.offset_pursuit):self.flags ^= behavior_type.offset_pursuit

	def flockingOff(self):
		self.cohesionOff()
		self.alignmentOff()
		self.separationOff()
		self.wanderOff()

	def isFleeOn(self):
		return self._behaviourOn(behavior_type.behavior_type.flee)

	def isSeekOn(self):
		return self._behaviourOn(behavior_type.behavior_type.seek)

	def isArriveOn(self):
		return self._behaviourOn(behavior_type.behavior_type.arrive)

	def isWanderOn(self):
		return self._behaviourOn(behavior_type.behavior_type.wander)

	def isPursuitOn(self):
		return self._behaviourOn(behavior_type.behavior_type.pursuit)

	def isEvadeOn(self):
		return self._behaviourOn(behavior_type.behavior_type.evade)

	def isCohesionOn(self):
		return self._behaviourOn(behavior_type.behavior_type.cohesion)

	def isSeparationOn(self):
		return self._behaviourOn(behavior_type.behavior_type.separation)

	def isAlignmentOn(self):
		return self._behaviourOn(behavior_type.behavior_type.allignment)

	def isObstacleAvoidanceOn(self):
		return self._behaviourOn(behavior_type.behavior_type.obstacle_avoidance)

	def isWallAvoidanceOn(self):
		return self._behaviourOn(behavior_type.behavior_type.wall_avoidance)

	def isFollowPathOn(self):
		return self._behaviourOn(behavior_type.behavior_type.follow_path)

	def isInterposeOn(self):
		return self._behaviourOn(behavior_type.behavior_type.interpose)

	def isHideOn(self):
		return self._behaviourOn(behavior_type.behavior_type.hide)

	def isOffsetPursuitOn(self):
		return self._behaviourOn(behavior_type.behavior_type.offset_pursuit)


