# svs_simulation.ai_lib.tasks

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Generic classes for task behaviours.

This module is based on the goal driven behaviours outlined in
Mat Buckland, 2005, Programming Game AI by Example, Wordware:Plano, 
see U{http://www.wordware.com/files/ai}, and U{http://www.ai-junkie.com}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities.lib import Enumeration


#############################
# CONSTANTS
#############################
task_state = Enumeration('task_state',['ACTIVE', 'INACTIVE', 'COMPLETED', 'FAILED'])


#############################
# EXCEPTIONS
#############################
class TaskException(Exception):pass


#############################
# BASE CLASSES
#############################
class Task:
	"""
	Generic task.
	"""
	def __init__(self, agent):
		self.agent = agent
		self.status = task_state.INACTIVE
		self.name = 'undefined task'
		
	def activate(self):
		"""
		Logic to run when the task is activated.
		"""
		self.status = task_state.ACTIVE

	def activateIfInactive(self):
		"""
		Logic to run when the task is activated.
		"""
		if self.status == task_state.INACTIVE:self.status = task_state.ACTIVE

	def process(self):
		"""
		Logic to run each update-step.
		"""
		self.activateIfInactive()

	def terminate(self):
		"""
		Logic to run prior to the task's destruction.
		"""
		self.status = task_state.INACTIVE

	def handleEvent(self, event):
		"""
		Handles events from game system.
		"""
		return False

	def addSubtask(self, subtask):
		"""
		Adds new subtask.
		"""
		raise TaskException("This class does not accept subtasks.")

	def isComplete(self):
		return self.status == task_state.COMPLETED

	def isActive(self):
		return self.status == task_state.ACTIVE

	def isInactive(self):
		return self.status == task_state.INACTIVE

	def hasFailed(self):
		return self.status == task_state.FAILED

	def reactivateIfFailed(self):
		"""
		If L{Task.status} is L{task_state.FAILED} this method 
		sets it to inactive so that the task will be reactivated 
		(replanned) on the next update-step.
		"""
 		if self.hasFailed():self.status = task_state.INACTIVE

	def activateIfInactive(self):
		"""
		Reactivates the task if L{Task.status} is L{task_state.INACTIVE}.
		"""
		if self.isInactive():self.activate()   



class CompositeTask(Task):
	"""
	A composite task has subtasks which it processes as part of its
	own behaviour.
	"""
	def __init__(self, agent):
		Task.__init__(self, agent)
		self.subtasks = []

	def activate(self):
		"""
		Logic to run when the task is activated.
		"""
		pass

	def process(self):
		"""
		Logic to run each update-step.
		"""
		pass

	def terminate(self):
		"""
		Logic to run prior to the task's destruction.
		"""
		pass

	def handleEvent(self, event):
		"""
		Handles event from game system.
		"""
		return self.forwardEventToFrontSubtask(event)

	def processSubtasks():
		"""
		Processes subtasks.
		"""
		# remove all completed and failed tasks from the front of the subtask list
		for task in self.subtasks:
			if task.isComplete() or task.hasFailed():
				task.terminate()
				self.subtasks.pop(0)
			else:break
		#if any subtasks remain, process the one at the front of the list
		if self.subtasks:
			# grab the status of the front-most subtask
			statusOfSubTasks = self.subtask[0].process()
			# we have to test for the special case where the front-most 
			# subtask reports 'completed' *and* the subtask list contains
			# additional tasks.When this is the case, to ensure the parent
			# keeps processing its subtask list we must return the 'active' status.
			if statusOfSubTasks == task_state.COMPLETED and len(self.subtasks) > 1:
				returntask_state.ACTIVE
		else:return task_state.COMPLETED

	def addSubtask(self, subtask):
		"""
		Adds new subtask.
		"""
		if not self.subtasks:self.subtasks = []
		self.subtasks.insert(0, subtask)

	def removeAllSubtasks(self):
		"""
		Clears all subtasks.
		"""
		self.subtasks = []

	def forwardEventToFrontSubtask(self, event):
		"""
		Passes the message to the task at the front of the queue
		"""
		if self.subtasks:return self.subtasks[0].handleEvent(event)
		else:return False


#############################
# TASK MANAGEMENT
#############################
class TaskManager(CompositeTask):
	"""
	Acts as the root of task hierarchy, manages all other tasks.
	"""
	def __init__(self, agent):
		CompositeTask.__init__(self, agent)
		self.evaluators = []

	def activate(self):
		"""
		Logic to run when the task is activated.
		"""
		self.status = task_state.ACTIVE
		self.arbitrate()

	def process(self):
		"""
		Logic to run each update-step.
		"""
		self.activateIfInactive()
		subgoalStatus = self.processSubgoals()
		if subgoalStatus == task_state.COMPLETED or subgoalStatus == task_state.FAILED:
			self.status = task_state.INACTIVE
		return self.status

	def terminate(self):
		"""
		Logic to run prior to the task's destruction.
		"""
		pass

	def arbitrate(self):
		"""
		Judges most desirable task to pursue.
		"""
		best = 0.0
		mostDesirable = None
		for evaluator in self.evaluators:
			desirability = evaluator.calculateDesirability(self.agent)
			if desirability >= best:
				best = desirability
				mostDesirable = evaluator
		if mostDesirable:mostDesirable.setTask(self.agent)
		

class TaskEvaluator:
	"""
	Base class from which evaluators are created.  These are
	used to determine which tasks should be currently performed.
	"""
	def __init__(self, taskClass=None):
		self.taskClass = taskClass

	def calculateDesirability(self, agent):
		"""
		This method is called to determine the desirability level
		of a speicific task.  It returns a value between 0.0 and 1.0.
		"""
		return 0.05 # default low desirability level

	def setTask(self, agent):
		"""
		This sets the current task.
		"""
		agent.setTask(self.taskClass)
