# svs_simulation.ai_lib.whiskerbot

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Simple obstacle avoiding movement behaviour based on a bot with four 
sensor whiskers.

This module is based on a system described in:

Mike Mika, Charles Charla, "Simple, Cheap Pathfinding", in AI Game Programming Wisdom, edited by Steve Rabin, Charles River Media, 2002, pp. 155 - 160

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# internal imports
from svs_simulation.numdata.vectors import Vector2D
from svs_simulation.numdata.geomlib import AxialBounds2D


class WhiskerBot:
	"""
	Simple sensor-based obstacle avoiding movement behaviour.
	"""
	def __init__(self, vehicle):
		self.vehicle = vehicle
		self.partition = self.vehicle.partition
		self.weightEast = 0.0
		self.weightWest = 0.0
		self.weightNorth = 0.0
		self.weightSouth = 0.0
		self.repel = 0.08	# sensor repel strength
		self.biasEast = 0.0	# bias, used to push bot in desired direction
		self.biasWest = 0.0
		self.biasNorth = 0.0
		self.biasSouth = 0.0
		self.sensorRadius = 24	# sensor radius
		self.sensorBounds = AxialBounds2D(width=24, height=24) # area used for sensing within
		self.sensorOffset = 8.0	# sensor offset from bot
		self.movementVector = Vector2D() # distance moved within single update
		self.deceleration = 0.9 # erodes repulsion over time

	def calculate(self):
		"""
		Calculates current position of bot.
		"""
		# west sensor
		self.sensorBounds.setLocation(self.vehicle.getX() - self.sensorOffset, self.vehicle.getY())
		if self.partition.hasObstaclesInArea(self.sensorBounds):self.weightWest += self.repel
		# east sensor
		self.sensorBounds.setLocation(self.vehicle.getX() + self.sensorOffset, self.vehicle.getY())
		if self.partition.hasObstaclesInArea(self.sensorBounds):self.weightEast += self.repel
		# north sensor
		self.sensorBounds.setLocation(self.vehicle.getX(), self.vehicle.getY() - self.sensorOffset)
		if self.partition.hasObstaclesInArea(self.sensorBounds):self.weightNorth += self.repel
		# south sensor
		self.sensorBounds.setLocation(self.vehicle.getX(), self.vehicle.getY() + self.sensorOffset)
		if self.partition.hasObstaclesInArea(self.sensorBounds):self.weightSouth += self.repel
		# update movement vector
		self.movementVector.x = self.weightWest - self.weightEast
		self.movementVector.y = self.weightNorth - self.weightSouth
		# erode movement
		self.weightWest *= self.deceleration
		self.weightEast *= self.deceleration
		self.weightNorth *= self.deceleration
		self.weightSouth *= self.deceleration
		# add bias for attraction
		self.weightWest += self.biasWest
		self.weightEast += self.biasEast
		self.weightNorth += self.biasNorth
		self.weightSouth += self.biasSouth
		# return movement
		return self.movementVector
