# svs_simulation.entities.customentities

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Provides an infrastructure for loading and managing custom simulation entities.

Custom-coded entities for a simulation can be loaded from an external file
and instantiated from instance data in separate files.

NOTE: for later development look at ways of incorporating updates in source code.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities import idbroker
from svs_core.utilities.objectmanager import PersistentObject
from svs_core.filehandling.sourcecode import SourceRepository
from svs_simulation.entities.base_entities import EntityGroup

#############################
# EXCEPTIONS
#############################
class CustomEntityException(Exception):pass

#############################
# FUNCTIONS
#############################
def createCustomEntityFromSVSFile(filename, entityClass):
	"""
	Reads in data file for custom entity and creates it.
	
	@rtype: L{svs_simulation.entities.base_entities.Entity}
	"""
	data = PersistentObject()
	data.read(filename)
	entity = entityClass()
	entity.decode(data.entity)
	entity._uid = idbroker.nextId()
	return entity


#############################
# CLASSES
#############################
class CustomEntityGroup(EntityGroup):
	"""
	Holds a collection of entities all generated from the same
	custom source code, and maintains reference to the source.
	"""
	def __init__(self, name=None, sourceReference=None):
		EntityGroup.__init__(self, name=name)
		self.sourceReference = sourceReference
		if self.sourceReference:self.entityClass = self.sourceReference.moduleInstance.__dict__[self.name]
		else:self.entityClass = None

	def createInstanceFromFile(self, filename):
		"""
		Instantiates version of custom entity class from given file
		and adds it to group.

		@rtype: L{svs_simulation.entities.base_entities.Entity}
		"""
		if not self.entityClass:raise CustomEntityException("Class not created for entity group: '%s'" % self.name)
		entity = createCustomEntityFromSVSFile(filename, self.entityClass)
		self.addMember(entity)
		return entity


class CustomEntityRepository:
	"""
	Manages groups of custom entities and provides repository 
	support for their source code.
	"""
	def __init__(self):
		self.groups = {}
		self.sourceRepository = SourceRepository()

	def createGroupFromSource(self, groupName, sourceCodeFilename):
		"""
		Creates a new group from a loaded source file.
		"""
		if self.groups.has_key(groupName):raise CustomEntityException("Group already exists: '%s'" % groupName)
		sourceRef = self.sourceRepository.loadSourceFile(groupName, sourceCodeFilename, createInstance=True, addToSystem=True)
		# except: bail out
		group = CustomEntityGroup(groupName, sourceRef)
		self.groups[groupName] = group
		return group

	def getGroup(self, groupName):
		"""
		Returns group with specified name.
		"""
		return self.groups.get(groupName, None)

	def getMemberInGroup(self, groupName, memberName):
		"""
		Returns memeber of group with specified name or ID.
		"""
		group = self.groups.get(groupName, None)
		if not group:return None
		return group.members.get(memberName, None)

	def createGroup(self, groupName, sourceRef=None):
		"""
		Creates custom entity group.  If a group with the given
		name already exists, return this.
		"""
		if self.groups.has_key(groupName):return self.groups[groupName]
		group = CustomEntityGroup(groupName, sourceRef)
		self.groups[groupName] = group
		return group

	

