# svs_simulation.network.clients

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Specialised clients for managing a simulation.


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.utilities.constants import sim_const
from svs_core.commands.scriptcommands import *
from svs_core.network.clientuser import ScriptableClient
from svs_core.network.packets import *
from svs_core.utilities.constants import svs_const



#############################
# SIMULATION CLIENT
#############################
class SimulationClient(ScriptableClient):
	"""
	Base client that manages an aspect of a simulation.

	This should be extended by specialised subclasses.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		ScriptableClient.__init__(self, name, passwd)
		self.simHandler = None
		self.isRunning = False
		self.tickDelay = tickDuration / 1000.0

	def reportChange(self):
		"""
		Sends changes in simulation to network.

		This should be 
		"""
		changes = self.simHandler.getChanges()
		if changes:self.notifyListeners(makeDataPacket(self.profile.name, content=changes, label=demo_const.WORLD_UPDATE_LABEL))

	def startClock(self):
		"""
		Starts simulation clock running.
		"""
		if not self.isRunning:self.isRunning=True
		reactor.callLater(self.tickDelay, self.tick)

	def stopClock(self):
		"""
		Stops simulation clock running.
		"""
		self.isRunning = False

	def startSimulation(self):
		"""
		Starts simulation running.
		"""
		if not self.isRunning:self.startClock()
		if not self.simHandler:return
		self.simHandler.start()
		self.reportChanges()

	def stopSimulation(self):
		"""
		Stops simulation running.
		"""
		self.stopClock()
		if not self.simHandler:return
		self.simHandler.stop()
		self.reportChanges()

	def tick(self):
		"""
		Called by clock to update gameworld.
		"""
		if not self.isRunning or not self.simHandler:return
		self.simHandler.update()
		self.reportChanges()
		reactor.callLater(self.tickDelay, self.tick)

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		if not self.simHandler:return ScriptableClient.getDataForLabel(dataRequest)
		return ScriptableClient.getDataForLabel(self, dataRequest)


#############################
# PARTITION CLIENT
#############################
class TerrainClient(SimulationClient):
	"""
	Client that manages a single terrain partiiton within a simulation.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		SimulationClient.__init__(self, name, passwd)
		self.terrain = None
		self.partitionClients = []

	def loadTerrain(self, filename):
		"""
		Load terrain info from local file.
		"""
		self.terrain = createTerrainFromSVSFile(filename)

	def getPartitionModel(self, partitionName):
		"""
		Returns partition with specified name.
		"""
		if partitionName in self.partitionClients: return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_PARTITION) # should be error
		self.partitionClients.append(partitionName)
		return makeDataPacket(self.profile.name, content=self.terrain.getPartition(partitionName), label=sim_const.LABEL_PARTITION)

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		# get partition data
		if dataRequest.label == sim_const.LABEL_PARTITION:
			return self.getPartitionModel(dataRequest.args[sim_const.LABEL_NAME])
		return ScriptableClient.getDataForLabel(self, dataRequest)


#############################
# PARTITION CLIENT
#############################
class PartitionClient(SimulationClient):
	"""
	Client that manages a single terrain partiiton within a simulation.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		SimulationClient.__init__(self, name, passwd)
		self.partitionName = name


	def setupPartition(self, model):
		"""
		Creates partition from received data.
		"""
		self.simHandler = Partition()
		self.simHandler.decode(model)
		print self.simHandler

	#######################
	# NETWORK
	#######################
	def avatarResult_joinClusterGroup(self, group):
		"""
		Handles result of L{joinClusterGroup} action.
		"""
		self.clusterGroup = group
		self.logMessage("joined cluster group <%s>" % self.clusterGroupName)
		self.getPartitionModel(self.partitionName)

	def getPartitionModel(self, partitionName):
		"""
		Retrieves model of partition from terrain client.
		"""
		if not self.terrain:return
		dataRequest = makeDataRequest(self, recipient=self.terrain, label=sim_const.LABEL_PARTITION, args={sim_const.LABEL_NAME:partitionName})
		self.getData(dataRequest)
		#self.startListeningTo(self.terrain, listenFor=sim_const.LABEL_UPDATE)

	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.

		This should be overidden by extending classes.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# partition model
		if dataLabel == sim_const.LABEL_PARTITION:
			self.setupPartition(dataPacket.content)
			pass
		elif dataLabel == sim_const.LABEL_UPDATE:
			#self.updateGameWorld(dataPacket.content)
			pass
		else:self.logMessage("data packet received: <%s>" % dataPacket.content)

	


#############################
# PROVIDER CLIENT
#############################
class SimProviderClient(ScriptableClient):
	"""
	Base client that provides information for other simulation clients.

	This should be extended by specialised subclasses.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		ScriptableClient.__init__(self, name, passwd)
		self.isRunning = False
		self.tickDelay = tickDuration / 1000.0
		self.dataHandler = None

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		if not self.dataHandler:return ScriptableClient.getDataForLabel(dataRequest)
		data = self.dataHandler.getDataForLabel(dataRequest)
		if not data:return ScriptableClient.getDataForLabel(self, dataRequest)
		return data
