# svs_simulation.network.simclients

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Basic clients for managing a simulation.


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.utilities.constants import sim_const
from svs_core.commands.scriptcommands import *
import svs_core.network.clientuser
from svs_core.network.clientuser import ScriptableClient
from svs_core.network.packets import *
from svs_core.utilities.constants import svs_const



#############################
# SIMULATION CLIENT
#############################
class SimulationClientBase:
	"""
	Base client that manages an aspect of a simulation.

	This should be extended by specialised subclasses.
	"""
	def __init__(self, tickDuration=1000):
		self.simHandler = None
		self.terrainName = None
		self.isRunning = False
		self.tickDelay = tickDuration / 1000.0

	def reportChange(self):
		"""
		Sends changes in simulation to network.

		This should be 
		"""
		changes = self.simHandler.getChanges()
		if changes:self.notifyListeners(makeDataPacket(self.profile.name, content=changes, label=demo_const.WORLD_UPDATE_LABEL))

	def startClock(self):
		"""
		Starts simulation clock running.
		"""
		if not self.isRunning:self.isRunning=True
		reactor.callLater(self.tickDelay, self.tick)

	def stopClock(self):
		"""
		Stops simulation clock running.
		"""
		self.isRunning = False

	def startWorld(self):
		"""
		Starts simulation running.
		"""
		if not self.isRunning:self.startClock()
		if not self.simHandler:return
		self.simHandler.start()
		self.reportChanges()

	def stopWorld(self):
		"""
		Stops simulation running.
		"""
		self.stopClock()
		if not self.simHandler:return
		self.simHandler.stop()
		self.reportChanges()

	def tick(self):
		"""
		Called by clock to update gameworld.
		"""
		if not self.isRunning or not self.simHandler:return
		self.simHandler.update()
		self.reportChanges()
		reactor.callLater(self.tickDelay, self.tick)



class SimulationClient(ScriptableClient, SimulationClientBase):
	"""
	Base client that manages an aspect of a simulation.

	This should be extended by specialised subclasses.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		ScriptableClient.__init__(self, name, passwd)
		SimulationClientBase.__init__(self, tickDuration)

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(self, dataRequest)
		if not self.simHandler:return ScriptableClient.getDataForLabel(self, dataRequest)
		return ScriptableClient.getDataForLabel(self, dataRequest)

class GUISimulationClient(svs_core.network.clientuser.GUIClient, SimulationClientBase):
	"""
	Base client that manages an aspect of a simulation.

	This should be extended by specialised subclasses.
	"""
	def __init__(self, name, passwd, guiClass=None, tickDuration=1000):
		svs_core.network.clientuser.GUIClient.__init__(self, name, passwd, guiClass=guiClass)
		SimulationClientBase.__init__(self, tickDuration)

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		if not self.simHandler:return ScriptableClient.getDataForLabel(dataRequest)
		return ScriptableClient.getDataForLabel(self, dataRequest)


#############################
# TERRAIN CLIENT
#############################
class TerrainClient(SimulationClient):
	"""
	Client that manages a single terrain partiiton within a simulation.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		SimulationClient.__init__(self, name, passwd)
		self.terrain = None
		self.partitionClients = []

	def loadTerrain(self, filename):
		"""
		Load terrain info from local file.
		"""
		from svs_simulation.terrain.base_classes import createTerrainFromSVSFile
		self.terrain = createTerrainFromSVSFile(filename)

	def getPartitionModel(self, partitionName):
		"""
		Returns partition with specified name.
		"""
		#print "getPartitionModel:", partitionName
		if partitionName in self.partitionClients: return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_PARTITION) # should be error
		partition = self.terrain.getPartition(partitionName)
		if not partition:return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_PARTITION) # should be error
		self.partitionClients.append(partitionName) # remember to release on disconnection
		return makeDataPacket(self.profile.name, content=partition.encode(), label=sim_const.LABEL_PARTITION)

	def getTerrainModel(self, lod=0):
		"""
		Returns model of self, this can be tailored in detail 
		according to the C{LOD} argument.
		"""
		if not lod:return makeDataPacket(self.profile.name, content=self.terrain.encodeForProxy(), label=sim_const.LABEL_TERRAIN)
		else:return makeDataPacket(self.profile.name, content=self.terrain.encode(), label=sim_const.LABEL_TERRAIN)
			

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		# get terrain data
		if dataRequest.label == sim_const.LABEL_TERRAIN:
			return self.getTerrainModel(dataRequest.args[sim_const.LABEL_LOD])
		# get partition data
		if dataRequest.label == sim_const.LABEL_PARTITION:
			return self.getPartitionModel(dataRequest.args[sim_const.LABEL_NAME])
		return ScriptableClient.getDataForLabel(self, dataRequest)


#############################
# PARTITION CLIENT
#############################
class PartitionClient(SimulationClient):
	"""
	Client that manages a single terrain partiiton within a simulation.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		SimulationClient.__init__(self, name, passwd)
		self.partitionName = name

	def setupPartition(self, model):
		"""
		Creates partition from received data.
		"""
		from svs_simulation.terrain.base_classes import Partition
		self.simHandler = Partition()
		self.simHandler.decode(model)

	def getPartitionModel(self, lod=0):
		"""
		Returns own partition model.
		"""
		if not self.simHandler:return None
		data = self.simHandler.encode() # add LOD stuff
		return makeDataPacket(self.profile.name, content=data, label=sim_const.LABEL_PARTITION)

	#######################
	# NETWORK
	#######################
	def avatarResult_joinClusterGroup(self, group):
		"""
		Handles result of L{joinClusterGroup} action.
		"""
		self.clusterGroup = group
		self.logMessage("joined cluster group <%s>" % self.clusterGroupName)
		self.getPartitionModelFromTerrain(self.partitionName)

	def getPartitionModelFromTerrain(self, partitionName):
		"""
		Retrieves model of partition from terrain client.
		"""
		if not self.terrainName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.terrainName, label=sim_const.LABEL_PARTITION, args={sim_const.LABEL_NAME:partitionName})
		self.getData(dataRequest)
		self.startListeningTo(self.terrainName, listenFor=sim_const.LABEL_UPDATE)


	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.

		This should be overidden by extending classes.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# partition model
		if dataLabel == sim_const.LABEL_PARTITION:
			self.setupPartition(dataPacket.content)
		elif dataLabel == sim_const.LABEL_UPDATE:
			#self.updateGameWorld(dataPacket.content)
			pass
		else:self.logMessage("data packet received: <%s>" % dataPacket.content)

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		# get partition data
		if dataRequest.label == sim_const.LABEL_PARTITION:
			return self.getPartitionModel(dataRequest.args[sim_const.LABEL_LOD])
		return ScriptableClient.getDataForLabel(self, dataRequest)


#############################
# AGENT GROUP CLIENT
#############################
class AgentGroupClient(SimulationClient):
	"""
	Client that manages a group of agents within a simulation.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		SimulationClient.__init__(self, name, passwd)
		self.groupName = name
		self.worldName = None
		self.agentGroup = None

	def setupAgents(self, agentData):
		"""
		Creates partition from received data.
		"""
		from svs_simulation.agents.basic_agents import AgentGroup
		self.agentGroup = AgentGroup()
		self.agentGroup.decode(agentData)

	def getAgents(self, groupName):
		"""
		Returns own agents.
		"""
		return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_AGENTS)

	def getAgent(self, agentName):
		"""
		Returns agent with specified name.
		"""
		return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_AGENT)

	#######################
	# NETWORK
	#######################
	def avatarResult_joinClusterGroup(self, group):
		"""
		Handles result of L{joinClusterGroup} action.
		"""
		self.clusterGroup = group
		self.logMessage("joined cluster group <%s>" % self.clusterGroupName)
		self.getAgentGroupFromWorld(self.groupName)

	def getAgentGroupFromWorld(self, groupName):
		"""
		Retrieves model of partition from terrain client.
		"""
		if not self.worldName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldName, label=sim_const.LABEL_AGENTGROUP, args={sim_const.LABEL_NAME:groupName})
		self.getData(dataRequest)
		self.startListeningTo(self.worldName, listenFor=sim_const.LABEL_UPDATE)


	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.

		This should be overidden by extending classes.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# agents
		if dataLabel == sim_const.LABEL_AGENTS:
			self.setupAgents(dataPacket.content)
		elif dataLabel == sim_const.LABEL_UPDATE:
			#self.updateGameWorld(dataPacket.content)
			pass
		else:self.logMessage("data packet received: <%s>" % dataPacket.content)

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		# get agent data
		if dataRequest.label == sim_const.LABEL_AGENTS:
			return self.getAgents()
		if dataRequest.label == sim_const.LABEL_AGENT:
			return self.getAgent(dataRequest.args[sim_const.LABEL_NAME])
		return ScriptableClient.getDataForLabel(self, dataRequest)



#############################
# PROVIDER CLIENT
#############################
class SimProviderClient(SimulationClient):
	"""
	Base client that provides information for other simulation clients.

	This should be extended by specialised subclasses.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		SimulationClient.__init__(self, name, passwd)
		self.isRunning = False
		self.tickDelay = tickDuration / 1000.0
		self.dataHandler = None

	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overridden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		if not self.dataHandler:return ScriptableClient.getDataForLabel(dataRequest)
		data = self.dataHandler.getDataForLabel(dataRequest)
		if not data:return ScriptableClient.getDataForLabel(self, dataRequest)
		return data
