# svs_simulation.numdata.geomlib

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Objects for representing basic geometric elements.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
import math

# internal imports
from svs_core.utilities.lib import Constants
from svs_simulation.utilities.constants import sim_const
from svs_simulation.numdata.mathlib import math_const


#############################
# CONSTANTS
#############################
geom_const = Constants()
geom_const.deg2rad = math.pi/180.0
geom_const.rad2deg = 180.0/math.pi
geom_const.half_pi = math.pi/2.0
geom_const.clockwise = 1
geom_const.anticlockwise = -1
geom_const.NORTH = 100
geom_const.SOUTH = 101
geom_const.EAST = 102
geom_const.WEST = 103
geom_const.NO_INTERSECT = 200
geom_const.CONTAINED = 201

#############################
# FUNCTIONS
#############################
def intersectsBounds(bounds1, bounds2):
	"""
	Checks if C{bounds2} intersects with C{bounds1}, return True or False.
	"""
	if bounds1.containsPoint(bounds2.origin.x, bounds2.origin.y):return True
	if bounds1.containsPoint(bounds2.origin.x + bounds2.size.dimX, bounds2.origin.y):return True
	if bounds1.containsPoint(bounds2.origin.x, bounds2.origin.y + bounds2.size.dimY):return True
	if bounds1.containsPoint(bounds2.origin.x + bounds2.size.dimX, bounds2.origin.y + bounds2.size.dimY):return True
	return False

def containsBounds(bounds1, bounds2):
	"""
	Checks if C{bounds2} lies within C{bounds1}, return True or False.
	"""
	if not bounds1.containsPoint(bounds2.origin.x, bounds2.origin.y):return False
	if not bounds1.containsPoint(bounds2.origin.x + bounds2.size.dimX, bounds2.origin.y):return False
	if not bounds1.containsPoint(bounds2.origin.x, bounds2.origin.y + bounds2.size.dimY):return False
	if not bounds1.containsPoint(bounds2.origin.x + bounds2.size.dimX, bounds2.origin.y + bounds2.size.dimY):return False
	return True


def angleFromAtoB(aX, aY, bX, bY):
	dX = bX - aX
	dY = bY - aY
	if dX == 0.0 and dY == 0.0:return 0.0
	if dX == 0.0:
		if dY > 0.0:return 90.0
		else:return 270.0
	if dY == 0.0:
		if dX > 0.0:return 0.0
		else:return 180.0	
	angle = math.atan(dY / dX) * geom_const.rad2deg
	if dX > 0.0 and dY > 0.0:return angle
	#print "dX > 0 and dY > 0: return angle"
	if dX < 0.0 and dY > 0.0:return 180.0 + angle
	#print "dX < 0 and dY > 0: return 180 + angle"
	if dX < 0.0 and dY < 0.0:return 180.0 + angle
	#print "dX < 0 and dY < 0: return 180 + angle"
	if dX > 0.0 and dY < 0.0:return 360.0 + angle
	#print "dX > 0 and dY < 0: return 360 + angle"


#############################
# CLASSES
#############################
class PolarCoordinate:
	"""
	Represents a point in polar space, such as on the 
	circumference of a circle.
	"""
	def __init__(self, angle=0.0, radius=0.0):
		self.angle = angle
		self.radius = radius

	def translate(self, angleDelta=0.0, radiusDelta=0.0):
		self.angle += angleDelta
		self.radius += radiusDelta


class Point2D:
	"""
	Represents a point in 2D space.
	"""
	def __init__(self, x=0.0, y=0.0):
		self.x = x
		self.y = y

	def translate(self, xDelta=0.0, yDelta=0.0):
		self.x += xDelta
		self.y += yDelta

	def setLocation(self, x=0.0, y=0.0):
		self.x = x
		self.y = y

	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		return {sim_const.LABEL_X:self.x,
			sim_const.LABEL_Y:self.y}

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		self.x = data.get(sim_const.LABEL_X)
		self.y = data.get(sim_const.LABEL_Y)


class Point3D:
	"""
	Represents a point in 3D space.
	"""
	def __init__(self, x=0.0, y=0.0, z=0.0):
		self.x = x
		self.y = y
		self.z = z

	def translate(self, xDelta=0.0, yDelta=0.0, zDelta=0.0):
		self.x += xDelta
		self.y += yDelta
		self.z += zDelta

	def setLocation(self, x=0.0, y=0.0, z=0.0):
		self.x = x
		self.y = y
		self.z = z

	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		return {sim_const.LABEL_X:self.x,
			sim_const.LABEL_Y:self.y,
			sim_const.LABEL_Z:self.z}

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		self.x = data.get(sim_const.LABEL_X)
		self.y = data.get(sim_const.LABEL_Y)
		self.z = data.get(sim_const.LABEL_Z)



class Size2D:
	"""
	Represents a size in 2D space.
	"""
	def __init__(self, dimX=0.0, dimY=0.0):
		self.dimX = dimX
		self.dimY = dimY

	def scale(self, scaleX=1.0, scaleY=1.0):
		self.dimX = self.dimX + scaleX
		self.dimY = self.dimY + scaleY

	def setDimensions(self, dimX=0.0, dimY=0.0):
		self.dimX = dimX
		self.dimY = dimY

class Rect2D:
	"""
	Represents an orthogonal rectangular area in 2D space.
	"""
	def __init__(self, x=0.0, y=0.0, dimX=0.0, dimY=0.0):
		self.origin = Point2D(x,y)
		self.size = Size2D(dimX, dimY)

	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		return {sim_const.LABEL_X:self.origin.x,
			sim_const.LABEL_Y:self.origin.y,
			sim_const.LABEL_DIMX:self.size.dimX,
			sim_const.LABEL_DIMY:self.size.dimY}

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		self.origin.x = data.get(sim_const.LABEL_X)
		self.origin.y = data.get(sim_const.LABEL_Y)
		self.size.dimX = data.get(sim_const.LABEL_DIMX)
		self.size.dimY = data.get(sim_const.LABEL_DIMY)
	
	def setOrigin(self, x=0.0, y=0.0):
		"""
		Sets origin of bounds.
		"""
		self.origin.setLocation(x,y)

	def setSize(self, dimX=0.0, dimY=0.0):
		"""
		Sets size of bounds.
		"""
		self.size.setDimensions(dimX, dimY)

	def containsPoint(self, x, y):
		"""
		Tests if specified x and y coordinates exist
		within the bounding area, returning C{True} 
		if so, C{False} otherwise.
		"""
		if x < self.origin.x: return False
		if y < self.origin.y: return False
		if x > self.origin.x + self.size.dimX:return False
		if y > self.origin.y + self.size.dimY:return False
		return True

	def intersects(self, minX, minY, maxX, maxY):
		"""
		Checks is specified area intersects with self.
		"""
		if maxX < self.origin.x:return False
		if minX > self.origin.x + self.size.dimX:return False
		if maxY < self.origin.y:return False
		if minY > self.origin.y + self.size.dimY:return False
		return True
		

class SphereBounds2D:
	"""
	Represents a spherical bounding box in 2D space.
	"""
	def __init__(self, world=False):
		self.reset()
		if not world:self.worldbounds = SphereBounds2D(True)

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		return {sim_const.LABEL_X:self.worldbounds.x,
			sim_const.LABEL_Y:self.worldbounds.y,
			sim_const.LABEL_RADIUS:self.radius}

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		x = data[sim_const.LABEL_X]
		y = data[sim_const.LABEL_Y]
		self.radius = data[sim_const.LABEL_RADIUS]
		self.radiusSq = self.radius**2
		self.setLocation(x, y)

	def reset(self):
		"""
		Sets bounds to zero sizes.
		"""
		self.x = 0
		self.y = 0
		self.radiusSq = 0
		self.radius = 0

	def addPoint(self, x, y):
		"""
		Adds points into the radius of the sphere.
		"""
		r = x**2 + y**2
		self.radiusSq = max(r, self.radiusSq)

	def setDimensions(self, width, height):
		"""
		Creates a bounding sphere for an object of
		the specified width and height.
		"""
		self.reset()
		self.addPoint(-width/2.0, -height/2.0)
		self.addPoint(width/2.0, -height/2.0)
		self.addPoint(width/2.0, height/2.0)
		self.addPoint(-width/2.0, height/2.0)
		self.radius = math.sqrt(self.radiusSq)

	def setLocation(self, x, y):
		"""
		Sets location of spehere in world.
		"""
		self.worldbounds.x = x
		self.worldbounds.y = y
		self.worldbounds.radius = self.radius
		self.worldbounds.radiusSq = self.radiusSq

	def translate(self, x, y):
		"""
		Translates location of spehere in world.
		"""
		self.setLocation(self.worldbounds.x + x, self.worldbounds.y + y)

	def collidesWith(self, sphere):
		"""
		Checks if this sphere collides with the specified sphere.

		Returns C{True} or C{False}.
		
		@rtype: boolean
		"""
		dx = self.worldbounds.x - sphere.worldbounds.x
		dy = self.worldbounds.y - sphere.worldbounds.y
		distance = dx**2 + dy**2
		if distance < self.radiusSq + sphere.radiusSq:return True
		else:return False

	def liesWithin(self, minX, minY, maxX, maxY):
		"""
		Checks if this sphere lies in the specified bounding area.

		Returns either C{geom_const.NO_INTERSECT} or a value 
		for the side it intersects with.
		
		@rtype: integer
		"""
		if self.worldbounds.x - self.radius < minX:return geom_const.WEST
		if self.worldbounds.x + self.radius > maxX:return geom_const.EAST
		if self.worldbounds.y - self.radius < minY:return geom_const.SOUTH
		if self.worldbounds.y + self.radius > maxY:return geom_const.NORTH
		return geom_const.CONTAINED

	def containedWithin(self, minX, minY, maxX, maxY):
		"""
		Checks if this sphere is fully contained within the specified bounding area.

		Returns C{True} or C{False}.

		@rtype: boolean
		"""
		return self.liesWithin(minX, minY, maxX, maxY) == geom_const.CONTAINED

	def locatedWithin(self, minX, minY, maxX, maxY):
		"""
		Checks if the location of the sphere lies in the specified bounding area.

		Returns C{True} or C{False}.

		@rtype: boolean
		"""
		if self.worldbounds.x < minX:return False
		if self.worldbounds.x > maxX:return False
		if self.worldbounds.y < minY:return False
		if self.worldbounds.y > maxY:return False
		return True

	def containsPoint(self, x, y):
		"""
		Checks if point lies wihtin sphere.

		Returns C{True} or C{False}.
		
		@rtype: boolean
		"""
		dx = x - self.worldbounds.x
		dy = y - self.worldbounds.y
		distance = dx**2 + dy**2
		if distance < self.radiusSq:return True
		else:return False

	def containsPointInWorld(self, x, y):
		"""
		Checks if point lies within bounds as mapped to world.

		Returns C{True} or C{False}.
		
		@rtype: boolean
		"""
		dx = x - self.worldbounds.x
		dy = y - self.worldbounds.y
		distance = dx**2 + dy**2
		if distance < self.radiusSq:return True
		else:return False


class AxialBounds2D:
	"""
	Represents anaxially-aligned rectangular bounding box in 2D space.
	"""
	def __init__(self, width=0, height=0, world=False):
		self.reset()
		if not world:self.worldbounds = AxialBounds2D(world=True)
		else:self.worldbounds = None
		if width or height:self.setDimensions(width, height)

	def reset(self):
		"""
		Sets bounds to zero sizes.
		"""
		self.x = 0
		self.y = 0
		self.minX = 0
		self.minY = 0
		self.maxX = 0
		self.maxY = 0
		self.dimX = 0
		self.dimY = 0
		self.autosize = False

	def prepareToAutosize(self):
		"""
		Prepares bounds for autosizing
		"""
		self.autosize = True
		self.x = 0
		self.y = 0
		self.minX = math_const.MAX_FLOAT
		self.minY = math_const.MAX_FLOAT
		self.maxX = 0
		self.maxY = 0
		#print "prepareToAutosize: %s, %s, %s, %s" % (self.minX, self.minY, self.maxX,self.maxY)

	def __str__(self):
		"""
		Returns string representation of self.
		"""
		if self.worldbounds: return "AxialBounds2D: %s, %s (%s, %s, %s, %s) (%s, %s, %s, %s)" % (self.worldbounds.x, self.worldbounds.y, self.minX, self.minY, self.maxX,self.maxY, self.worldbounds.minX, self.worldbounds.minY, self.worldbounds.maxX, self.worldbounds.maxY) 
		else: return "AxialBounds2D: %s, %s (%s, %s, %s, %s)" % (self.x, self.y, self.minX, self.minY, self.maxX,self.maxY)

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		return {sim_const.LABEL_X:self.worldbounds.x,
			sim_const.LABEL_Y:self.worldbounds.y,
			sim_const.LABEL_MINX:self.minX,
			sim_const.LABEL_MINY:self.minY,
			sim_const.LABEL_MAXX:self.maxX,
			sim_const.LABEL_MAXY:self.maxY}

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		x = data[sim_const.LABEL_X]
		y = data[sim_const.LABEL_Y]
		self.minX = data[sim_const.LABEL_MINX]
		self.minY = data[sim_const.LABEL_MINY]
		self.maxX = data[sim_const.LABEL_MAXX]
		self.maxY = data[sim_const.LABEL_MAXY]
		self.setLocation(x, y)

	def addPoint(self, x, y):
		"""
		Adds points into the bounding area.
		"""
		if not self.autosize:self.prepareToAutosize()
		if x > self.maxX:self.maxX = x
		if x < self.minX:self.minX = x
		if y > self.maxY:self.maxY = y
		if y < self.minY:self.minY = y
		self.dimX = self.maxX -self.minX
		self.dimY = self.maxY -self.minY
		self.x = self.minX
		self.y = self.minY

	def setDimensions(self, dimX, dimY):
		"""
		Creates a bounding area for an object of
		the specified width and height.
		"""
		self.reset()
		self.addPoint(0.0,0.0)
		self.addPoint(dimX, dimY)

	def setLocation(self, x, y):
		"""
		Sets location of bounds in world.
		"""
		self.worldbounds.x = x
		self.worldbounds.y = y
		self.worldbounds.minX = self.worldbounds.x - self.minX
		self.worldbounds.minY = self.worldbounds.y - self.minY
		self.worldbounds.maxX = self.worldbounds.x + self.maxX
		self.worldbounds.maxY = self.worldbounds.y + self.maxY

	def translate(self, x, y):
		"""
		Translates location of spehere in world.
		"""
		self.setLocation(self.worldbounds.x + x, self.worldbounds.y + y)

	def collidesWith(self, other):
		"""
		Checks if this sphere collides with the specified sphere.

		Returns C{True} or C{False}.
		
		@rtype: boolean
		"""
		if type(other) is SphereBounds2D:
			dx = self.worldbounds.x - sphere.worldbounds.x
			dy = self.worldbounds.y - sphere.worldbounds.y
			distance = dx**2 + dy**2
			if distance < self.radiusSq + sphere.radiusSq:return True
			else:return False
		elif type(other) is AxialBounds2D:pass
			#if other.worldbounds.minX 

	def liesWithin(self, minX, minY, maxX, maxY):
		"""
		Checks if this sphere lies in the specified bounding area.

		Returns either C{geom_const.NO_INTERSECT} or a value 
		for the side it intersects with.
		
		@rtype: integer
		"""
		if self.worldbounds.x - self.radius < minX:return geom_const.WEST
		if self.worldbounds.x + self.radius > maxX:return geom_const.EAST
		if self.worldbounds.y - self.radius < minY:return geom_const.SOUTH
		if self.worldbounds.y + self.radius > maxY:return geom_const.NORTH
		return geom_const.CONTAINED

	def intersects(self, minX, minY, maxX, maxY):
		"""
		Checks is specified area intersects with self.
		"""
		if maxX < self.worldbounds.minX:return False
		if minX > self.worldbounds.maxX:return False
		if maxY < self.worldbounds.minY:return False
		if minY > self.worldbounds.maxY:return False
		return True

	def containedWithin(self, minX, minY, maxX, maxY):
		"""
		Checks if this sphere is fully contained within the specified bounding area.

		Returns C{True} or C{False}.

		@rtype: boolean
		"""
		return self.liesWithin(minX, minY, maxX, maxY) == geom_const.CONTAINED

	def locatedWithin(self, minX, minY, maxX, maxY):
		"""
		Checks if the location of the sphere lies in the specified bounding area.

		Returns C{True} or C{False}.

		@rtype: boolean
		"""
		if self.worldbounds.x < minX:return False
		if self.worldbounds.x > maxX:return False
		if self.worldbounds.y < minY:return False
		if self.worldbounds.y > maxY:return False
		return True

	def containsPoint(self, x, y):
		"""
		Checks if point lies within local bounds.

		Returns C{True} or C{False}.
		
		@rtype: boolean
		"""
		#print "containsPoint: %f, %f" % (x, y)
		#print "%f, %f, %f, %f" % (self.minX, self.minY, self.maxX, self.maxY)
		if x < self.minX:return False
		if x > self.maxX:return False
		if y < self.minY:return False
		if y > self.maxY:return False
		return True

	def containsPointInWorld(self, x, y):
		"""
		Checks if point lies within world bounds.

		Returns C{True} or C{False}.
		
		@rtype: boolean
		"""
		if x < self.worldbounds.minX:return False
		if x > self.worldbounds.maxX:return False
		if y < self.worldbounds.minY:return False
		if y > self.worldbounds.maxY:return False
		return True




class CentredAxialBounds2D(AxialBounds2D):
	"""
	Axial bounds that places its origin in the centre.
	"""
	def __init__(self, width=0, height=0, world=False):
		AxialBounds2D.__init__(self, width=0, height=0, world=False)

	def addPoint(self, x, y):
		"""
		Adds points into the bounding area.
		"""
		if not self.autosize:self.prepareToAutosize()
		if x > self.maxX:self.maxX = x
		if x < self.minX:self.minX = x
		if y > self.maxY:self.maxY = y
		if y < self.minY:self.minY = y
		self.dimX = self.maxX -self.minX
		self.dimY = self.maxY -self.minY
		self.x = self.minX + (self.dimX/2.0)
		self.y = self.minY + (self.dimY/2.0)

	def setLocation(self, x, y):
		"""
		Sets location of bounds in world.
		"""
		self.worldbounds.x = x
		self.worldbounds.y = y
		self.worldbounds.minX = self.worldbounds.x - self.x
		self.worldbounds.minY = self.worldbounds.y - self.y
		self.worldbounds.maxX = self.worldbounds.x + self.x
		self.worldbounds.maxY = self.worldbounds.y + self.y
