# svs_simulation.numdata.transformations

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Functions for handling geometric transformations.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# internal imports
from svs_simulation.numdata.matrices import Matrix2D


def vec2DRotateAroundOrigin(vector, angle):
	"""
	Rotates a vector angle rads around the origin.
	
	@type vector: L{svs_core.numdata.vectors.Vector2D}
	@type angle: float
	"""
	mat = Matrix2D()
	mat.rotate(angle)
	mat.transformVector2D(vector)


def pointToWorldSpace(point, agentHeading, agentSide, agentPosition):
	"""
	Transforms a point from the agent's local space into world space.

	@type point: L{svs_core.numdata.vectors.Vector2D}
	@type agentHeading: L{svs_core.numdata.vectors.Vector2D}
	@type agentSide: L{svs_core.numdata.vectors.Vector2D}
	@type agentPosition: L{svs_core.numdata.vectors.Vector2D}
	@rtype: L{svs_core.numdata.vectors.Vector2D}
	"""
	# make a copy of the point
	transPoint = point
	# create a transformation matrix
	matTransform = Matrix2D()
	# rotate
	matTransform.rotate(agentHeading, agentSide)
	# and translate
	matTransform.translate(agentPosition.x, agentPosition.y)
	# now transform the vertices
	matTransform.transformVector2Ds(transPoint)
	return transPoint

def vectorToWorldSpace(vector, agentHeading, agentSide):
	"""
	Transforms a vector from the agent's local space into world space.

	@type vector: L{svs_core.numdata.vectors.Vector2D}
	@type agentHeading: L{svs_core.numdata.vectors.Vector2D}
	@type agentSide: L{svs_core.numdata.vectors.Vector2D}
	@rtype: L{svs_core.numdata.vectors.Vector2D}
	"""
	# make a copy of the point
	transVec = vector
	# create a transformation matrix
	matTransform = Matrix2D()
	# rotate
	matTransform.rotate(agentHeading, agentSide)
	# now transform the vertices
	matTransform.transformVector2Ds(transVec)
	return transVec

def pointToLocalSpace(point, agentHeading, agentSide, agentPosition):
	"""
	Transforms a point to an agent's local space.

	@type point: L{svs_core.numdata.vectors.Vector2D}
	@type agentHeading: L{svs_core.numdata.vectors.Vector2D}
	@type agentSide: L{svs_core.numdata.vectors.Vector2D}
	@type agentPosition: L{svs_core.numdata.vectors.Vector2D}
	@rtype: L{svs_core.numdata.vectors.Vector2D}
	"""
	# make a copy of the point
	transPoint = point
	# create a transformation matrix
	matTransform = Matrix2D()
	tx = -agentPosition.dot(agentHeading)
	ty = -agentPosition.dot(agentSide)
	# create the transformation matrix
	matTransform._11 = agentHeading.x
	matTransform._12 = agentSide.x
	matTransform._21 = agentHeading.y
	matTransform._22 = agentSide.y
	matTransform._31 = tx         
	matTransform._32 = ty
	# now transform the vertices
	matTransform.transformVector2Ds(transPoint)
	return transPoint


def vectorToLocalSpace(vector, agentHeading, agentSide):
	"""
	Transforms a point to an agent's local space.

	@type vector: L{svs_core.numdata.vectors.Vector2D}
	@type agentHeading: L{svs_core.numdata.vectors.Vector2D}
	@type agentSide: L{svs_core.numdata.vectors.Vector2D}
	@type agentPosition: L{svs_core.numdata.vectors.Vector2D}
	@rtype: L{svs_core.numdata.vectors.Vector2D}
	"""
	# make a copy of the point
	transPoint = vector
	# create a transformation matrix
	matTransform = Matrix2D()
	# create the transformation matrix
	matTransform._11 = agentHeading.x
	matTransform._12 = agentSide.x
	matTransform._21 = agentHeading.y
	matTransform._22 = agentSide.y
	# now transform the vertices
	matTransform.transformVector2Ds(transPoint)
	return transPoint
