# svs_simulation.numdata.vectors

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
2 dimensional vectors and vector functions.

Some of the classes and functions are based on the C{Vector2D} class outlined in
Mat Buckland, 2005, Programming Game AI by Example, Wordware:Plano, 
see U{http://www.wordware.com/files/ai}, and U{http://www.ai-junkie.com}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
import math

# internal imports
from svs_simulation.numdata.geomlib import Point2D, geom_const
from svs_simulation.numdata.mathlib import math_const



class Vector2D(Point2D):
	"""
	Represents a vector in 2D space.
	"""
	def __init__(self, x=0.0, y=0.0):
		Point2D.__init__(self, x, y)

	def __str__(self):
		"""
		Returns string representation of self.
		"""
		return "Vector2D [%f, %f]" % (self.x, self.y)

	def zero(self):
		"""
		Resets vector to zero values.
		"""
		self.x = 0.0
		self.y = 0.0

	def length(self):
		"""
		Returns the length of a 2D vector.
		"""
		return math.sqrt(self.x * self.x + self.y * self.y)

	def lengthSq(self):
		"""
		Returns the squared length of a 2D vector.
		"""
		return (self.x * self.x + self.y * self.y)

	def dot(self, otherVector):
		"""
		Calculates the dot product.
		"""
		return self.x*otherVector.x + self.y*otherVector.y

	def sign(self, otherVector):
		"""
		Returns positive if v2 is clockwise of this vector,
		minus if anticlockwise (Y axis pointing down, X axis to right).
		"""
		if (self.y*otherVector.x > self.x*otherVector.y):return anticlockwise
		else:return clockwise

	def perp(self):
		"""
		Returns a vector perpendicular to this vector.
		"""
		return Vector2D(-self.y, self.x)

	def distance(self, otherVector):
		"""
		Calculates the euclidean distance between two vectors.
		"""
		ySeparation = otherVector.y - self.y
		xSeparation = otherVector.x - self.x
		return math.sqrt(ySeparation*ySeparation + xSeparation*xSeparation)


	def distanceSq(self, otherVector):
		"""
		Calculates the euclidean distance squared between two vectors.
		"""
		ySeparation = otherVector.y - self.y
		xSeparation = otherVector.x - self.x
		return ySeparation*ySeparation + xSeparation*xSeparation


	def truncate(self, maxValue):
		"""
	`	Truncates a vector so that its length does not exceed C{maxValue}.
		"""
		if self.length() > maxValue:
			self.normalize()
			self *= maxValue


	def reflect(self,  norm):
		"""
		Given a normalized vector this method reflects the vector it
		is operating upon (like the path of a ball bouncing off a wall).
		"""
  		self += 2.0 * self.dot(norm) * norm.getReverse()


	def getReverse(self):
		"""
		Returns the vector that is the reverse of this vector.
		"""
		return Vector2D(-self.x, -self.y)


	def normalize(self):
		"""
		Normalizes a 2D Vector.
		"""
		vector_length = self.length()
		self.x /= vector_length
		self.y /= vector_length

	###############################
	# OPERATOR OVERLOADS
	###############################
	def __add__(self, rhs):
		"""
		Overloads '+' operator.
		"""
		result = Vector2D(self.x, self.y)
		result.x += rhs.x
		result.y += rhs.y
		return result

	def __sub__(self, rhs):
		"""
		Overloads '-' operator.
		"""
		result = Vector2D(self.x, self.y)
		result.x -= rhs.x
		result.y -= rhs.y
		return result

	def __div__(self, rhs):
		"""
		Overloads '/' operator.
		"""
		result = Vector2D(self.x, self.y)
		result.x /= rhs
		result.y /= rhs
		return result

	def __rmul__(self, lhs):
		"""
		Overloads '*' operator when vector is on right hand side of operator.
		"""
		result = Vector2D(self.x, self.y)
		result.x *= lhs
		result.y *= lhs
		return result

	def __mul__(self, rhs):
		"""
		Overloads '*' operator when vector is on left hand side of operator.
		"""
		result = Vector2D(self.x, self.y)
		result.x *= rhs
		result.y *= rhs
		return result

	def __cmp__(self, otherVector):
		"""
		Overloads comparison operator.
		"""
		if otherVector.__class__ is not self.__class__:raise ValueError("comparion must be with another Vector2D object")
		if self.x == otherVector.x and self.y == otherVector.y:return 0
		return 1



#############################
# FUNCTIONS
#############################
def vec2DNormalize(vector):
	"""
	Normalizes specified vector.

	@type vector: L{Vector2D}
	@rtype: L{Vector2D}
	"""
	vec = vector
	vector_length = vec.length()
	if vector_length > math_const.EPSILON:
		vec.x /= vector_length
		vec.y /= vector_length
	return vec


def vec2DDistance(vector1, vector2):
	"""
	Returns distance between two vectors.

	@type vector1: L{Vector2D}
	@type vector2: L{Vector2D}
	@rtype: float
	"""
	ySeparation = vector2.y - vector1.y
	xSeparation = vector2.x - vector1.x
	return math.sqrt(ySeparation*ySeparation + xSeparation*xSeparation)


def vec2DDistanceSq(vector1, vector2):
	"""
	Returns distance squared between two vectors.

	@type vector1: L{Vector2D}
	@type vector2: L{Vector2D}
	@rtype: float
	"""
	ySeparation = vector2.y - vector1.y
	xSeparation = vector2.x - vector1.x
	return ySeparation*ySeparation + xSeparation*xSeparation


def vec2DLength(vector):
	"""
	Returns length of vector.

	@type vector: L{Vector2D}
	@rtype: float
	"""
	return math.sqrt(vector.x*vector.x + vector.y*vector.y)

def vec2DLengthSq(vector):
	"""
	Returns length of vector squared.

	@type vector: L{Vector2D}
	@rtype: float
	"""
	return vector.x*vector.x + vector.y*vector.y


def pointToVector(point):
	"""
	Converts point to vector.

	@type point: L{svs_core.numdata.geomlib.Point2D}
	@rtype: L{Vector2D}
	"""
	return Vector2D(point.x, point.y)


def vectorToPoint(vector):
	"""
	Converts vector to point.

	@type vector: L{Vector2D}
	@rtype: L{svs_core.numdata.geomlib.Point2D}
	"""
	return Point2D(vector.x, vector.y)


def insideBounds2D(point, top_left=None, bottom_right=None, left=None, top=None, right=None, bottom=None):
	"""
	Returns true if the point is inside the defined region.

	The region can either be defined by two points or vectors, or by four coordinates.
	"""
	if top_left and bottom_right:
		return not ((point.x < top_left.x) or (point.x > bottom_right.x) or (point.y < top_left.y) or (point.y > bottom_right.y))
	else:
		return not ((point.x < left) or (point.x > right) or (point.y < top) or (point.y > bottom))

def notInsideBounds2D(point, top_left=None, bottom_right=None, left=None, top=None, right=None, bottom=None):
	"""
	Returns true if the point is NOT inside the defined region.

	The region can either be defined by two points or vectors, or by four coordinates.
	"""
	if top_left and bottom_right:
		return ((point.x < top_left.x) or (point.x > bottom_right.x) or (point.y < top_left.y) or (point.y > bottom_right.y))
	else:
		return ((point.x < left) or (point.x > right) or (point.y < top) or (point.y > bottom))


def isSecondInFOVOfFirst(posFirst, facingFirst, posSecond, fov):
		"""
		Returns true if the target position is in the field of view of the entity
		positioned at posFirst facing in facingFirst.

		@type posFirst:L{Vector2D}
		@type facingFirst:L{Vector2D}
		@type posSecond:L{Vector2D}
		@type fov:float
		"""
		toTarget = vec2DNormalize(posSecond - posFirst)
		return facingFirst.dot(toTarget) >= math.cos(fov/2.0)
