# svs_simulation.simtime.schedules

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Schedules for time-based activities.

NOTES: needs class for matching simulation time against real-world calender dates.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.simtime.timelib import time_const, SimTime
from svs_simulation.utilities.constants import sim_const
from svs_core.utilities.lib import PathBasedCollection
from svs_simulation.actions.actionlib import SimAction


#######################
# EXCEPTIONS
#######################
class ScheduleException(Exception):pass


#######################
# FUNCTIONS
#######################
def createScheduleFromSVSFile(filename):
	from svs_core.utilities.objectmanager import PersistentObject
	from svs_core.utilities.lib import validateFilePath
	filename = validateFilePath(filename)
	data = PersistentObject()
	data.read(filename)
	schedule = Schedule()
	schedule.decode(data.schedule)
	return schedule


#######################
# SCHEDULE DATA CLASSES
#######################
class Schedule:
	"""
	Represents a schedule of actions or actions.
	"""
	def __init__(self, name=None):
		self.name = name
		self.timeline = PathBasedCollection()

	def addAction(self, timecode, action):
		"""
		Adds an action at the specified time.
		"""
		self.timeline.addEntry(timecode.toTuple(), action)

	def getActions(self, timecode):
		"""
		Returns actions for specified timecode.
		"""
		#return self.timeline.get(timecode.toTuple()) # proper version
		# hack for exhib
		tcode = timecode.toTuple()
		return self.timeline.get((0,0,tcode[2], tcode[3]))

	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded representation of self.
		"""
		data = {sim_const.LABEL_NAME:self.name}
		data[sim_const.LABEL_TIMELINE] = self.encodeEntries(self.timeline.collection)
		return data

	def encodeEntries(self, node):
		data = {}
		for key, value in node.items():
			if key == PathBasedCollection.dataKey:
				entryData = []
				for entry in value:entryData.append(entry.encode())
				data[key] = entryData
			else:data[key] = self.encodeEntries(value)
		return data
		

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		self.name = data.get(sim_const.LABEL_NAME, None)
		entryData = data.get(sim_const.LABEL_TIMELINE, None)
		if entryData:self.timeline.collection = self.decodeEntries(entryData)

	def decodeEntries(self, node):
		data = {}
		for key, value in node.items():
			if key == PathBasedCollection.dataKey:
				entryData = []
				for entry in value:
					action = SimAction()
					action.decode(entry)
					entryData.append(action)
				data[key] = entryData
			else:data[key] = self.decodeEntries(value)
		return data



#######################
# PLAYER
#######################
class SchedulePlayer:
	"""
	Provides playback for schedules.
	"""
	def __init__(self, schedule=None, 
			startTime=time_const.DEFAULT_STARTTIME, 
			endTime=time_const.DEFAULT_ENDTIME,
			timeStep=time_const.DEFAULT_TIMESTEP,
			looping=False):
		self.schedule = schedule
		self.startTime = startTime
		self.endTime = endTime
		self.timeStep = timeStep
		self.looping = looping
		self.reset()

	def setSchedule(self, schedule, startTime=None):
		"""
		Sets new schedule and resets playback.
		"""
		self.schedule = schedule
		if startTime:self.startTime = startTime
		self.reset()

	def reset(self):
		"""
		Resets playback time to start.
		"""
		self.currentTime = self.startTime
		self.playing = False
		

	def start(self, timeStep=None):
		"""
		Starts playback.  If time step is specified uses this
		instead of default value (L{time_const.DEFAULT_TIMESTEP}).
		"""
		if timeStep:self.timeStep = timeStep
		self.playing = True

	def stop(self):
		"""
		Stops playback.
		"""
		self.playing = False

	def nextFrame(self):
		"""
		Returns next frame in schedule.
		"""
		if not self.playing:return None
		if not self.schedule:return None
		actions = self.schedule.getActions(self.currentTime)
		self.updateTime()
		return actions

	def updateTime(self):
		"""
		Increments L{SchedulePlayer.currentTime} by L{SchedulePlayer.timeStep}.
		"""
		self.currentTime += self.timeStep
		if self.currentTime >= self.endTime:
			if not self.looping:self.stop()
			else:self.currentTime = self.startTime

	def updateWorld(self, simTime):
		"""
		Responds to update cycle in world.
		"""
		pass

	


#######################
# MANAGER
#######################
class ScheduleManager:
	"""
	Manages use of schules within simulation.
	"""
	def __init__(self):
		self.target = None
		self.schedules = {}
		self.calender = None # needs class for matching simulation time against real-world calender dates.
		self.currentSchedule = None
		self.prevTimecode = SimTime(seconds=-1)
		self.player = SchedulePlayer()
		self.weekdays = {sim_const.LABEL_MONDAY:None,
			sim_const.LABEL_TUESDAY:None,
			sim_const.LABEL_WEDNESDAY:None,
			sim_const.LABEL_THURSDAY:None,
			sim_const.LABEL_FRIDAY:None,
			sim_const.LABEL_SATURDAY:None,
			sim_const.LABEL_SUNDAY:None}

	def addSchedule(self, schedule):
		"""
		Adds schedule.
		"""
		if schedule.name == None:return
		self.schedules[schedule.name] = schedule

	def setScheduleForDay(self, dayName, schedule):
		"""
		Sets schedule for specified day, this overwrites any existing
		schedule for that day.
		"""
		if not self.weekdays.has_key(dayName):raise ScheduleException("Invalid name of day: %s" % dayName)
		self.weekdays[dayName] = schedule

	def getScheduleForDay(self, dayName):
		"""
		Returns schedule for specified day.
		"""
		return self.weekdays.get(dayName, None)
	
	def getActions(self, timecode):
		"""
		Returns actions for specified timecode.
		"""
		#print "getActions:", self.currentSchedule
		if not self.currentSchedule:return None
		return self.currentSchedule.getActions(timecode)


	def updateWorld(self, simTime):
		"""
		Responds to update cycle in world.
		"""
		#print "ScheduleManager updateWorld:", simTime[0]
		if simTime[0] == self.prevTimecode:return
		self.prevTimecode = simTime[0]
		return self.getActions(simTime[0])
