# svs_simulation.simtime.timelib

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Representation of time within simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities.constants import Constants
from svs_simulation.numdata.mathlib import math_const
from svs_simulation.utilities.constants import sim_const


#######################
# EXCEPTIONS
#######################
class SimTimeException(Exception):pass

#######################
# CLASSES
#######################
class SimTime:
	"""
	Class representing a specific time.
	"""
	def __init__(self, days=0, hours=0, minutes=0, seconds=0):
		self.days = days
		self.hours = hours % 24
		self.minutes = minutes % 60
		self.seconds = seconds % 60
		self.__totalise()

	def __totalise(self):
		"""
		Calculates time as total seconds.
		"""
		self.totalSeconds = self.seconds + (self.minutes * 60) + (self.hours * 3600) + (self.days * 86400)

	def __str__(self):
		"""
		Returns string representation of self.
		"""
		return "%s:%s:%s:%s" % (self.days, self.hours, self.minutes, self.seconds)

	def toTuple(self):
		"""
		Returns values as tuple, in the form: (days, hours, minutes, seconds).
		"""
		return (self.days, self.hours, self.minutes, self.seconds)

	#############################
	# OPERATORS
	#############################
	def __add__(self, newValue):
		"""
		Adds specified time value to own value.
		"""		
		newTime = SimTime()
		newSecs = self.seconds + newValue.seconds
		carryMins = 0
		carryHours = 0
		carryDays = 0
		if newSecs >= 60:carryMins = newSecs/60.0
		#print "carryMins:", carryMins
		newTime.seconds = int(newSecs % 60)
		newMins = self.minutes + newValue.minutes + carryMins
		#print "newMins:", newMins
		if newMins >= 60:carryHours = newMins/60.0
		newTime.minutes = int(newMins % 60)
		newHours = self.hours + newValue.hours + carryHours
		if newHours >= 24:carryDays = newHours/24.0
		newTime.hours = int(newHours % 24)
		newTime.days = self.days + newValue.days + carryDays
		newTime.__totalise()
		return newTime
		
	def __sub__(self, rhs):
		"""
		Overloads '-' operator.

		NOTE: not implemented, just returns self at present.
		"""
		result = self
		return result

	def __div__(self, rhs):
		"""
		Overloads '/' operator.

		NOTE: not implemented, just returns self at present.
		"""
		result = self
		return result

	def __rmul__(self, lhs):
		"""
		Overloads '*' operator when vector is on right hand side of operator.

		NOTE: not implemented, just returns self at present.
		"""
		result = self
		return result

	def __mul__(self, rhs):
		"""
		Overloads '*' operator when vector is on left hand side of operator.

		NOTE: not implemented, just returns self at present.
		"""
		result = self
		return result

	def __cmp__(self, otherTime):
		"""
		Overloads comparison operator.
		"""
		if otherTime.__class__ is not self.__class__:raise ValueError("comparion must be with another Vector2D object")
		if self.totalSeconds > otherTime.totalSeconds:return 1
		elif self.totalSeconds < otherTime.totalSeconds:return -1
		return 0

	#############################
	# PROPERTIES
	#############################
	def setHours(self, newValue):
		"""
		Sets hours ensuring it is within valid bounds.
		"""
		self.hours = newValue % 24
		self.__totalise()

	def setMinutes(self, newValue):
		"""
		Sets minutes ensuring it is within valid bounds.
		"""
		self.minutes = newValue % 60
		self.__totalise()

	def setSeconds(self, newValue):
		"""
		Sets seconds ensuring it is within valid bounds.
		"""
		self.seconds = newValue % 60
		self.__totalise()
		
	def encode(self):
		"""
		Returns encoded representation of self.
		"""
		return (self.day, self.hours, self.minutes, self.seconds)

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		if len(data) < 4:raise SimTimeException("Invalid format for encoded SimTime data.")
		self.day = int(data[0])
		self.setHours(int(data[1]))
		self.setMinutes(int(data[2]))
		self.setSeconds(int(data[3]))


#######################
# CONSTANTS
#######################
time_const = Constants()

# defaults
time_const.DEFAULT_TIMESTEP = SimTime(seconds=1.0)
time_const.DEFAULT_STARTTIME = SimTime()
time_const.DEFAULT_ENDTIME = SimTime(days=math_const.MAX_INTEGER)
	
#######################
# CLOCK
#######################
class SimClock:
	"""
	Provides time playback in simulation.
	"""
	def __init__(self, startTime=time_const.DEFAULT_STARTTIME, 
			endTime=time_const.DEFAULT_ENDTIME,
			timeStep=time_const.DEFAULT_TIMESTEP,
			timeStepDuration=1.0, 
			looping=False):
		self.setup(startTime=startTime, 
			endTime=endTime,
			timeStep=timeStep,
			timeStepDuration=timeStepDuration, 
			looping=looping)

	def reset(self):
		"""
		Resets playback time to start.
		"""
		self.currentTime = self.startTime
		self.running = False
		self.prevTickTime = 0.0
		self.prevRealTime = 0.0

	def start(self, timeStep=None):
		"""
		Starts playback.  If time step is specified uses this
		instead of default value (l{time_const.DEFAULT_TIMESTEP}).
		"""
		if timeStep:self.timeStep = timeStep
		self.running = True

	def stop(self):
		"""
		Stops playback.
		"""
		self.running = False

	def tick(self):
		"""
		Increments L{SimClock.currentTime} by L{SimClock.timeStep}.
		"""
		self.currentTime += self.timeStep
		if self.currentTime >= self.endTime:
			if not self.looping:self.stop()
			else:self.currentTime = self.startTime

	def update(self, realTime):
		"""
		Updates clock in relation to real time.

		This is called by an external event loop.
		"""
		if (realTime - self.prevTickTime) >= self.timeStepDuration:
			self.tick()
			self.prevTickTime = realTime
		tickle = (realTime - self.prevRealTime) / self.timeStepDuration
		self.prevRealTime = realTime
		return self.currentTime, tickle

	#############################
	# ARCHIVING
	#############################
	def setup(self, startTime=time_const.DEFAULT_STARTTIME, 
			endTime=time_const.DEFAULT_ENDTIME,
			timeStep=time_const.DEFAULT_TIMESTEP,
			timeStepDuration=1.0, 
			looping=False):
		"""
		Initialises basic clock properties.
		"""
		self.startTime = startTime
		self.endTime = endTime
		self.timeStep = timeStep
		self.timeStepDuration = timeStepDuration
		self.looping = looping
		self.reset()
		

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		data = {sim_const.LABEL_STARTTIME:self.startTime.encode(),
			sim_const.LABEL_ENDTIME:self.endTime.encode(),
			sim_const.LABEL_TIMESTEP:self.timeStep.encode(),
			sim_const.LABEL_TIMESTEPDURATION:self.timeStepDuration,
			sim_const.LABEL_LOOPING:self.looping}
		return data

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		self.startTime = SimTime()
		self.startTime.decode(data[sim_const.LABEL_STARTTIME])
		self.endTime = SimTime()
		self.endTime.decode(data[sim_const.LABEL_ENDTIME])
		self.timeStep = SimTime()
		self.timeStep.decode(data[sim_const.LABEL_TIMESTEP])
		self.timeStepDuration = data[sim_const.LABEL_TIMESTEPDURATION]
		self.looping = data[sim_const.LABEL_LOOPING]
		self.reset()
		
