# svs_simulation.terrain.linkpaths

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Link paths defines points of movement between two structures.

They are used to generate high-level paths of structures across
a terrain as well as define entrance and exit points on buildings.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.utilities.constants import sim_const
from svs_simulation.terrain.paths import PathNode
from svs_core.utilities.lib import Constants

#############################
# EXCEPTIONS
#############################
class LinkPathException(Exception):pass

#############################
# CONSTANTS
#############################
linkpath_const = Constants()

linkpath_const.TRANSITION_ENTRANCE = 400
linkpath_const.TRANSITION_OPEN = 401
linkpath_const.TRANSITION_PATH = 402
linkpath_const.TRANSITION_WORLDEXIT = 403


#############################
# CLASSES
#############################
class LinkPath:
	"""
	Defines a point of movement between two structures.
	"""
	def __init__(self, startStructure=None, startNode=None, endStructure=None, endNode=None):
		self.startStructure = startStructure
		self.startNode = startNode
		self.endStructure = endStructure
		self.endNode = endNode
		self.transition = linkpath_const.TRANSITION_OPEN

	def __str__(self):
		"""
		Returns string representation of self.
		"""
		if self.isResolved():return "linkpath [%s (%f, %f), %s (%f, %f)]" % (self.startStructure.name, self.startNode.x, self.startNode.y, self.endStructure.name, self.endNode.x, self.endNode.y)
		else:return "linkpath unresolved [(%f, %f), (%f, %f)]" % (self.startNode.x, self.startNode.y, self.endNode.x, self.endNode.y)

	def isResolved(self):
		"""
		Checks if structures are defined for path.
		Returns C{True} or C{False}.
		"""
		return self.startStructure and self.endStructure

	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded description of self.

		NOTE: only the nodes of the path are encoded, these 
		should always be resolved against the terrain to ensure
		they map to the correct structures.

		@rtype:dict
		"""
		data = {}
		if self.startNode:data[sim_const.LABEL_START] = self.startNode.encode()
		if self.endNode:data[sim_const.LABEL_END] = self.endNode.encode()
		data[sim_const.LABEL_TRANSITION] = self.transition
		return data

	def decode(self, data):
		"""
		Sets self to encoded values.

		NOTE: only the nodes of the path are encoded, these 
		should always be resolved against the terrain to ensure
		they map to the correct structures.

		@type data:dict
		"""
		if data.has_key(sim_const.LABEL_START):
			self.startNode = PathNode()
			self.startNode.decode(data[sim_const.LABEL_START])
		if data.has_key(sim_const.LABEL_END):
			self.endNode = PathNode()
			self.endNode.decode(data[sim_const.LABEL_END])
		self.transition = data.get(sim_const.LABEL_TRANSITION, linkpath_const.TRANSITION_OPEN)


class LinkPathGraph:
	"""
	Stores all link paths, and enable high-level paths to be constructed from them.
	"""
	def __init__(self, name=None):
		self.name = name
		self.paths = {}
		self.resolvedPaths = []
		self.unresolvedLinkPaths = []

	def addLinkPath(self, path):
		"""
		Adds path to graph.
		"""
		if path in self.resolvedPaths:return
		if not path.isResolved():
			self.unresolvedLinkPaths.append(path)
			return
		self.resolvedPaths.append(path)
		if not self.paths.has_key(path.startStructure.name): self.paths[path.startStructure.name] = []
		self.paths[path.startStructure.name].append(path)
		if not self.paths.has_key(path.endStructure.name): self.paths[path.endStructure.name] = []
		self.paths[path.endStructure.name].append(path)

	def getPathsBetween(self, startStructureName, endStructureName):
		"""
		Returns list of link paths between two structures.
		"""
		startList = self.paths.get(startStructure, None)
		if not startList:return None
		retPaths = []
		for path in startList:
			if path.endStructure == endStructure:retPaths.append(path)
		return retPaths

	def getClosestPathBetween(self, startStructure, endStructure, x, y):
		"""
		Returns path that has start node closest to specified position.
		"""
		from svs_simulation.numdata.mathlib import math_const
		pathList = self.getPathsBetween(startStructure, endStructure)
		if not pathList:return None
		distance = math_const.MAX_FLOAT
		closest = None
		for path in pathList:
			delta = abs(path.startNode.x - x) + abs(path.startNode.y - y)
			if delta < distance:
				distance = delta
				closest = path
		return closest

	def getAllPaths(self):
		return self.resolvedPaths

	def resolveLinkPaths(self, terrain):
		"""
		Resolves all paths in L{LinkPathGraph.unresolvedLinkPaths} 
		and adds them to main list. L{LinkPathGraph.unresolvedLinkPaths} 
		is then cleared.
		"""
		for path in self.unresolvedLinkPaths:
			mapLinkPathToStructures(path, terrain)
			if path.isResolved():self.addLinkPath(path)
			else:
				print "Unable to resolve path: %s" % path
				#raise LinkPathException("Unable to resolve path: %s" % path)
		self.unresolvedLinkPaths = []
		
			

	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		data = {}
		if self.name:data[sim_const.LABEL_NAME] = self.name
		pathData = []
		for path in self.resolvedPaths:pathData.append(path.encode())
		data[sim_const.LABEL_PATHS] = pathData
		return data

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		if data.has_key(sim_const.LABEL_NAME):self.name = data[sim_const.LABEL_NAME]
		pathData = data.get(sim_const.LABEL_PATHS, None)
		if not pathData:return
		for pathEntry in pathData:
			path = LinkPath()
			path.decode(pathEntry)
			self.addLinkPath(path)
		
	
#############################
# FUNCTIONS
#############################
def createLinkPath(transitionLabel, vertices):
	"""
	Factory method to create a link path with given name and vertices.
	"""
	if not transitionLabel:raise LinkPathException("Unable to create link path, no transition type defined.")
	if len(vertices) < 2:raise LinkPathException("Unable to create link path, insufficient vertices: %s" % vertices)
	startNode = PathNode(x=vertices[0][0], y=vertices[0][1])
	endNode = PathNode(x=vertices[1][0], y=vertices[1][1])
	path = LinkPath(startNode=startNode, endNode=endNode)
	if transitionLabel == sim_const.LABEL_ENTRANCE:path.transition = linkpath_const.TRANSITION_ENTRANCE
	elif transitionLabel == sim_const.LABEL_PATH:path.transition = linkpath_const.TRANSITION_PATH
	if transitionLabel == sim_const.LABEL_WORLDEXIT:path.transition = linkpath_const.TRANSITION_WORLDEXIT
	else:path.transition = linkpath_const.TRANSITION_OPEN
	return path

def mapLinkPathToStructures(path, terrain):
	"""
	Maps the locations of the start and end nodes within a link path
	to structures within the specified terrain.
	"""
	path.startStructure = terrain.getStructureContainingPoint(path.startNode.x, path.startNode.y)
	path.endStructure = terrain.getStructureContainingPoint(path.endNode.x, path.endNode.y)
