# svs_simulation.terrain.navgraphs

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Navigation graphs.

The nodes of the graph are represented by custom classes contained in
this module.  The main graph istelf is handled by the undirected
graph from C{pygraphlib}:

U{http://pygraphlib.sourceforge.net/}


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from pygraphlib.pygraph import UGraph

# internal imports
from svs_simulation.numdata.geomlib import Point2D
from svs_simulation.utilities.constants import sim_const
from svs_simulation.simdata.simdatalib import SimData


#############################
# FUNCTIONS
#############################
def createNavGraphForStructure(structure, seedX, seedY, spaceX, spaceY, agentRadius):
	"""
	Builds a navigation graph for inside a structure.

	This uses a flood fill algorithm based on:
	U{http://www.student.kuleuven.ac.be/~m0216922/CG/floodfill.html}.

	@var structure: structure that the garph is built for
	@type structure: L{svs_simulation.terrain.structures.Structure}
	@var seedPoint: starting point for generating graph
	@type seedPoint: L{Point2D}
	@var spaceX: space between nodes on x axis
	@type spaceX: float
	@var spaceY: space between nodes on y axis
	@type spaceY: float
	@var agentRadius: standard agent radius, used to create collision margin from edges of spaces
	@type agentRadius: float
	"""
	graph = NavGraph()
	# horizontal flood - right
	nodeX = seedX
	nodeY = seedY
	countA = 0
	#print "seedX, seedY: %f, %f" % (seedX, seedY)
	while 1:
		if structure.bounds.worldbounds.containsPoint(nodeX, nodeY):
			#print "in worldbounds"
			_fillVertical(structure, graph, nodeX, nodeY, spaceX, spaceY, agentRadius)
			nodeX += spaceX
			countA += 1
		else:
			#print "not in worldbounds"
			break
	# horizontal flood - left
	print "countA:", countA
	nodeX = seedX - spaceX
	countB = 0
	while 1:
		#if structure.floorplane.containsNavPoint(nodeX, nodeY):
		if structure.bounds.worldbounds.containsPoint(nodeX, nodeY):
			_fillVertical(structure, graph, nodeX, nodeY, spaceX, spaceY, agentRadius)
			nodeX -= spaceX
			countB += 1
		else:break
	print "countB:", countB
	return graph


def _fillVertical(structure, graph, startX, startY, spaceX, spaceY, agentRadius):
	# vertical flood - up
	nodeX = startX
	nodeY = startY
	while 1:
		if structure.floorplane.withinBounds(nodeX, nodeY):
			if structure.floorplane.containsNavPoint(nodeX, nodeY):
				_linkNeighboursFromNode(graph.addNode(nodeX, nodeY), structure, graph, spaceX, spaceY)
			nodeY += spaceY
		else:break
	# vertical flood - down
	nodeY = startY - spaceY
	while 1:
		if structure.floorplane.withinBounds(nodeX, nodeY):
			if structure.floorplane.containsNavPoint(nodeX, nodeY):
				_linkNeighboursFromNode(graph.addNode(nodeX, nodeY), structure, graph, spaceX, spaceY)
			nodeY -= spaceY
		else:break

def _linkNeighboursFromNode(node, structure, graph, spaceX, spaceY):
	"""
	Creates links to neighbour points around a node.
	"""
	# north
	if structure.floorplane.containsNavPoint(node.x, node.y - spaceY):
		nodeN = graph.addNode(node.x, node.y - spaceY)
		graph.addEdge(node, nodeN)
	# south
	if structure.floorplane.containsNavPoint(node.x, node.y + spaceY):
		nodeS = graph.addNode(node.x, node.y + spaceY) 
		graph.addEdge(node, nodeS)
	# east
	if structure.floorplane.containsNavPoint(node.x + spaceX, node.y):
		nodeE = graph.addNode(node.x + spaceX, node.y) 
		graph.addEdge(node, nodeE)
	# west 
	if structure.floorplane.containsNavPoint(node.x - spaceX, node.y):
		nodeW = graph.addNode(node.x - spaceX, node.y)
		graph.addEdge(node, nodeW)
	# north-east 
	if structure.floorplane.containsNavPoint(node.x + spaceX, node.y - spaceY):
		nodeNE = graph.addNode(node.x + spaceX, node.y - spaceY)
		graph.addEdge(node, nodeNE)
	# north-west 
	if structure.floorplane.containsNavPoint(node.x - spaceX, node.y - spaceY):
		nodeNW = graph.addNode(node.x - spaceX, node.y - spaceY)
		graph.addEdge(node, nodeNW)
	# south-east 
	if structure.floorplane.containsNavPoint(node.x + spaceX, node.y + spaceY):
		nodeSE = graph.addNode(node.x + spaceX, node.y + spaceY)
		graph.addEdge(node, nodeSE)
	# south-west
	if structure.floorplane.containsNavPoint(node.x - spaceX, node.y + spaceY):
		nodeSW = graph.addNode(node.x - spaceX, node.y + spaceY) 
		graph.addEdge(node, nodeSW)


def linkNodesWithinTransition(self, graph, transition): ### maybe ...
	"""
	Links nodes in graph that are within
	transition area to transition.
	"""
	nodesInTransition = []
	for node in graph.transitionNodes.values():
		if transition.containsPoint(node.x, node.y):
			nodesInTransition.append(node)
			node.transitionLink = transition
	
		

#############################
# CLASSES
#############################
class NavGraphNode(Point2D):
	"""
	Node in a navigation graph.
	"""
	def __init__(self, x, y, simdata=None):
		Point2D.__init__(self, x=x, y=y)
		self.simdata = simdata
		self.transitionLink = None
		self.occupant = None

	def isTransitionNode(self):
		"""
		Returns C{True} if transition node, otherwise C{False}.
		"""
		return not self.transitionLink == None


	#############################
	# NAVIGATION
	#############################
	def enterNode(self, entity):
		self.occupant = entity

	def exitNode(self, entity):
		self.occupant = None

	def isOccupied(self):
		if self.occupant:return True
		return False

	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		data = Point2D.encode(self)
		if self.simdata:data[sim_const.LABEL_SIMDATA] = self.simdata.encode()
		return data

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		Point2D.decode(self, data)
		if data.has_key(sim_const.LABEL_SIMDATA):
			self.simdata = SimData()
			self.simdata.decode(data[sim_const.LABEL_SIMDATA])


class NavGraph(UGraph):
	"""
	Navigation graph.
	"""
	def __init__(self, name=None):
		UGraph.__init__(self)
		self.name = name
		self.nodePoints = {}
		self.edgePoints = {}
		self.transitionNodes = {}

	#############################
	# NODES
	#############################
	def addNode(self, x, y, simdata=None):
		"""
		Creates new node and adds it to graph.

		Checks that node does not already exist, if it does
		return the existing node rather than a new one.
		"""
		point = (x,y)
		if self.nodePoints.has_key(point):return self.nodePoints[point]
		node = NavGraphNode(x, y, simdata=simdata)
		self.nodePoints[point] = node
		self.add_node(node,data = {sim_const.LABEL_VALID:True})
		return node

	def removeNode(self, node):
		"""
		Removes node from the graph.

		This actually 'hides' the node, which can be restored later.
		See:
		U{http://pygraphlib.sourceforge.net/doc/index.html}
		"""
		data = self.node_data(node)
		data[sim_const.LABEL_VALID] = False
		self.hide_node(node)

	def isValidNode(self, node):
		"""
		Checks if node is hidden or not.
		"""
		data = self.node_data(node)
		if not data:return True
		return data.get(sim_const.LABEL_VALID, True)

	def getValidNodes(self):
		"""
		Returns a list of valid nodes.
		"""
		validNodes = []
		for node in self.node_list():
			if self.isValidNode(node):validNodes.append(node)
		return validNodes

	def setTransitionNode(self, node, link=None):
		"""
		Sets transition link for node.
		"""
		if not link:node.transitionLink = ()
		else:node.transitionLink = link
		self.transitionNodes[(node.x, node.y)] = node

	def clearTransitionNode(self, node):
		"""
		Clears transition link for node.
		"""
		node.transitionLink = None
		try:self.transitionNodes.pop((node.x, node.y))
		except:pass
			

	#############################
	# EDGES
	#############################
	def addEdge(self, head, tail):
		"""
		Adds new edge.

		Checks that edge does not already exist, if it does
		return the existing node rather than a new one.
		"""
		point1 = (head.x, head.y, tail.x, tail.y)
		if self.edgePoints.has_key(point1):return
		point2 = (tail.x, tail.y, head.x, head.y)
		if self.edgePoints.has_key(point2):return
		self.add_edge(head, tail, create_nodes=False)
		egde = self.get_edge(head, tail)
		self.edgePoints[point1] = egde
		return egde

	def removeEdge(self, edge):
		"""
		Removes an edge from the graph.

		This actually 'hides' the edge, which can be restored later.
		See:
		U{http://pygraphlib.sourceforge.net/doc/index.html}
		"""
		data = self.edge_data(edge)
		if not data:data = {sim_const.LABEL_VALID:False}
		else:data[sim_const.LABEL_VALID] = False
		self.set_edge_data(edge, data)
		try:self.hide_edge(edge)
		except:pass # seems to be bug in graph class

	def removeEdges(self, edges):
		"""
		Removes listed edges from graph.
		"""
		for edge in edges:self.removeEdge(edge)

	def isValidEdge(self, edge):
		"""
		Checks if edge is hidden or not.
		"""
		data = self.edge_data(edge)
		if not data:return True
		return data.get(sim_const.LABEL_VALID, True)

	def getValidEdges(self):
		"""
		Returns a list of valid edges.
		"""
		validEdges = []
		for edge in self.edge_list():
			if self.isValidEdge(edge):validEdges.append(edge)
		return validEdges

	def setSimDataForEdge(self, edge, simdata):
		"""
		Attaches simulation data to edge.

		@type simdata: L{svs_simulation.simdata.simdatalib.SimData}
		"""
		data = self.edge_data(edge)
		if not data:data = {sim_const.LABEL_VALID:True}
		data[sim_const.LABEL_SIMDATA] = simdata
		self.set_edge_data(edge, data)

	def getSimDataForEdge(self, edge):
		"""
		Returns simulation data for edge.

		@rtype: L{svs_simulation.simdata.simdatalib.SimData}
		"""
		data = self.edge_data(edge)
		if not data:return None
		return data.get(sim_const.LABEL_SIMDATA, None)

	#############################
	# NAVIGATION
	#############################
	def isAccessibleArea(self, area):
		"""
		Checks if area can be entered.
		"""
		if self.density < 1.0 and len(self.occupants) == 0:return True
		else:return False

	def getNodesInArea(self, area):
		"""
		Collects nodes in specified area.
		"""
		areaNodes = []
		for point, node in self.nodePoints.items():
			if area.containsPoint(point[0], point[1]):areaNodes.append(node)
		return areaNodes

	def enterNode(self, entity):
		pass

	def exitNode(self, entity):
		pass

	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		data = {}
		nodeData = []
		for node in self.getValidNodes():nodeData.append(node.encode())
		data[sim_const.LABEL_NODES] = nodeData
		edgeData = []
		for edge in self.getValidEdges():edgeData.append(self.encodeEdge(edge))
		data[sim_const.LABEL_EDGES] = edgeData
		return data

	def encodeEdge(self, edge):
		"""
		Encodes edge data.
		"""
		data = {}
		head = self.edge_head(edge)
		if head:data[sim_const.LABEL_HEAD] = (head.x, head.y)
		tail = self.edge_tail(edge)
		if tail:data[sim_const.LABEL_TAIL] = (tail.x, tail.y)
		edgeData = self.getSimDataForEdge(edge)
		if edgeData:data[sim_const.LABEL_SIMDATA] = edgeData
		return data

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		# nodes
		if data.has_key(sim_const.LABEL_NODES):
			for entry in data[sim_const.LABEL_NODES]:
				node = self.addNode(entry[sim_const.LABEL_X], entry[sim_const.LABEL_Y])
				if entry.has_key(sim_const.LABEL_SIMDATA):
					node.simdata = SimData()
					node.simdata.decode(entry[sim_const.LABEL_SIMDATA])
		# edges
		if data.has_key(sim_const.LABEL_EDGES):
			for entry in data[sim_const.LABEL_EDGES]:
				head = self.nodePoints[entry[sim_const.LABEL_HEAD]]
				tail = self.nodePoints[entry[sim_const.LABEL_TAIL]]
				edge = self.addEdge(head, tail)
				if entry.has_key(sim_const.LABEL_SIMDATA):
					simdata = SimData()
					simdata.decode(entry[sim_const.LABEL_SIMDATA])
					self.setSimDataForEdge(edge, simdata)
				
		
