# svs_simulation.terrain.paths

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Paths across terrains.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.utilities.constants import sim_const
from svs_simulation.terrain.navgraphs import NavGraph, NavGraphNode


#############################
# CLASSES
#############################
class PathNode(NavGraphNode):
	"""
	Node in a path.
	"""
	def __init__(self, x=0, y=0, name=None, simdata=None):
		NavGraphNode.__init__(self, x, y, simdata=None)
		self.name = name
		self.structure = None

	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		data = NavGraphNode.encode(self)
		if self.name:data[sim_const.LABEL_NAME] = self.name
		if self.structure:data[sim_const.LABEL_STRUCTURE] = self.structure
		return data

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		NavGraphNode.decode(self, data)
		if data.has_key(sim_const.LABEL_NAME):self.name = data[sim_const.LABEL_NAME]
		if data.has_key(sim_const.LABEL_STRUCTURE):self.structure = data[sim_const.LABEL_STRUCTURE]



class Path(NavGraph):
	"""
	A path defines a routes between two or more locations
	in terms of a series of linked node points.
	"""
	def __init__(self, name=None):
		self.name=name
		self.nodes = []
		self.vertices = []

	def __str__(self):
		"""
		Returns representation of self as string.
		"""
		return "path [%s]" % self.name

	def reverse(self):
		"""
		Creates reverse version of path.
		"""
		path = Path(self.name)
		for i in range(len(self.nodes)):
			path.nodes.append(self.nodes[i])
			path.vertices.append(self.vertices[i])
		path.nodes.reverse()
		path.vertices.reverse()
		return path

	#############################
	# NODES
	#############################
	def addNode(self, x, y, name=None):
		"""
		Creates new node and adds it to graph.
		"""
		node = PathNode(x, y, name=name)
		self.nodes.append(node)
		self.vertices.append((x,y))
		return node

	def addNodeObject(self, nodeObject):
		"""
		Adds existing node to graph.
		"""
		self.nodes.append(nodeObject)
		self.vertices.append((nodeObject.x, nodeObject.y))

	def mapToTerrain(self, terrain):
		"""
		Checks nodes in path route against terrain,
		linking each node to the structure in which it
		is located.
		"""
		for node in self.nodes:
			node.structure = terrain.getStructureContainingPoint(node.x, node.y)

	#############################
	# NAVIGATION
	#############################
	def enterNode(self, entity):
		pass

	def exitNode(self, entity):
		pass


	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded description of self.

		@rtype:dict
		"""
		data = {}
		if self.name:data[sim_const.LABEL_NAME] = self.name
		nodeData = []
		for node in self.nodes:nodeData.append(node.encode())
		if nodeData:data[sim_const.LABEL_NODES] = nodeData
		return data

	def decode(self, data):
		"""
		Sets self to encoded values.

		@type data:dict
		"""
		if data.has_key(sim_const.LABEL_NAME):self.name = data[sim_const.LABEL_NAME]
		nodeDataList = data.get(sim_const.LABEL_NODES, None)
		if nodeDataList:
			for nodeData in nodeDataList:
				nodeObject = PathNode()
				nodeObject.decode(nodeData)
				self.addNodeObject(nodeObject)
		


class PathFollower:
	"""
	Used by entity to follow path.
	"""
	def __init__(self, entity, path, loops=1):
		self.entity = entity
		self.path = path
		self.totalLoops = loops
		self.loopCount = 0

	def beginPath(self):
		"""
		Called when an entity starts to follow the path.
		"""
		print "beginPath"
		self.traversedNodes = []
		self.nextNode = self.path.nodes[0]
		self.currentNode = None
		self.totalNodes = len(self.path.nodes)
		self.nodeIndex = 0
		self.entity.moveToNextPathNode(self.nextNode)


	def endPath(self):
		"""
		Called when end of path is reached.
		"""
		if not self.totalLoops:self.beginPath()
		self.loopCount += 1
		if self.loopCount < self.totalLoops:self.beginPath()
		else:self.entity.endPath()

	def updateRoute(self):
		"""
		Updates current position of entity on route.
		"""
		if self.currentNode:
			if not self.entity.isOverPoint(self.currentNode.x, self.currentNode.y):
				self.exitNode(self.currentNode)
				self.currentNode = None
		if self.entity.isOverPoint(self.nextNode.x, self.nextNode.y):
			self.enterNode(self.nextNode)
			if self.getNextNode():self.entity.moveToNextPathNode(self.nextNode)
			else:self.endPath()

	def getNextNode(self):
		"""
		Sets next node on route.
		"""
		self.traversedNodes.append(self.nextNode)
		self.nodeIndex += 1
		if self.nodeIndex < self.totalNodes:self.nextNode = self.path.nodes[self.nodeIndex]
		else:self.nextNode = None
		return self.nextNode
		
	def enterNode(self, node):
		"""
		Called when the entity enters a new node.
		"""
		if self.currentNode:self.exitNode(self.currentNode)
		self.currentNode = node
		self.entity.enterPathNode(node)

	def exitNode(self, node):
		"""
		Called when the entity leaves a node.
		"""
		self.entity.exitPathNode(node)


class PathBuilder:
	"""
	Creates a path between two specified locations.

	NOTE: currently not implemented.
	"""
	def __init__(self):
		pass


class PathStore:
	"""
	Maintains a list of paths, which can be accessed via start and destination.
	"""
	def __init__(self):
		self.pathList = {}
		self.namedPaths = {}
		self.allpaths = []

	def addPath(self, path, listByName=True):
		"""
		Adds path to list.  Each path is listed by the names
		of its first and last nodes.

		If C{listByName} is C{True} adds to named path list as well.
		If C{listByName} is C{False}, then it can only be searched by
		its start and destination nodes.
		"""
		start = path.nodes[0].name
		destination = path.nodes[-1].name
		if not self.pathList.has_key(start):self.pathList[start] = {}
		self.pathList[start][destination] = path
		if not path in self.allpaths:self.allpaths.append(path)
		if listByName:self.namedPaths[path.name] = path

	def addNamedPath(self, path, listByNodes=True):
		"""
		Adds path to list by name.

		If C{listByNodes} is C{True} adds to main path list as well.
		This means it can also be searched by the names of its first
		and last nodes.  If C{listByNodes} is C{False}, then it can
		only be saerched by its name.
		"""
		if listByNodes:self.addPath(path)
		else:
			self.namedPaths[path.name] = path
			if not path in self.allpaths:self.allpaths.append(path)

	def getPath(self, name=None, start=None, destination=None):
		"""
		Returns path.  If C{name} given tries that first, then, if
		C{start} and C{destination} given tries these.  Alternatively,
		can just be called with C{start} and C{destination}.
		"""
		if name:
			path = self.getPathNamed(name)
			if path:return path
		if start and destination:return self.getPathWithNodes(start, destination)
		return None
		

	def getPathWithNodes(self, start, destination):
		"""
		Returns path between C{start} and C{destination}.

		If no path exists, returns C{None}.
		"""
		startList = self.pathList.get(start, None)
		if not startList:return None
		return startList.get(destination, None)

	def getPathNamed(self, pathName):
		"""
		Returns path with specified name.

		If no path exists, returns C{None}.
		"""
		return self.namedPaths.get(pathName, None)

	def getAllPaths(self):
		"""
		Returns a list of all paths in store.
		"""
		return self.allpaths


#############################
# FUNCTIONS
#############################
def createPath(pathName, vertices):
	"""
	Factory method to create a path with given name and vertices.
	"""
	path = Path(pathName)
	for vert in vertices:path.addNode(vert[0], vert[1])
	return path

		
