# svs_simulation.terrain.structures

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Classes for terrain structures.

A structure can be a building, path, river, etc. Any entity
within the terrain which can potentially be entered by an agent.

Support for polygonal forms in a structure are provided by the 
C{Polygon} library:
U{http://www.dezentral.de/soft/Polygon/}


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Polygon import Polygon
import math

# internal imports
from svs_simulation.numdata.geomlib import geom_const, AxialBounds2D
from svs_simulation.utilities.constants import sim_const
from svs_simulation.entities.base_entities import BasicSpatialEntity
from svs_simulation.terrain.util_classes import GenericTerrainObject
from svs_simulation.terrain.navgraphs import NavGraph
from svs_core.utilities.objectmanager import PersistentObject


#############################
# FUNCTIONS
#############################
def createStructureFromSVSFile(filename):
	"""
	Reads in data file for structure and creates it.
	
	@rtype: L{Structure}
	"""
	data = PersistentObject()
	data.read(filename)
	structure = Structure()
	structure.decode(data.structure)
	return structure

def createStructureFromSVGFile(filename):
	"""
	Converts data in SVG file to Structure object.

	@rtype: L{Structure}
	"""
	from svs_simulation.terrain.svgloaders import createStructureSVGLoader
	structure = Structure()
	loader = createStructureSVGLoader(structure)
	loader.read(filename)
	return structure

def saveStructureToFile(structure, filename):
	"""
	Writes structure data to file.
	"""
	from svs_core.utilities.lib import validateFilePath
	filename = validateFilePath(filename)
	data = structure.encode()
	dataStr = "structure = %s" % data
	f = open(filename, 'w')
	f.write(dataStr)
	f.close()


def createOutlineplane(vertices):
	"""
	Factory for constructing floorplane object.
	"""
	outlineplane = OutlinePlane()
	outlineplane.setup()
	outlineplane.setOutline(vertices)
	return outlineplane


def createFloorplane(vertices, movementcost=0.0):
	"""
	Factory for constructing floorplane object.
	"""
	floorplane = FloorPlane()
	floorplane.setup(movementcost)
	floorplane.setOutline(vertices)
	return floorplane

def createWallplane(vertices, density=1.0, height=1.0):
	"""
	Factory for constructing wallplane object.
	"""
	wallplane = WallPlane()
	wallplane.setup(density,height)
	wallplane.setOutline(vertices)
	return wallplane

def createInnerMargin(floorplane, margin):
	"""
	Creates inner margin within floorplane.

	This is used for simplifying collision by agents.
	"""
	verts = floorplane.getVertices()
	innerVerts = []
	innerLines = []
	for idx in range(len(verts) - 1):
		innerLines.append(_makeInnerVertices(floorplane, verts[idx], verts[idx + 1], margin))
	innerVerts.append(innerLines[0][0])
	for idx in range(len(innerLines) - 1):
		if not _linesAreParallel(innerLines[idx], innerLines[idx+1]):
			innerVerts.append(_intersectionPoint(innerLines[idx], innerLines[idx+1]))
		else:print "_____parallel lines______"
	innerVerts.append(innerLines[len(innerLines) - 1][1])
	return innerVerts

def _makeInnerVertices(floorplane, vert1, vert2, margin):
	theta = None
	if (vert2[0] - vert1[0]) != 0:
		theta = math.atan((vert2[1] - vert1[1]) / (vert2[0] - vert1[0] * 1.0))
		deltaTheta = math.radians(math.degrees(theta) + 90)
	else: deltaTheta = math.radians(180)
	deltaX = math.cos(deltaTheta) * margin
	deltaY = math.sin(deltaTheta) * margin
	testPoint = _getTestPoint(vert1, vert2)
	testPointX = testPoint[0] + deltaX
	testPointY = testPoint[1] + deltaY
	if floorplane.containsPoint(testPointX, testPointY):
		return ((vert1[0] + deltaX, vert1[1] + deltaY), (vert2[0] + deltaX, vert2[1] + deltaY))
	else:
		if theta != None:deltaTheta = math.radians(math.degrees(theta) - 90)
		else:deltaTheta = math.radians(0)
		deltaX = math.cos(deltaTheta) * margin
		deltaY = math.sin(deltaTheta) * margin
		return ((vert1[0] + deltaX, vert1[1] + deltaY), (vert2[0] + deltaX, vert2[1] + deltaY))


def _getTestPoint(vert1, vert2):
	if vert2[0] == vert1[0]: # vertical
		x = vert1[0]
		y = vert1[1] + ((vert2[1] - vert1[1]) / 2.0)
		return (x, y)
	else:
		m1 = (vert2[1] - vert1[1]) / (vert2[0] - vert1[0])
		c1 = vert1[1] - (m1 * vert1[0])
		x = vert1[0] + ((vert2[0] - vert1[0]) / 2.0)
		y = c1 + (m1 * x)
		return (x, y)


def _linesAreParallel(line1, line2):
	if (line1[1][0] - line1[0][0]) == 0.0 and (line2[1][0] - line2[0][0]) == 0.0:return True
	if (line1[1][1] - line1[0][1]) == 0.0 and (line2[1][1] - line2[0][1]) == 0.0:return True
	return False
	


def _intersectionPoint(line1, line2):
	# deal with special cases
	if (line1[1][1] - line1[0][1]) == 0.0 and (line2[1][0] - line2[0][0]) == 0.0:
		# line1 is horizontal, line2 is vertical
		return (line2[0][0], line1[0][1])
	elif (line1[1][0] - line1[0][0]) == 0.0 and (line2[1][1] - line2[0][1]) == 0.0:
		# line1 is vertical, line2 is horizontal
		return (line1[0][0], line2[0][1])
	elif (line1[1][0] - line1[0][0]) == 0.0 and (line2[1][1] - line2[0][1]) != 0.0:
		# line1 is vertical, line2 is sloping
		m2 = (line2[1][1] - line2[0][1]) / (line2[1][0] - line2[0][0] * 1.0)
		c2 = line2[0][1] - (m2 * line2[0][0])
		xi = line1[0][0] 
		yi = c2 + (m2 * xi)
		return (xi, yi)
	elif (line1[1][1] - line1[0][1]) != 0.0 and (line2[1][0] - line2[0][0]) == 0.0:
		# line1 is sloping, line2 is vertical
		try:m1 = (line1[1][1] - line1[0][1]) / (line1[1][0] - line1[0][0] * 1.0) # bug
		except ZeroDivisionError:
			print ZeroDivisionError
			print "line1[1][0] ", line1[1][0]
			print "line1[0][0] ", line1[0][0]
			raise ZeroDivisionError
		c1 = line1[0][1] - (m1 * line1[0][0])
		xi = line2[0][0]
		yi = c1 + (m1 * xi)
		return (xi, yi)
	elif (line1[1][0] - line1[0][0]) != 0.0 and (line2[1][0] - line2[0][0]) != 0.0:
		# two sloping lines
		m1 = (line1[1][1] - line1[0][1]) / (line1[1][0] - line1[0][0] * 1.0)
		m2 = (line2[1][1] - line2[0][1]) / (line2[1][0] - line2[0][0] * 1.0)
		c1 = line1[0][1] - (m1 * line1[0][0])
		c2 = line2[0][1] - (m2 * line2[0][0])
		if (m1 - m2) == 0.0:xi = 0.0
		else:xi = -(c1 - c2) / (m1 - m2) 
		yi = c1 + (m1 * xi)
		return (xi, yi)
		


#############################
# CLASSES
#############################
class Structure(GenericTerrainObject, BasicSpatialEntity):
	"""
	Self contained spatial structure within a terrain,
	such as a building, etc.
	"""
	def __init__(self):
		GenericTerrainObject.__init__(self) # this must be constructed first
		BasicSpatialEntity.__init__(self)
		self.outlineplane = None
		self.floorplane = None
		self.wallplanes = []
		self.externalDensity = 0.0
		self.internalDensity = 0.0
		self.navGraph = None
		self.seedX = None
		self.seedY = None
		self.name = None
		self.filepath = None
		self.bounds.reset()
		self.extraProperties = {}

	def setup(self, _uid=None, name=None, filepath=None):
		BasicSpatialEntity.setup(self)
		self.name = name
		self.filepath = filepath

	#############################
	# BASIC INFO
	#############################
	def getName(self):
		"""
		Returns name for structure.  If the structure
		has no name retuens its _uid.
		"""
		if not self.name:return self._uid
		return self.name

	def getBoundsMinX(self):
		"""
		Returns minimum x coordinate from bounds.
		"""
		return self.bounds.minX

	def getBoundsMinY(self):
		"""
		Returns minimum y coordinate from bounds.
		"""
		return self.bounds.minY

	def getBoundsMaxX(self):
		"""
		Returns maximum x coordinate from bounds.
		"""
		return self.bounds.maxX

	def getBoundsMaxY(self):
		"""
		Returns maximum y coordinate from bounds.
		"""
		return self.bounds.maxY

	def containsPoint(self, x, y):
		"""
		Tests if coordinates are contained within structure bounds.

		If a floorplane is defines for the structure uses that, otherwise uses
		world bounds.
		"""
		if self.floorplane:return self.floorplane.containsPoint(x, y)
		return self.bounds.containsPointInWorld(x, y)

	def __str__(self):
		"""
		Returns string representation of structure.
		"""
		return "structure [%s]" % self.name

	def getWorldbounds(self):
		"""
		Returns the bounds of the structure as mapped to the world coordinates.
		"""
		return self.bounds.worldbounds

	def setFloorplane(self, floorplane):
		"""
		Sets floorplane and adjust bounds to contain it.
		"""
		self.floorplane = floorplane
		for vertice in floorplane.getVertices():
			self.bounds.worldbounds.addPoint(vertice[0], vertice[1])
		self.bounds.worldbounds.x = self.bounds.worldbounds.maxX
		self.bounds.worldbounds.y = self.bounds.worldbounds.maxY
		self.bounds.maxX = self.bounds.worldbounds.maxX - self.bounds.worldbounds.minX
		self.bounds.maxY = self.bounds.worldbounds.maxY - self.bounds.worldbounds.minY

	def createInnerBoundary(self, margin=None):
		"""
		Creates inner boundary on floorplane at C{margin} distance from edges.
		"""
		print "createInnerBoundary:", self.name
		if margin != None:self.margin = margin
		self.floorplane.createInnerBoundary(self.margin)

	def addWallplane(self, wallplane):
		"""
		Adds wallplane and adjust bounds to contain it.
		"""
		if wallplane in self.wallplanes:return
		self.wallplanes.append(wallplane)
		for vertice in wallplane.getVertices():
			self.bounds.worldbounds.addPoint(vertice[0], vertice[1])
		self.bounds.worldbounds.x = self.bounds.worldbounds.maxX
		self.bounds.worldbounds.y = self.bounds.worldbounds.maxY
		self.bounds.maxX = self.bounds.worldbounds.maxX - self.bounds.worldbounds.minX
		self.bounds.maxY = self.bounds.worldbounds.maxY - self.bounds.worldbounds.minY
		self.externalDensity = max(self.externalDensity, wallplane.density)


	#############################
	# ARCHIVING
	############################
	def encode(self):
		"""
		Returns encoded representation of self.

		@rtype: dict
		"""
		data = BasicSpatialEntity.encode(self)
		if self.name:data[sim_const.LABEL_NAME] = self.name
		if self.margin:data[sim_const.LABEL_MARGIN] = self.margin
		if self.floorplane:data[sim_const.LABEL_FLOORPLANE] = self.floorplane.encode()
		if self.wallplanes:data[sim_const.LABEL_WALLPLANES] = self.encodeWallplanes()
		if self.navGraph:data[sim_const.LABEL_NAVGRAPH] = self.navGraph.encode()
		if self.extraProperties:data[sim_const.LABEL_EXTRAPROPERTIES] = self.extraProperties
		return data

	def encodeWallplanes(self):
		"""
		Returns and encoded representation of wallplanes.
		"""
		wpdata = []
		for wallplane in self.wallplanes:
			wpdata.append(wallplane.encode())
		return wpdata

	def decode(self, data, margin=None):
		"""
		Decodes data and applies it to self.
		"""
		BasicSpatialEntity.decode(self, data)
		if data.has_key(sim_const.LABEL_NAME):
			self.name = data[sim_const.LABEL_NAME]
		if data.has_key(sim_const.LABEL_FLOORPLANE):
			floorplane = FloorPlane()
			floorplane.decode(data[sim_const.LABEL_FLOORPLANE])
			self.setFloorplane(floorplane)
		if data.has_key(sim_const.LABEL_WALLPLANES):
			self.decodeWallplanes(data[sim_const.LABEL_WALLPLANES])
		if data.has_key(sim_const.LABEL_NAVGRAPH):
			navGraph = NavGraph()
			navGraph.decode(data[sim_const.LABEL_NAVGRAPH])
			self.navGraph = navGraph
			print "self.navGraph:", self.navGraph
		if data.has_key(sim_const.LABEL_MARGIN):
			self.margin = data[sim_const.LABEL_MARGIN]
			#if self.floorplane:self.floorplane.createInnerBoundary(self.margin)
		if data.has_key(sim_const.LABEL_EXTRAPROPERTIES):
			self.extraProperties = data[sim_const.LABEL_EXTRAPROPERTIES]
		elif margin != None:
			self.margin = margin
			#print "createInnerBoundary:", self.name
			#if self.floorplane:self.floorplane.createInnerBoundary(self.margin) ### !! buggy !!

	def decodeWallplanes(self, data):
		"""
		Decodes representation of wallplanes.
		"""
		for entry in data:
			wallplane = WallPlane()
			wallplane.decode(entry)
			self.addWallplane(wallplane)
			


	############################
	# PARTITIONS
	############################
	def enterPartition(self, partition):
		"""
		Called when structure is added to a partition.
		"""
		self.partition = partition

	def exitPartition(self, partition):
		"""
		Called when structure is removed from a partition.
		"""
		self.partition = None
		

############################
# PLANES
############################
class Plane:
	"""
	Generic component part of a structure.

	Planes may be polygonal in form. 
	"""
	def __init__(self):
		pass

	def setup(self):
		self.outline = Polygon()
		self.bounds = None

	def setOutline(self, vertices):
		"""
		Defines outline of plane.

		@type vertices: array 
		"""
		self.outline.addContour(vertices)
		verts = self.outline.boundingBox(0)
		self.bounds = AxialBounds2D(True)
		self.bounds.minX = verts[0]
		self.bounds.minY = verts[2]
		self.bounds.maxX = verts[1]
		self.bounds.maxY = verts[3]

	def getVertices(self):
		"""
		Returns a list of vertices for the outline of the plane.
		"""
		return self.outline.contour(0)


	def containsPoint(self, x, y):
		"""
		Tests if specified coordinates are present in plane.
		"""
		return self.outline.isInside(x, y)
		
	def withinBounds(self, x, y):
		"""
		Tests if specified coordinates are within the bounds of the plane.
		"""
		return self.bounds.containsPoint(x, y)

	def encode(self):
		"""
		Returns encoded representation of self.
		"""
		return {sim_const.LABEL_VERTICES: self.getVertices()}

	def decode(self, data):
		"""
		Applies encoded representation to self.
		"""
		self.setOutline(data[sim_const.LABEL_VERTICES])


class OutlinePlane(Plane):
	"""
	Outerboundary of structure.
	"""
	def __init__(self):
		Plane.__init__(self)

class FloorPlane(Plane):
	"""
	Horizontal structural component that agents may move over.
	"""
	def __init__(self):
		Plane.__init__(self)
		self.innerspace = None

	def setup(self, movementcost=0.0):
		Plane.setup(self)
		self.movementcost = movementcost

	def setOutline(self, vertices):
		"""
		Defines outline of plane.

		If C{margin} is defined creates an inner zone
		shrunk to accommodate collision margin.

		@type vertices: array 
		"""
		Plane.setOutline(self, vertices)

	def createInnerBoundary(self, margin):
		"""
		Creates inner boundary for collisions.
		"""
		self.innerspace = Polygon()
		self.innerspace.addContour(createInnerMargin(self, margin))

	def containsNavPoint(self, x, y):
		"""
		Tests if specified coordinates are present in plane.
		"""
		#return self.innerspace.isInside(x, y)
		return self.containsPoint(x, y)

	def getInnerVertices(self):
		"""
		Returns a list of vertices for the inner space of the plane.
		"""
		if not self.innerspace:return None
		try:
			verts = self.innerspace.contour(0)
		except:verts = None
		return verts

	def encode(self):
		"""
		Returns encoded representation of self.
		"""
		return {sim_const.LABEL_VERTICES: self.getVertices(), sim_const.LABEL_MOVEMENTCOST:self.movementcost}

	def decode(self, data):
		"""
		Applies encoded representation to self.
		"""
		self.setup(data[sim_const.LABEL_MOVEMENTCOST])
		self.setOutline(data[sim_const.LABEL_VERTICES])


class WallPlane(Plane):
	"""
	Vertical structural component that defines edge.
	"""
	def __init__(self):
		Plane.__init__(self)

	def setup(self, density=1.0, height=1.0):
		Plane.setup(self)
		self.density = density
		self.height = height

	def encode(self):
		"""
		Returns encoded representation of self.
		"""
		return {sim_const.LABEL_VERTICES:self.getVertices(), 
			sim_const.LABEL_DENSITY:self.density,
			sim_const.LABEL_HEIGHT:self.height}

	def decode(self, data):
		"""
		Applies encoded representation to self.
		"""
		self.setup(data[sim_const.LABEL_DENSITY], data[sim_const.LABEL_HEIGHT])
		self.setOutline(data[sim_const.LABEL_VERTICES])


