# svs_simulation.terrain.svgloaders

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Loads and converts SVG data for terrains, structures, etc.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# internal imports
from svs_core.filehandling.loaders import DocumentLoadingException
from svs_simulation.utilities.constants import sim_const, sim_xml
from svs_simulation.utilities.svgutils import SVGLoader, SVGContentHandler
from svs_simulation.terrain.structures import Structure, createFloorplane, createWallplane, createOutlineplane
from svs_simulation.terrain.base_classes import Terrain, Partition
from svs_simulation.terrain.paths import createPath
from svs_simulation.terrain.linkpaths import createLinkPath


#############################
# LOADER FACTORIES
#############################
def createStructureSVGLoader(target):
	"""
	Creates loader and parser for converting SVG data into a Structure object..
	""" 
	return SVGLoader(target, StructureSVGContentHandler)

def createTerrainSVGLoader(target):
	"""
	Creates loader and parser for converting SVG data into a Structure object..
	""" 
	return SVGLoader(target, TerrainSVGContentHandler)	

#############################
# CONTENT HANDLERS
#############################
class StructureSVGContentHandler(SVGContentHandler):
	"""
	Handler for parsing structure SVG document.
	"""
	
	def __init__(self, target):
		SVGContentHandler.__init__(self, target)

	def createSimComponent(self, vertices, svs_data): #!!! needs updated, see terrain below
		"""
		Creates structural component from data.
		"""
		if svs_data[sim_const.LABEL_COMPONENT] == sim_const.LABEL_FLOORPLANE:
			self.target.setFloorplane(createFloorplane(vertices, svs_data.get(sim_const.LABEL_DENSITY, 0.0)))
		elif svs_data[sim_const.LABEL_COMPONENT] == sim_const.LABEL_WALLPLANE:
			self.target.addWallplane(createWallplane(vertices, svs_data.get(sim_const.LABEL_DENSITY, 0.0), svs_data.get(sim_const.LABEL_HEIGHT, 0.0)))


class TerrainSVGContentHandler(SVGContentHandler):
	"""
	Handler for parsing terrain SVG document.
	"""
	
	def __init__(self, target):
		SVGContentHandler.__init__(self, target)
		self.structuresToLoad = {}

	def createSimComponent(self, vertices, svs_data):
		"""
		Creates structural component from data.
		"""
		if svs_data == None or len(svs_data) == 0:raise DocumentLoadingException("Empty or malformed svs_data attribute.")
		component = svs_data.get(sim_const.LABEL_COMPONENT, None)
		if not component:return
		if component == sim_const.LABEL_STRUCTURE:
			structure = Structure()
			structure.setup(name=svs_data.get(sim_const.LABEL_NAME, None), 
					filepath=svs_data.get(sim_const.LABEL_FILE, None), 
					externalDensity=svs_data.get(sim_const.LABEL_EXTDENSITY, None))
			self.target.addStructure(structure, svs_data.get(sim_const.LABEL_PARTITION, None))
		elif component == sim_const.LABEL_WALLPLANE:
			#print "wallplane"
			structure = self.target.createStructureFromName( 
					svs_data.get(sim_const.LABEL_NAME, None), 
					svs_data.get(sim_const.LABEL_PARTITION, None))
			for verts in vertices:
				structure.addWallplane(createWallplane(verts))
		elif component == sim_const.LABEL_FLOORPLANE:
			#print "floorplane"
			structure = self.target.createStructureFromName( 
					svs_data.get(sim_const.LABEL_NAME, None), 
					svs_data.get(sim_const.LABEL_PARTITION, None))
			structure.setFloorplane(createFloorplane(vertices[0]))
		elif component == sim_const.LABEL_OBJECT:
			print "sim object"
			self.target.createSimObjectFromName( 
					svs_data.get(sim_const.LABEL_NAME, None),
					svs_data.get(sim_const.LABEL_CLASS, None),
					svs_data.get(sim_const.LABEL_STRUCTURE, None), 
					vertices[0])
		# paths
		elif component == sim_const.LABEL_PATH:
			self.target.addPath(createPath(svs_data.get(sim_const.LABEL_NAME, None), vertices[0]))
		# linkpaths
		elif component == sim_const.LABEL_LINKPATH:
			self.target.addLinkPath(createLinkPath(svs_data.get(sim_const.LABEL_TRANSITION, None), vertices[0]))
		else:pass

