# svs_simulation.world.proxyworlds

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Proxy for client-side representation of simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# internal imports
from svs_simulation.utilities.constants import sim_const
from svs_simulation.simtime.timelib import SimTime
from svs_simulation.entities.customentities import CustomEntityRepository
from svs_simulation.agents.basic_agents import SimAgentProxy
from svs_simulation.simobjects.basic_objects import SimObjectProxy


class SimWorldProxy:
	"""
	Provides a lightweight client-side representation of a simulation world.
	"""
	def __init__(self):
		self.name = None
		self.terrain = None
		self.agentGroups = CustomEntityRepository()
		self.objectGroups = CustomEntityRepository()
		self.entities = {}
		self.reset()

	def reset(self):
		"""
		Resets playback time to start.
		"""
		self.running = False
		self.currentTime = None

	#############################
	# PROCESS FUNCTIONS
	#############################
	def startWorld(self, simTime):
		"""
		Starts playback.  If time step is specified uses this
		instead of default value (L{time_const.DEFAULT_TIMESTEP}).
		"""
		if not self.terrain:raise SimWorldException("Unable to start world, no terrain")
		self.running = True
		self.currentTime = simTime
		for process in self.processHandlers:process.startWorld(simTime)

	def stopWorld(self, simTime):
		"""
		Stops playback.
		"""
		self.currentTime = simTime
		self.running = False
		for process in self.processHandlers:process.stopWorld(simTime)

	def updateWorld(self, simTime):
		"""
		Updates world.
		"""
		self.currentTime = simTime
		for process in self.processHandlers:process.updateWorld(simTime)

	#############################
	# PROCESS HANDLERS
	#############################
	def handleUpdateWorld(self, data):
		"""
		Handles update data from world.
		"""
		#print data
		for entry in data:
			entity = self.entities.get(entry[sim_const.LABEL_SOURCE], None)
			if entity:entity.decodeChangeData(entry[sim_const.LABEL_DATA])


	#############################
	# SETUP FUNCTIONS
	#############################
	def setupWorld(self, worldData):
		"""
		Sets up basic world properties, such as name and clock.

		NOTE: clock data to be added later.
		"""
		self.name=worldData[sim_const.LABEL_NAME]
		# add clock later

	def setupTerrain(self, model, lod):
		"""
		Creates terrain from received data.
		"""
		from svs_simulation.terrain.terrainproxies import TerrainProxy
		self.terrain = TerrainProxy()
		self.terrain.decode(model, lod)

	def setupPartition(self, model):
		"""
		Creates partition from received data.
		"""
		return self.terrain.decodePartition(model)

	def setupAgentGroups(self, groups):
		"""
		Handles received list of agent groups.
		"""
		for group in groups:self.agentGroups.createGroup(group)

	def setupAgentGroup(self, groupData):
		"""
		Handles received data for agent group.
		"""
		group = self.getAgentGroup(groupData[sim_const.LABEL_NAME])
		membersData = groupData[sim_const.LABEL_MEMBERS]
		for itemData in membersData:
			agent = SimAgentProxy()
			agent.decode(itemData)
			agent.sourceClass = group.name
			group.addMember(agent)
			self.entities[agent._uid] = agent
		#print "setupAgentGroup:", group.members
		

	def setupObjectGroups(self, groups):
		"""
		Handles received list of object groups.
		"""
		for group in groups:self.objectGroups.createGroup(group)

	def setupObjectGroup(self, groupData):
		"""
		Handles received data for object group.
		"""
		group = self.getObjectGroup(groupData[sim_const.LABEL_NAME])
		membersData = groupData[sim_const.LABEL_MEMBERS]
		for itemData in membersData:
			simObj = SimObjectProxy()
			simObj.decode(itemData)
			simObj.sourceClass = group.name
			group.addMember(simObj)
			self.entities[simObj._uid] = simObj
		#print "setupObjectGroup:", group.members


	#############################
	# INFO FUNCTIONS
	#############################
	def getCurrentTime(self):
		"""
		Returns current simulation time in world.
		"""
		return self.currentTime

	def getPartitionDisplayInfo(self):
		"""
		Resturns a string listing basic info about partitions.
		"""
		if not self.terrain:return "No terrain available."
		partitionStr = "Terrain partitions:\n"
		for partitionName, data in self.terrain.partitions.items():
			if data:partitionStr += "  %s - loaded" % partitionName
			else:partitionStr += "  %s - not loaded" % partitionName
		return partitionStr


	#############################
	# AGENTS
	#############################
	def getAgentGroups(self):
		"""
		Returns list of agent groups.
		"""
		return self.agentGroups.groups

	def getAgentGroup(self, groupName):
		"""
		Returns specified agent group.
		"""
		return self.agentGroups.getGroup(groupName)

	#############################
	# SIMOBJECTS
	#############################
	def getObjectGroups(self):
		"""
		Returns list of object groups.
		"""
		return self.objectGroups.groups

	def getObjectGroup(self, groupName):
		"""
		Returns specified object group.
		"""
		return self.objectGroups.getGroup(groupName)
