# svs_simulation.world.worlds

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Root class for simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from time import time
import os

# internal imports
from svs_core.utilities.objectmanager import PersistentObject
from svs_core.utilities.lib import validateFilePath
from svs_simulation.events.eventlib import SimEventCentre
from svs_simulation.utilities.constants import sim_const
from svs_simulation.simtime.timelib import SimTime, SimClock
from svs_simulation.simtime.schedules import Schedule, ScheduleManager, createScheduleFromSVSFile
from svs_simulation.entities.customentities import CustomEntityRepository


#############################
# EXCEPTIONS
#############################
class SimWorldException(Exception):pass

#############################
# FUNCTIONS
#############################
def createWorldFromSVSFile(filename, worldClass=None, incSchedules=True, incTerrain=True, incAgents=True, incObjects=True, incBehaviours=True, incProcessHandlers=True):
	"""
	Reads in data file for world and creates it.

	World data is stored in an hierachical file structure:

	- root
		- world
			- world_data.svs
			- schedules
		- terrain
			- terrain data
			- partitions
			- paths
		- agents
			for each agent type:
				- classes
				- instances
				- schedules
		- objects
			for each object type:
				- classes
				- instances
				- schedules
		- behaviours
			for each behaviour type:
				- classes
				- instances
				- schedules
	
	@rtype: L{SimWorld}
	"""
	from svs_core.utilities.objectmanager import PersistentObject
	from svs_simulation.terrain.base_classes import createTerrainFromSVSFile
	filename = validateFilePath(filename)
	# read world
	data = PersistentObject()
	data.read(filename + '/world/world_data.svs')
	if not worldClass: worldClass = SimWorld
	world = worldClass()
	world.decode(data.world)
	# read schedules
	if incSchedules:
		scheduleDir = filename + '/world/schedules/'
		if validateFilePath(scheduleDir, create=False):
			__createSchedulesForWorld(scheduleDir, world)
	# read terrain
	if incTerrain:
		terrainDir = filename + '/terrain/terrain_data.svs'
		if validateFilePath(terrainDir, create=False):
			terrain = createTerrainFromSVSFile(terrainDir)
			#except:raise SimWorldException("Unable to create terrain.")
			world.setTerrain(terrain)
	# read agents
	if incAgents:
		agentRootDir = filename + '/agents/'
		if validateFilePath(agentRootDir, create=False):
			agentDirs = os.listdir(agentRootDir)
			for agentDir in agentDirs:
				if agentDir == 'CVS':continue
				__createAgentsForWorld(agentRootDir + agentDir, world)
				#except:raise SimWorldException("Unable to create agent.")
	# read objects
	if incObjects:
		objectRootDir = filename + '/objects/'
		if validateFilePath(objectRootDir, create=False):
			objectDirs = os.listdir(objectRootDir)
			for objectDir in objectDirs:
				if objectDir == 'CVS':continue
				__createSimObjectsForWorld(objectRootDir + objectDir, world)
				#except:raise SimWorldException("Unable to create simulation object.")
	# read behaviours
	if incBehaviours:
		behaviourRootDir = filename + '/behaviours/'
		if validateFilePath(behaviourRootDir, create=False):
			behaviourDirs = os.listdir(behaviourRootDir)
			for behaviourDir in behaviourDirs:
				if behaviourDir == 'CVS':continue
				try:__createBehavioursForWorld(behaviourRootDir + behaviourDir, world)
				except:raise SimWorldException("Unable to create behaviour.")
	# read process handlers
	if incProcessHandlers:
		procHandlerRootDir = filename + '/processes/'
		if validateFilePath(procHandlerRootDir, create=False):
			processDirs = os.listdir(procHandlerRootDir)
			if not processDirs:__createDeafultProcessHandlerForWorld(world)
			for processDir in processDirs:
				if processDir == 'CVS':continue
				__createProcessHandlersForWorld(procHandlerRootDir + processDir, world)
				#except:raise SimWorldException("Unable to create process handler.")
		else: __createDeafultProcessHandlerForWorld(world) # use default process handler
	return world


def __createSchedulesForWorld(scheduleDir, world):
	"""
	Creates schedules from data in directory.
	"""
	instanceFiles =  os.listdir(scheduleDir)
	for instanceFile in instanceFiles:
		if not instanceFile.lower().endswith('.svs'):continue
		schedule = createScheduleFromSVSFile(scheduleDir + instanceFile)
		world.scheduleManager.addSchedule(schedule)
	#### HACK FOR TEST #####
	world.scheduleManager.currentSchedule = world.scheduleManager.schedules['monday']

def __createAgentsForWorld(agentDir, world):
	"""
	Creates agents from data in structured directory 
	and applies them to the world.
	"""
	agentData = PersistentObject()
	agentData.read(agentDir + '/class_data.svs')
	# create class
	#print agentData.__dict__
	agentGroupName = agentData.agent_class_name
	codeFilename = agentDir + '/' + agentData.agent_class_file
	agentGroup = world.agentGroups.createGroupFromSource(agentGroupName, codeFilename)
	# create instances
	instanceDir = agentDir + '/instances/'
	instanceFiles =  os.listdir(instanceDir)
	for instanceFile in instanceFiles:
		if not instanceFile.lower().endswith('.svs'):continue
		agent = agentGroup.createInstanceFromFile(instanceDir + instanceFile)
		agent.enterWorld(world)


def __createSimObjectsForWorld(objectDir, world):
	"""
	Creates objects from data in structured directory 
	and applies them to the world.
	"""
	objectData = PersistentObject()
	objectData.read(objectDir + '/class_data.svs')
	# create class
	objectGroupName = objectData.object_class_name
	codeFilename = objectDir + '/' + objectData.object_class_file
	objectGroup = world.objectGroups.createGroupFromSource(objectGroupName, codeFilename)
	# create instances
	instanceDir = objectDir + '/instances/'
	instanceFiles =  os.listdir(instanceDir)
	for instanceFile in instanceFiles:
		if not instanceFile.lower().endswith('.svs'):continue
		simObject = objectGroup.createInstanceFromFile(instanceDir + instanceFile)
		simObject.enterWorld(world)



def __createBehavioursForWorld(behaviourDir, world):
	"""
	Creates behaviours from data in structured directory 
	and applies them to the world.
	"""
	from svs_simulation.behaviour.behaviourlib import createBehaviourFromSVSFile
	behaviourData = PersistentObject()
	behaviourData.read(behaviourDir + '/class_data.svs')
	behaviourClass = behaviourData.behaviour_class
	instanceFiles = os.listdir(behaviourDir + '/instances')
	for instanceFile in instanceFiles:
		if not instanceFile.lower().endswith('.svs'):continue
		behaviour = createBehaviourFromSVSFile(instanceFile, behaviourClass=behaviourClass)
		world.addBehaviour(behaviour)


def __createProcessHandlersForWorld(processDir, world):
	"""
	Creates process handlers from data in structured directory 
	and applies them to the world.
	"""
	processData = PersistentObject()
	processData.read(processDir + '/class_data.svs')
	# create class
	processGroupName = processData.process_class_name
	codeFilename = processDir + '/' + processData.process_class_file
	processGroup = world.processGroups.createGroupFromSource(processGroupName, codeFilename)
	# create instances
	instanceDir = processDir + '/instances/'
	instanceFiles =  os.listdir(instanceDir)
	for instanceFile in instanceFiles:
		if not instanceFile.lower().endswith('.svs'):continue
		processHandler = processGroup.createInstanceFromFile(instanceDir + instanceFile)
		world.addProcessHandler(processHandler)


def __createDeafultProcessHandlerForWorld(world):
	"""
	Creates default process handler and applies it to world.

	The default process handler is L{svs_simulation.world.processes.StandaloneProcessHandler}.
	"""
	from svs_simulation.world.processes import StandaloneProcessHandler
	procHandler = StandaloneProcessHandler()
	world.addProcessHandler(procHandler)


#############################
# SAVE FILE
#############################
def saveWorldToFile(world, filename):
	"""
	Writes world data to file.
	"""
	from svs_core.utilities.lib import validateFilePath
	if not filename[-1] == '/':filename = filename + '/'
	filename = validateFilePath(filename)
	worlddir = validateFilePath(filename + 'world/')
	data = world.encode()
	dataStr = "world = %s" % data
	f = open(worlddir + 'world_data.svs', 'w')
	f.write(dataStr)
	f.close()

#############################
# CLASSES
#############################
class SimWorld:
	"""
	The simulation world is the root for an entire simulation.

	It provides terrain data, simulation time and event management.
	"""
	def __init__(self):
		self.clock = None
		self.eventCentre = SimEventCentre()
		self.scheduleManager = ScheduleManager()
		self.agentGroups = CustomEntityRepository()
		self.objectGroups = CustomEntityRepository()
		self.processGroups = CustomEntityRepository()
		self.processHandlers = []
		self.reset()

	def reset(self):
		"""
		Resets playback time to start.
		"""
		self.realStartTime = 0
		self.realTime = 0
		self.realElapsedTime = 0
		self.running = False

	#############################
	# PROCESS FUNCTIONS
	#############################
	def startWorld(self, timeStep=None):
		"""
		Starts playback.  If time step is specified uses this
		instead of default value (L{time_const.DEFAULT_TIMESTEP}).
		"""
		if not self.terrain:raise SimWorldException("Unable to start world, no terrain")
		if not self.clock:raise SimWorldException("Unable to start world, no clock")
		if not self.schedule:raise SimWorldException("Unable to start world, no schedule")
		self.realStartTime = time()
		self.realTime = self.realStartTime
		self.realElapsedTime = 0
		self.running = True
		self.clock.start()
		simTime = self.clock.update(self.realTime)
		for process in self.processHandlers:process.startWorld(simTime)

	def stopWorld(self):
		"""
		Stops playback.
		"""
		self.running = False
		simTime = self.clock.update(self.realTime)
		for process in self.processHandlers:process.stopWorld(simTime)
		self.clock.stop()

	def updateWorld(self):
		"""
		Updates world.
		"""
		self.realTime = time()
		self.realElapsedTime = self.realTime - self.realStartTime
		simTime = self.clock.update(self.realTime)
		self.updateSchedule(simTime)
		self.eventCentre.updateWorld(simTime)
		for process in self.processHandlers:process.updateWorld(simTime)

	def updateSchedule(self, simTime):
		"""
		Updates schedule.
		"""
		actions = self.scheduleManager.updateWorld(simTime)
		if not actions:return
		print simTime[0]
		for action in actions:
			print action
			target = self.getTargetForAction(action.target)
			print target
			if target:action.execute(target)


	def getChanges(self):
		"""
		Returns list of changes within simulation that 
		have occurred during the last update.
		"""
		changes = []
		for process in self.processHandlers:
			procChanges = process.getChanges()
			if procChanges:changes.extend(procChanges)
		return changes
		

	#############################
	# PROCESS HANDLERS
	#############################
	def addProcessHandler(self, procHandler):
		"""
		Adds new proces shandler to world ensuring that it is unique.
		"""
		if procHandler in self.processHandlers:return
		procHandler.addToWorld(self)
		self.processHandlers.append(procHandler)

	#############################
	# ACTIONS
	#############################
	def getTargetForAction(self, targetID):
		"""
		Tries to find agent or object, matching target ID.

		Target ID format: ('entity type', 'group name' , 'entity name')
		"""
		if targetID == None:return None
		if targetID[0] == sim_const.LABEL_AGENT:
			return self.agentGroups.getMemberInGroup(targetID[1], targetID[2])
		elif targetID[0] == sim_const.LABEL_OBJECT:
			return self.objectGroups.getMemberInGroup(targetID[1], targetID[2])
		return None

	#############################
	# TERRAIN
	#############################
	def setTerrain(self, terrain):
		"""
		Sets new terrain on world.
		"""
		self.terrain = terrain
		self.terrain.world = self

	#############################
	# AGENTS
	#############################
	def getAgentGroups(self):
		"""
		Returns list of agent groups.
		"""
		return self.agentGroups.groups

	def getAgentGroup(self, groupName):
		"""
		Returns specified agent group.
		"""
		return self.agentGroups.getGroup(groupName)

	def getAgentContainingPoint(self, x, y):
		"""
		Returns agent containing specified point.
		"""
		for group in self.agentGroups.groups.values():
			for member in group.members.values():
				if member.bounds.containsPoint(x,y):return member
		return None

	#############################
	# SIMOBJECTS
	#############################
	def getObjectGroups(self):
		"""
		Returns list of object groups.
		"""
		return self.objectGroups.groups

	def getObjectGroup(self, groupName):
		"""
		Returns specified object group.
		"""
		return self.objectGroups.getGroup(groupName)

	def getObjectContainingPoint(self, x, y):
		"""
		Returns object containing specified point.
		"""
		for group in self.objectGroups.groups.values():
			for member in group.members.values():
				if member.bounds.containsPoint(x,y):return member
		return None

	#############################
	# INFO FUNCTIONS
	#############################
	def getCurrentTime(self):
		"""
		Returns current simulation time in world.
		"""
		if not self.clock:return None
		return self.clock.currentTime

	#############################
	# ARCHIVING
	#############################
	def setup(self, name=None, clock=None, schedule=None):
		self.name = name
		self.terrain = None
		self.clock = clock
		self.schedule = schedule
		self.eventCentre = SimEventCentre()
		self.agents = {}
		self.reset()
		

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		data = {sim_const.LABEL_NAME:self.name}
		if self.clock:data[sim_const.LABEL_SIMCLOCK] = self.clock.encode()
		if self.schedule:data[sim_const.LABEL_SCHEDULE] = self.schedule.encode()
		return data

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		if data.has_key(sim_const.LABEL_SIMCLOCK):
			clock= SimClock()
			clock.decode(data[sim_const.LABEL_SIMCLOCK])
		else:clock = None
		if data.has_key(sim_const.LABEL_SCHEDULE):
			schedule = Schedule()
			schedule.decode(data[sim_const.LABEL_SCHEDULE])
		else:schedule = None
		self.setup(name=data[sim_const.LABEL_NAME],
			clock=clock,
			schedule=schedule)
