# svs_soya.basic.gameviews

#    Copyright (c) 2005 Simon Yuill, Stefan Gartner.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


"""
Wrapper classes for Soya.

Note: in its current form, aspects of the Soya client
are specific to the 'spring_alpha:spring_city' demo: U{http://www.spring-alpha.org}

@author:	Simon Yuill, Stefan Gartner
@copyright:	2005 Simon Yuill, Stefan Gartner
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org, stefang@aon.at
"""

# TODO: experiment with shaders for cellshading

# external imports
import soya, os, soya.sphere
from soya import ode
from svs_soya.basic.soya_ext import Cal2dVolume, GravObject

# these models are part of other models, thus we don't have to create a shape for those structures
# the StructureView is still hanging around, though
no_shape_structures_3d = ["riverbank1", "riverbank2", "riverbank3", "riverbank4",
					   "riverbank5", "riverbank6", "playpark2", "suburbhouse1gardengrass3"]

class View:
	"""
	Base class for all *Views
	"""
	def __init__(self, data, parent, context):
		self.parent = parent
		self.data = data
		self.scale = 1.0
		
		if self.parent:
			self.world = soya.World(self.parent.world)
			self.scale = self.parent.scale
		else:
			self.world = soya.World()
		
		self.context = context
		
	def createShape(self):
		"""
		create soya shape from data
		"""
		self.drawBounds()
		
	def drawBounds(self):
		"""
		draw bounds of object
		"""
		min_x = self.data.bounds.minX / self.scale
		min_y = self.data.bounds.minY / self.scale
		max_x = self.data.bounds.maxX / self.scale
		max_y = self.data.bounds.maxY / self.scale
		v0 = soya.Vertex(self.world, min_x, 0, min_y)
		v1 = soya.Vertex(self.world, min_x, 0, max_y)
		v2 = soya.Vertex(self.world, max_x, 0, max_y)
		v3 = soya.Vertex(self.world, max_x, 0, min_y)

		material = soya.Material()
		material.diffuse = (0.0, 0.5, 0.0, 1.0)
		soya.Face(self.world, [v0, v1, v2, v3], material)
		
	def onSingleClick(self, event):
		# convert local coords to global coords
		# convert_to() modifies event, thus we have to make a copy of it
		p2 = soya.Point(event[0].parent)
		p2.set_xyz(event[0].x, event[0].y, event[0].z)
		p2.convert_to(self.world.get_root())
		
		# location of click in svs coordinates
		click_coords2d = (p2.x * self.scale, p2.z * self.scale)
		print "click: (%.4f, %.4f)"  % (click_coords2d)
		
		self.context.infoMessage(str(self.data.name))
		
	def onDoubleClick(self, event):
		# need to pass event back to client and/or gui
		print "double click on %s (%s)" %(self.data.name, self.data)
		
	def displayName(self):
		self.context.infoMessage(str(self.data.name))

class TerrainView(View):
	"""
	Soya representation of a Terrain
	"""
	def __init__(self, data, parent=None, context=None):
		# Terrains have no parent entitiy,
		# but their Worlds are children of svs_soya.simclients.Soya.ClientGUI.gameWorld
		# thus we cannot call View.__init__() directly
		self.data = data
		self.parent = parent
		self.context = context
		self.world = soya.World(self.context.gameWorld)
		self.scale = self.context.scale
		
		minX = self.data.bounds.minX
		maxX = self.data.bounds.maxX
		minY = self.data.bounds.minY
		maxY = self.data.bounds.maxY
		
		self.centre = ((minX + (maxX / 2.0)) / self.scale, (minY + (maxY / 2.0)) / self.scale)
		
		
		if self.context.draw3D:
			self.context.gameWorld.limits = (minX / self.scale - 1.0, maxX / self.scale + 1.0,
											 minY / self.scale - 1.0, maxY / self.scale + 1.0)
		else:
			self.context.gameWorld.limits = (minX / self.scale, maxX / self.scale,
											 minY / self.scale, maxY / self.scale)
		
# 		self.light = soya.Light(self.world)
# 		self.light.cast_shadow = False
# 		#self.light.directional = True
# 		self.light.static = False
# 		self.light.diffuse = (0.5, 0.5, 0.6, 1.0)
# 		self.light.set_xyz(self.centre[0], 1000.0, self.centre[1])
# 		
		self.partitions = {}
		self.portals = None
		
		# file containing description of portals
		portals_filename = "spring_city.portals"
		self.loadPortals(portals_filename)

	def createShape(self):
		"""
		Create soya shape representing terrain
		"""
		min_x = self.data.bounds.minX / self.scale
		min_y = self.data.bounds.minY / self.scale
		max_x = self.data.bounds.maxX / self.scale
		max_y = self.data.bounds.maxY / self.scale
		v0 = soya.Vertex(self.world, min_x, -0.01, min_y)
		v1 = soya.Vertex(self.world, min_x, -0.01, max_y)
		v2 = soya.Vertex(self.world, max_x, -0.01, max_y)
		v3 = soya.Vertex(self.world, max_x, -0.01, min_y)
		
		material = soya.Material()
		material.diffuse = (0.2, 0.2, 0.2, 1.0)
		
		f = soya.Face(self.world, [v0, v1, v2, v3], material)
		f.context = self
		self.shape = self.world.shapify()
		
	def addPartitionView(self, data):
		part_portals = None
		if self.portals:
			part_portals = []
			for portal in self.portals:
				if portal['partition'] == data.name:
					part_portals.append(portal)
		
		pv = PartitionView(data, self, self.context, part_portals)
		self.partitions[data.name] = pv
		
	def loadPortals(self, filename):
		portals_file = None
		self.portals = []
		
		for directory in soya.path:
			try:
				portals_file = open(directory+os.sep+filename)
			except IOError:
				continue
			else:
				break
		
		if portals_file:
			self.portals = eval(portals_file.read())
			portals_file.close()
		
class PartitionView(View):
	"""
	Soya representation of a Partition
	"""
	def __init__(self, data, parent=None, context=None, portals=None):
		View.__init__(self, data, parent, context)
		
		self.centre = ((self.data.bounds.minX + (self.data.bounds.maxX / 2.0)) / self.scale,
					   (self.data.bounds.minY + (self.data.bounds.maxY / 2.0)) / self.scale)
		
		self.structures = {}
		# list of portal descriptions (as exported from blender)
		self.portals = portals
		# list of actual portals
		self.portList = []
		
		for key, data in self.data.structures.iteritems():
			self.addStructureView(data)
			
# 		if self.context.draw3D:
# 			# just a test
# 			self.createPortals()
# 			for structure in self.structures.values():
# 				structure.createPortals()

		self.createShape()
			
	def addStructureView(self, data):
		struct_portals = None
		if self.portals:
			struct_portals = []
			for portal in self.portals:
				if data.name in [portal['w1'], portal['w2']]:
					struct_portals.append(portal)

		sv = StructureView(data, self, self.context, struct_portals)
		self.structures[data.name] = sv
		
	def createShape(self):
		partitionWorld = None
		partitionName = self.data.name
		if self.context.draw3D:
			partitionName += "_3d"
		else:
			partitionName += "_2d"
		try:
			partitionWorld = soya.World.get(partitionName)
		except:
			pass
			#print "not loading model %s for partition %s" % (partitionName, self.data.name)
		else:
			self.shape = partitionWorld.shapify()
			self.shape.context = self
			
			self.world.set_shape(self.shape)
			self.world.set_xyz(self.centre[0], 0.001, self.centre[1])
			self.world.context = self
	
	def drawBounds(self):
		v0 = soya.Vertex(self.world, self.data.bounds.minX / self.scale, 0.01, self.data.bounds.minY / self.scale)
		v1 = soya.Vertex(self.world, self.data.bounds.minX / self.scale, 0.01, self.data.bounds.maxY / self.scale)
		v2 = soya.Vertex(self.world, self.data.bounds.maxX / self.scale, 0.01, self.data.bounds.maxY / self.scale)
		v3 = soya.Vertex(self.world, self.data.bounds.maxX / self.scale, 0.01, self.data.bounds.minY / self.scale)
		material = soya.Material()
		material.diffuse = (0.0, 0.5, 0.0, 1.0)
		soya.Face(self.world, [v0, v1], material)
		soya.Face(self.world, [v1, v2], material)
		soya.Face(self.world, [v2, v3], material)
		soya.Face(self.world, [v3, v0], material)
		
	# just a test
	def createPortals(self):
		if self.portals:
			for portal in self.portals:
				loc = portal['loc']

				# crashes raypicking?
				p = soya.Portal(self.world)
				
				if portal['w1'] == self.data.name:
					remote_name = portal['w2']
				else:
					remote_name = portal['w1']
				
				p.beyond = self.structures[remote_name].world
				
				p.use_clip_plane = 1
				p.nb_clip_planes = 4
				p.turn_lateral(portal['z_rot']+180)
				p.set_xyz(loc[0], 0.0, -loc[1])
				p.scale(4.0, 4.0, 1.0)
				
				self.portList.append(p)
	
class StructureView(View):
	"""
	Soya representation of a Structure
	"""
	def __init__(self, data, parent, context=None, portals=None):
		self.parent = parent
		self.data = data
		self.scale = 1.0
		
		if self.parent:
			self.world = soya.World(self.parent.world)
			self.scale = self.parent.scale
		else:
			self.world = soya.World()
		
		self.context = context
		# for ode
		self.geom = None
		
		# list containing portal description
		self.portals = portals
		# list containing the actual soya portals
		self.portList = []
		
		worldbounds = self.data.getWorldbounds()
		objbounds = self.data.bounds
		self.centre = ((worldbounds.minX + (objbounds.maxX / 2.0)) / self.scale,
		               (worldbounds.minY + (objbounds.maxY / 2.0)) / self.scale)
		
		self.createShape()
			
	def createShape(self):
		shapename = self.data.name
		
		# don't create a shape for structures which are covered by other structures
		if self.context.draw3D and shapename in no_shape_structures_3d:
			return
		
		structure = None

		if self.context.draw3D:
			shapename += "_3d"
		
		try:
			structure = soya.World.get(shapename)
		except:
			print shapename, "missing"
 		else:
			# whether a model should be drawn using cellshading is defined in the .blend file
			# of the corresponding model
			self.shape = structure.shapify()
			self.shape.context = self
			
			self.world.set_shape(self.shape)
			self.world.set_xyz(self.centre[0], 0.001, self.centre[1])
			self.world.context = self
		
	# just a test
	def createPortals(self):
		if self.portals:
			for portal in self.portals:
				loc = portal['loc']

				# crashes raypicking?
				p = soya.Portal(self.world)
				
				if portal['w1'] == self.data.name:
					remote_name = portal['w2']
				else:
					remote_name = portal['w1']
				
				# these are partitions, not structures
				# right now, structures can only have portals to their parent partition
				if remote_name in ['west', 'east', 'south', 'north']:
					if remote_name == self.parent.data.name:
						p.beyond = self.parent.world
					else:
						print "warning: portal from %s to %s, which is not %s's parent partition" %(self.data.name, remote_name, self.data.name)
				else:
					p.beyond = self.parent.structures[remote_name].world
				
				p.use_clip_plane = 1
				p.nb_clip_planes = 4
				p.turn_lateral(portal['z_rot'])
				p.set_xyz(self.centre[0] + loc[0], 0.0, self.centre[1] - loc[1])
				p.scale(4.0, 4.0, 1.0)
				
				self.portList.append(p)


# objects and agents are slightly different in that they don't have a parent
# object themselves, but their worlds are children of the context's gameWorld
class ObjectView(View):
	"""
	Soya representation of an Object
	"""
	def __init__(self, data, parent=None, context=None):
		self.parent = parent
		self.data = data
		self.context = context
		self.scale = self.context.scale
		self.angle = -self.data.facing
		
		worldbounds = self.data.getWorldbounds()
		self.centre = (worldbounds.x / self.scale, worldbounds.y / self.scale)
		
		#self.world = GravObject(self.context.gameWorld, self.context.draw3D, rotate=False)
		self.world = soya.World(self.context.gameWorld)
		self.world.context = self
		self.world.set_xyz(self.centre[0], 0.0015, self.centre[1])
		self.world.turn_lateral(self.angle)
		
		self.createShape()
		
	def createShape(self):
		objname =  self.data.sourceClass.lower()
		if self.context.draw3D:
			objname += "_3d"
		else:
			objname += "_2d"
		
		try:
			obj = soya.World.get(objname)
		except:
			print "object %s missing" % objname
		else:
			shapifier = soya.CellShadingShapifier()
			shapifier.outline_width = 2.0
			shapifier.outline_attenuation = 0.5
			
			#if self.context.draw3D:
				#shader = soya.Material()
				#shader.texture = soya.Image.get("shader2.png")
				#shapifier.shader = shader
			
			obj.shapifier = shapifier
			
			shape = obj.shapify()
			self.world.set_shape(shape)
	
	def setLocation(self, location):
		self.world.x = location[0] / self.scale
		self.world.z = location[1] / self.scale
	
	def setFacing(self, facing):
		self.angle = -(facing - self.facing)
		self.world.turn_lateral(self.angle)
		self.facing = facing
		
	def update(self, data):
		print "object", self.data.name, "update", data
		if data.has_key('location'):
			self.setLocation(data['location'])
		if data.has_key('facing'):
			self.setFacing(data['facing'])


class AgentView(View):
	"""
	Soya representation of an Agent
	"""
	def __init__(self, data, parent=None, context=None):
		self.parent = parent
		self.data = data
		self.context = context
		self.scale = self.context.scale
		self.angle = -self.data.facing
		self.facing = self.data.facing
		
		worldbounds = self.data.getWorldbounds()
		self.centre = (worldbounds.x / self.scale, worldbounds.y / self.scale)
		
		self.world = GravObject(self.context.gameWorld, self.context.draw3D, rotate=False)
		self.world.set_xyz(self.centre[0], 0.0015, self.centre[1])
		self.world.context = self
		self.world.turn_lateral(self.angle)
		
		self.createShape()
		
	def createShape(self):
		objname =  self.data.sourceClass.lower()
		
		if self.context.draw3D:
			objname += "_3d"
		else:
			objname +="_2d"
		
		try:
			agent = soya.World.get(objname)
		except:
			print "object %s missing" % objname
		else:
			shapifier = soya.CellShadingShapifier()
			shapifier.outline_width = 2.0
			shapifier.outline_attenuation = 0.5
			
			#if self.context.draw3D:
				#shader = soya.Material()
				#shader.texture = soya.Image.get("shader2.png")
			#	shapifier.shader = shader
		
			agent.shapifier = shapifier
			shape = agent.shapify()
			self.world.set_shape(shape)
			
		if objname == "armytank_3d":
			self.world.rotate = True
		
	def setLocation(self, location):
		self.world.x = location[0] / self.scale
		self.world.z = location[1] / self.scale
	
	def setFacing(self, facing):
		#print "agent", self.data.name, "facing", facing
		self.angle = -(facing - self.facing)
		self.world.turn_lateral(self.angle)
		self.facing = facing
		
	def update(self, data):
		#print "agent", self.data.name, "update", data
		if data.has_key('location'):
			self.setLocation(data['location'])
		if data.has_key('facing'):
			self.setFacing(data['facing'])
		
