# svs_soya.soya_ext

#    Copyright (c) 2005 Simon Yuill, Stefan Gartner.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Some extended Soya classes

@author:	Stefan Gartner, Simon Yuill
@copyright:	2005 Stefan Gartner, Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	stefang@aon.at, simon@lipparosa.org
"""

# external imports
import soya, soya.widget
import time, os, sys
from math import sqrt, acos, degrees

# internal imports
from svs_simulation.numdata.mathlib import math_const

# for InteractiveCamera
ZOOM_MIN =  0
ZOOM_MAX =  8

class RootWorld(soya.World):
	"""
	The root world, mainly used for events handling
	there can only be one RootWorld in a game
	"""
	def __init__(self, parent=None, context=None):
		soya.World.__init__(self, parent)
		self.event_consumers = []
		self.context = context

	def addEventConsumer(self, consumer):
		if consumer not in self.event_consumers and consumer not in self.children:
			self.event_consumers.append(consumer)
				
	def process_event(self, event):
		consumed_event = False
		if self.context.fullscreen and event[0] == soya.sdlconst.MOUSEBUTTONDOWN:
			if event[1] == soya.sdlconst.BUTTON_RIGHT:
				consumed_event = True
				if self.context.root.IsShown():
					self.context.root.Hide()
				else:
					self.context.root.Show(True)
		# should be disabled for exhibition
		if not self.context.kiosk:
			if event[0] == soya.sdlconst.KEYDOWN:
				if event[1] in [soya.sdlconst.K_ESCAPE, soya.sdlconst.K_q]:
					consumed_event = True
					self.context.destroy()
			elif event[0] == soya.sdlconst.QUIT:
					consumed_event = True
					self.context.destroy()
			
		# if we consume events ourselves, we should not pass them
		# on to our children
		if not consumed_event:
			for child in self.children:
				if hasattr(child, "process_event"):
					child.process_event(event)
			# pass on events to entities which are not our children, but
			# registered as event consumers
			# TODO: pass mouseclicks only to scenes/cameras when they're in their
			# viewport
			for consumer in self.event_consumers:
				if consumer and hasattr(consumer, "process_event"):
					consumer.process_event(event)

	def begin_round(self):
		for event in soya.process_event():
			self.process_event(event)	

class InteractiveWorld(soya.World):
	"""
	event consuming world
	usage:
		rootWorld=RootWorld()
		myWorld = InterActiveWorld(rootWorld)
		rootWorld.addEventConsumer(myWorld)
	"""
	def __init__(self, parent=None, draw3D=False):
		soya.World.__init__(self, parent)
		self.draw3D = draw3D
		self.event_consumers = []

	def addEventConsumer(self, consumer):
		self.event_consumers.append(consumer)
		
	def process_event(self, event):
		consumed_event = False

		if not consumed_event:
			# pass event on to objects that registered as event consumers with us
			for consumer in self.event_consumers:
				consumer.process_event(event)
					
	def advance_time(self, proportion):
		soya.World.advance_time(self, proportion)

# for info view.. a World that only processes mouse events
class InfoWorld(soya.World):
	def __init__(self, parent=None):
		soya.World.__init__(self, parent)
		self.clicked = False
		self.dragging = False
		self.startDragPos = None
		self.dragLine = None
		self.impact = None
		self.old_mouse = None
		
	def process_event(self, event):
		if event[0] == soya.sdlconst.MOUSEBUTTONDOWN:
			#shift event coordinates to match viewport
			localcoords = (event[2]-self.camera.left, event[3]-self.camera.top)
			# ignore events outside our viewport
			if (localcoords[0] < 0 or localcoords[1] < 0) or (localcoords[0] > self.camera.width or localcoords[1] > self.camera.height):
				return
				
			if event[1] == soya.sdlconst.BUTTON_LEFT:
				mouse = self.camera.coord2d_to_3d(localcoords[0], localcoords[1])				
				result = self.raypick(self.camera, self.camera.vector_to(mouse))
				if result:
					self.impact, normal = result
					if not self.dragging:
						self.dragging = True
						self.impact.convert_to(self.camera)
						self.old_mouse = self.camera.coord2d_to_3d(event[2], event[3], self.impact.z)
						self.dragLine = soya.World(self)
						self.dragLine.start = soya.Vertex(self.dragLine, self.impact.x, self.impact.y, self.impact.z)
						self.dragLine.end = soya.Vertex(self.dragLine, self.impact.x, self.impact.y, self.impact.z)
						soya.Face(self.dragLine, [self.dragLine.start, self.dragLine.end])

		elif event[0] == soya.sdlconst.MOUSEBUTTONUP:	
			if event[1] == soya.sdlconst.BUTTON_LEFT:
				self.dragging = False
					
		elif event[0] == soya.sdlconst.MOUSEMOTION:
			if self.dragging:
				# Computes the new mouse position, at the same Z value than impact.
				new_mouse = self.camera.coord2d_to_3d(event[1], event[2], self.impact.z)
				v = self.old_mouse.vector_to(new_mouse)
				self.dragLine.end.set_xyz(v.x, v.y, v.z)

class FixedViewportCamera(soya.Camera):
	"""
	a simple camera, filling only parts of the window
	"""
	def __init__(self, parent, left, top, width, height):
		soya.Camera.__init__(self, parent)
		self.set_viewport(left, top, width, height)
		self.partial = 1
		
	def resize(self, left, top, width, height):
		pass

class InteractiveCamera(FixedViewportCamera):
	"""
		a simple movable camera (filling only parts of the window)
		must be child of an InteractiveWorld
	"""
	def __init__(self, parent, left, top, width, height, draw3D=False):
		FixedViewportCamera.__init__(self, parent, left, top, width, height)
		self.draw3D = draw3D
		self.clicked = False
		self.recording = False
		self.recframe = 0
		self.click_timer = 0.0
		self.zoomstep = 5
		self.zoomfactors = (1.0, 2.0, 5.0, 10.0, 20.0, 50.0, 100.0)
	
	def process_event(self, event):
		consumed_event = False
		if event[0] == soya.sdlconst.MOUSEBUTTONDOWN:
			#shift event coordinates to match viewport
			localcoords = (event[2]-self.left, event[3]-self.top)
			# ignore events outside our viewport
			if (localcoords[0] < 0 or localcoords[1] < 0) or (localcoords[0] > self.width or localcoords[1] > self.height):
				return
			if event[1] == soya.sdlconst.BUTTON_LEFT:
				double_clicked = False
				# check for doubleclick:
				if not self.clicked:
					self.click_timer = 0.0
					self.clicked = True
				else:
					# double click?
					# TODO: check delta of mouse coordinates
					double_clicked = True
					self.clicked = False
				
				if self.draw3D:
					mouse = self.coord2d_to_3d(localcoords[0], localcoords[1])
					# doesn't work in combination with portals (yet?)
					result = self.to_render.raypick(self, self.vector_to(mouse))
					if result:
						# check if the picked object is from us
						if hasattr(result[0].parent, "context"):
							consumed_event = True
							if double_clicked:
								result[0].parent.context.onDoubleClick(result)
							else:
								result[0].parent.context.onSingleClick(result)
			
		elif event[0] == soya.sdlconst.KEYDOWN:
			if event[1] in [soya.sdlconst.K_KP_MINUS, soya.sdlconst.K_MINUS]:
				# "zoom" only when in 2D view
				if not self.draw3D and self.zoomstep < ZOOM_MAX:
					self.zoomstep += 1
					self.scale(2.0, 2.0, 2.0)
			elif event[1] in [soya.sdlconst.K_KP_PLUS, soya.sdlconst.K_PLUS]:
				if not self.draw3D and self.zoomstep > ZOOM_MIN:
					self.zoomstep -= 1
					self.scale(0.5, 0.5, 0.5)
# 		elif event[0] == soya.sdlconst.KEYUP:
# 			if event[1] == soya.sdlconst.K_r:
# 				if self.recording:
# 					self.recording = False
# 					print "stop recording"
# 				else:
# 					self.recording = True
#  					print "start recording"

	
	def begin_round(self):
		FixedViewportCamera.begin_round(self)
		if self.to_render:
			# from the soya tutorials:
			# Checks if the camera has passed through a portal.
			# First, collects all portals in the camera's root world.
			# the World.search_all method take a predicat (a one argument callable), and
			# returns a list of all items (recursively) in the world that satisfy the predicat.

			portals = self.to_render.search_all(lambda item: isinstance(item, soya.Portal))
			
			# Then for each portal, checks if the camera has pass through it, and if so,
			# transfers the camera in the world beyond the portal.
			# The has_passed_through method takes two argument : the old position of the object
			# and the new one or (as here) the speed vector.
    
			for portal in portals:
				if portal.has_passed_through(self, self.speed):
					portal.pass_through(self)
					print "pass !", self.position(), self.speed, self.to_render
					self.to_render = portal.beyond

	def advance_time(self, proportion):
		FixedViewportCamera.advance_time(self, proportion)
			
		if self.clicked:
			self.click_timer += proportion
			if self.click_timer > 10.0:
				self.click_timer = 0.0
				self.clicked = False
		
		if self.recording:
			filename = str(self.recframe).zfill(4)+".jpg"
			soya.screenshot().save(os.path.join(os.path.dirname(sys.argv[0]), "render"+os.sep+filename))
			self.recframe += 1
				

class GravObject(soya.World):
	"""
	an object obeying gravitiy. not quite the proper name...
	use this if you want an object to follow the landscape
	"""
	def __init__(self, parent=None, draw3D=False, offset=0.0, rotate=False):
		soya.World.__init__(self, parent)
		self.draw3D = draw3D
		# for raypicking (simple collision detection with ground)
		self.down = soya.Vector(self, 0.0, -1.0, 0.0)
		self.up   = soya.Vector(self, 0.0,  1.0, 0.0)
		self.front = soya.Vector(self, 1.0, 0.0, 0.0)
		self.left = soya.Vector(self, 0.0, 0.0, -1.0)
		self.offset = offset
		self.rotate=rotate
		self.lastAngle = 0.0
		
	def collide(self, pos, direction, count):
		# abort after 6 recursions
		if count == 5:
			return None
		
		impact = self.parent.raypick(pos, direction)
		# to prevent smaller objects floating in the air
		# because the first object the ray intersects is the object itself
		if impact and impact[0].parent == self:
			pos.y = impact[0].y
			return self.collide(pos, direction, count+1)
		
		return impact
	
	def begin_round(self):
		soya.World.begin_round(self)
		if self.draw3D:
			# inspired by soya/laser.py
			# TODO: make falling more realistic (trajectory?)
			pos = self.position()
			# we always want to make sure that there's something 'below', thus
			# we move the y coordinate up a bit. Maybe there's a better way?
			pos.y += 0.8
			impact = self.collide(pos, self.down, 0)
			
			if impact:
				# not sure if this works/is necessary everywhere
				if str(impact[0].y) != 'nan':
					self.y = impact[0].y+self.offset
				
				if self.rotate:
					normal = impact[1]
					nla = normal.angle_to(self.left)
					nfa = normal.angle_to(self.front)
					nua = normal.angle_to(self.up)
					
					rot_angle = self.lastAngle
					# > 90: down, < 90: up
					# don't ask what that / 2.0 is doing here...
					# using the actual angle causes the rotation to overshoot
					# and create lots of jitter
					if nfa - 90.0 > 0.001:
						rot_angle = nua / 2.0
					elif nfa - 90.0 < -0.001:
						rot_angle = -nua / 2.0
						
					self.turn_incline(rot_angle-self.lastAngle)
					self.lastAngle = rot_angle
					
			else:
				self.y = self.offset
	

class InteractiveDolly(GravObject):
	"""
	a camera dolly, steerable with cursor keys
	"""
	def __init__(self, parent=None, draw3D=False, offset=1.0):
		GravObject.__init__(self, parent, draw3D, offset)
		# movement
		self.speed = soya.Vector(self)
		self.angle = 0.0

	def process_event(self, event):
		if event[0] == soya.sdlconst.KEYDOWN:
			if event[1] == soya.sdlconst.K_UP:
				consumed_event = True
				if self.draw3D:
					self.speed.x =  0.2
				else:
					self.speed.y =  0.2
			elif event[1] == soya.sdlconst.K_DOWN:
				consumed_event = True
				if self.draw3D:
					self.speed.x =  -0.2
				else:
					self.speed.y = -0.2
			elif event[1] == soya.sdlconst.K_LEFT:
				consumed_event = True
				if self.draw3D:
					self.angle = 5.0
				else:
					self.speed.x = -0.2
			elif event[1] == soya.sdlconst.K_RIGHT:
				consumed_event = True
				if self.draw3D:
					self.angle = -5.0
				else:
					self.speed.x =  0.2

		elif event[0] == soya.sdlconst.KEYUP:
			if event[1] in [soya.sdlconst.K_UP, soya.sdlconst.K_DOWN]:
				consumed_event = True
				if self.draw3D:
					self.speed.x = 0.0
				else:
					self.speed.y = 0.0
			elif event[1] in [soya.sdlconst.K_LEFT, soya.sdlconst.K_RIGHT]:
				consumed_event = True
				if self.draw3D:
					self.angle = 0.0
				else:
					self.speed.x = 0.0

		elif event[0] == soya.sdlconst.JOYAXISMOTION:
			# up / down
			if event[1] == 1: 
				# let's assume it's a digital joystick
				if event[2] < 0:
					consumed_event = True
					if self.draw3D:
						self.speed.x =  0.2
					else:
						self.speed.y =  0.2
				elif event[2] > 0:
					consumed_event = True
					if self.draw3D:
						self.speed.x =  -0.2
					else:
						self.speed.y =  -0.2
				else:
					if self.draw3D:
						self.speed.x =  0.0
					else:
						self.speed.y =  0.0
			# left / right
			elif event[1] == 0: 
				# let's assume it's a digital joystick
				if event[2] < 0:
					consumed_event = True
					if self.draw3D:
						self.angle = 2.0
					else:
						self.speed.x = -0.2
				elif event[2] > 0:
					consumed_event = True
					if self.draw3D:
						self.angle = -2.0
					else:
						self.speed.x = 0.2
				else:
					consumed_event = True
					if self.draw3D:
						self.angle = 0.0
					else:
						self.speed.x = 0.0

	def advance_time(self, proportion):
		GravObject.advance_time(self, proportion)
		self.turn_lateral(self.angle * proportion)
		
		self.add_mul_vector(proportion, self.speed)
		
		# reset position to be within the game's limits
		pos = self.position()
		if self.parent.limits:
			x = min(max(pos.x, self.parent.limits[0]), self.parent.limits[1])
			z = min(max(pos.z, self.parent.limits[2]), self.parent.limits[3])
			self.x = x
			self.z = z

class FPSLabel(soya.widget.Label):
	def __init__(self, parent=None, font=soya.widget.default_font):
		soya.widget.Label.__init__(self, parent, font=font)
		self.lasttime = time.time()
		self.text = "0.00"
		
	def widget_begin_round(self):
		soya.widget.Label.widget_begin_round(self)
		curtime = time.time()
		laptime = curtime - self.lasttime
		fps = 1.0 / laptime
		self.text = "%.2f" %fps
		self.lasttime = curtime

class Cal2dVolume(soya.Cal3dVolume):
	def __init__(self, scene, shape):
		soya.Cal3dVolume.__init__(self, scene, shape)
		self.states = []
		self.state = "rest_pose"
		
	def setState(self, state):
		# todo: self.state should be list of valid states we compare against
		# we could also just compare against self.shape.animations
		if self.state != state:
			self.state = state
			self.animate_blend_cycle(state)
		
	def advance_time(self, proportion):
		soya.Cal3dVolume.advance_time(self, proportion)
		# todo: compare with current state? timeout?
		self.state = "rest_pose"
		# it should stop here, but it doesn't????
		self.animate_blend_cycle(self.state)

