# svs_soya.clients

#    Copyright (c) 2005 Simon Yuill, Stefan Gartner.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


# TODO: find proper place for SoyaClientGUI
"""
Specialised clients for managing a simulation (soya variant).

Note: this version contains code that is specific to the
'spring_alpha:spring_city' demo: U{http://www.spring-alpha.org}


@author:	Simon Yuill, Stefan Gartner
@copyright:	2005 Simon Yuill, Stefan Gartner
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org, stefang@aon.at

"""

# external imports
import sys, os, soya, soya.widget as widget
import soya.sphere, soya.cube
from soya import ode
from random import randint
from wxPython.wx import *
from twisted.internet import reactor, wxsupport

# local svs imports
from svs_simulation.network.playerclients import PlayerClient
from svs_core.network.clientuser import ScriptableClient
from svs_core.commands.scriptcommands import *
from svs_core.network.packets import *
from svs_core.utilities.constants import svs_const
from svs_simulation.utilities.constants import sim_const

# local soya import
from svs_soya.basic.soya_ext import *
from svs_soya.basic.gameviews import *



# wxPython events
ID_ENTER_BUTTON=110
ID_HISTORY_BUTTON=111
ID_RESET_BUTTON=112
ID_QUIT_BUTTON=113
ID_INPUTFIELD=120

#slogans to be displayed at random times in the red bar at the bottom
slogans = ["land is shared" , "roads are dug up" , "school is not compulsory",
           "money is destroyed",  "people take turns to do the difficult jobs",
		   "everything is shared", "people have relaxing orgasms", 
		   "mercy is shown", "obedience doesn't relieve pain",
		   "people build homes and grow food", "soldiers leave the armed forces",
		   "wounds are healed", "armour dissolves", "no one charges no one pays", 
           "people stop using things"]

class SoyaGUI:
	def __init__(self, client, draw3D=False, fullscreen=False, showInfo=True, showCursor=True, kiosk=False):
		self.client = client
		self.rootWorld = RootWorld(context=self)
		self.rootWorld.atmosphere = soya.Atmosphere()
		self.rootWorld.atmosphere.bg_color = (0.67, 0.0667, 0.0667, 0.0)
		self.draw3D = draw3D
		self.fullscreen = fullscreen
		self.agent_views = {}
		self.object_views = {}
		self.logging = False
		self.showInfo = showInfo
		self.showCursor = showCursor
		self.kiosk = kiosk
		
		# for changing slogans
		self.lastTime = None
		self.switchTime = 60
		
	def build(self):
		# soya stuff..
		
		# scale factor for soya models
		self.scale = 50.0
		
		if self.fullscreen:
			w = 1024
			h = 768
		else:
			w = 640
			h = 480
		
		soya.init(title="spring_alpha", width=w, height=h, fullscreen=self.fullscreen)
		
		if not self.showCursor:
			soya.cursor_set_visible(0)
			
		self.small_font = soya.Font(os.path.join(soya.path[0],"fonts", "spring_alpha.pfb"), 16, 16)
		self.big_font   = soya.Font(os.path.join(soya.path[0],"fonts", "spring_alpha.pfb"), 38, 38)
		
		width = w
		if self.showInfo:
			height = int(soya.get_screen_height() * (216.0 / 254.0))
		else:
			height = h
		
		
		self.rootCamera = soya.Camera(self.rootWorld)
		self.rootCamera.to_render = self.rootWorld
		
		self.gameWorld = InteractiveWorld(parent=self.rootWorld, draw3D=self.draw3D)
		if self.draw3D:
			self.gameWorld.atmosphere = soya.Atmosphere()
			self.gameWorld.atmosphere.bg_color = (0.757, 0.757, 1.0, 0.0)
			self.gameWorld.atmosphere.ambient  = (0.7, 0.7, 0.7, 0.0)
			
		else:
			self.gameWorld.atmosphere = soya.Atmosphere()
			self.gameWorld.atmosphere.bg_color = (0.4, 0.4, 0.4, 0.0)
			self.gameWorld.atmosphere.ambient  = (1.0, 1.0, 1.0, 0.0)
		
		self.gameWorld.atmosphere.ambient = (1.0, 1.0, 1.0, 0.0)
		
		self.rootWorld.addEventConsumer(self.gameWorld)
		
		# bounds for vistor movement
		self.gameWorld.limits = None
		
		self.gameDolly = InteractiveDolly(parent=self.gameWorld, draw3D=self.draw3D, offset=1.0)
		self.gameWorld.addEventConsumer(self.gameDolly)
		self.gameDolly.z =  1500.0 / self.scale
		self.gameDolly.x =  5200.0 / self.scale
		
		self.gameCamera = InteractiveCamera(self.gameDolly, 0, 0, width, height, draw3D=self.draw3D)
		self.gameCamera.back = 100.0
		self.gameCamera.to_render = self.gameWorld
		self.gameWorld.addEventConsumer(self.gameCamera)
		
		if self.draw3D:
			self.gameCamera.turn_lateral(-90.0)
			self.gameDolly.y = 3.0
		
		else:
			self.gameDolly.turn_vertical(-90.0)
			self.gameDolly.y = 20.0
			self.gameCamera.scale(10.0, 10.0, 10.0)
			self.gameCamera.ortho = 1
		
		self.terrain_view = None

		# information display
		if self.showInfo:
			self.infoGroup = soya.widget.Group()
			
			y = height
			barheight = h - height
			barcenter = y + barheight / 2
			
# 			self.titleLabel = soya.widget.Label(self.infoGroup, text="spring_alpha", font=self.small_font)
# 			self.titleLabel.top = y
# 			self.titleLabel.left = 2
# 			
# 			y = self.titleLabel.top + self.titleLabel.height + 15
# 			
# 			# ideal candidate for subclass of soya.widget.Label
# 			self.helpText = soya.widget.Label(self.infoGroup,font=self.small_font)
# 			self.helpText.width = w-2
# 			# console doesn't work in fullscreen mode (might be window manager related)
# 			if not self.fullscreen:
# 				self.helpText.text = "rmb for console"
# 			
# 			self.helpText.top = y
# 			self.helpText.left = 2
# 			
			self.sloganLabel = soya.widget.Label(self.infoGroup,font=self.big_font)
			self.sloganLabel.top = barcenter - self.big_font.height / 2
			self.sloganLabel.width = w-2
			self.sloganLabel.align = 1
			self.sloganLabel.text = slogans[randint(0,len(slogans)-1)]
			
# 			y += self.helpText.height + 20
# 			
# 			self.fpsLabel = FPSLabel(self.infoGroup, font=self.small_font)
# 			self.fpsLabel.top = y
# 			self.fpsLabel.left = 2
# 			
# 			self.infoWorld = InfoWorld()
# 			self.infoWorld.atmosphere = soya.Atmosphere()
# 			self.infoWorld.atmosphere.bg_color = (0.1, 0.1, 0.1, 0.0)
# 			self.rootWorld.addEventConsumer(self.infoWorld)
# 			
# 			self.infoLight = soya.Light(self.infoWorld)
# 			self.infoLight.cast_shadow = False
# 			self.infoLight.set_xyz(10, 10, 10)
# 			
# 			self.infoSphere = soya.sphere.Sphere(self.infoWorld)
# 			
# 			cam_h = int(h - (height + 4))
# 			cam_w = int(cam_h / 3.0) * 4
# 			
# 			self.infoCamera = FixedViewportCamera(self.infoWorld, width-(cam_w+2), height+2, cam_w, cam_h)
# 			self.infoCamera.partial = 1
# 			self.infoCamera.z = 10
# 			#self.infoCamera.ortho = 1
# 			self.infoCamera.to_render = self.infoWorld
# 			self.infoWorld.camera = self.infoCamera
# 			self.infoGroup.add(self.infoCamera)
# 			
		self.rootGroup = soya.widget.Group()
		self.rootGroup.add(self.rootCamera)
		
		gameGroup = soya.widget.Group(self.rootGroup)
		gameGroup.add(self.gameCamera)
		
		if self.showInfo:
			self.rootGroup.add(self.infoGroup)
		
		soya.set_root_widget(self.rootGroup)
		
		#self.client.idler.scenes.extend([self.rootWorld, self.gameWorld, self.infoWorld])
		self.client.idler.scenes.extend([self.rootWorld, self.gameWorld])

		if not self.fullscreen:
			# wxpython GUI
			self.app = wxPySimpleApp()
			
			self.root = wxFrame(None, wxID_ANY, title="SVS Soya Client", size =(400,300),
								style=wxDEFAULT_FRAME_STYLE|wxNO_FULL_REPAINT_ON_RESIZE)
			
			# register event handler for close event ((x)-button of window decoration)
			EVT_CLOSE(self.root, self.onClose)
			
			self.mainSizer = wxBoxSizer(wxVERTICAL)
			
			self.gameInfoLabel = wxStatusBar(self.root, -1)
			self.gameInfoLabel.SetFieldsCount(1)
			self.gameInfoLabel.SetStatusText("Hello World", 0)
			
			#console
			self.consoleSizer = wxBoxSizer(wxVERTICAL)
			self.consoleLabel = wxStaticText(self.root, -1, "Console Terminal:")
			self.inputField = wxTextCtrl(self.root, ID_INPUTFIELD)
			EVT_CHAR(self.inputField, self.enterCommand)
			self.console = wxTextCtrl(self.root, -1, style=wxTE_MULTILINE | wxTE_READONLY)
			
			self.inputBuffer = []
			self.inputBufferMaxSize = 20
			self.inputBufferIndex = 0
			self.echoInput=False
			
			self.consoleSizer.Add(self.consoleLabel, 0, wxEXPAND)
			self.consoleSizer.Add(self.inputField, 0, wxEXPAND)
			self.consoleSizer.Add(self.console, 1, wxEXPAND)
		
			self.mainSizer.Add(self.consoleSizer, 1, wxEXPAND)
			self.mainSizer.Add(self.gameInfoLabel, 0, wxEXPAND)
			
			#Layout sizers
			self.root.SetSizer(self.mainSizer)
			self.root.SetAutoLayout(1)
			
			wxsupport.install(self.app)

	###############################
	### wxPython event handlers ###
	###############################

	def enterCommand(self, event):
		if not self.fullscreen:
			keycode = event.GetKeyCode()
			if keycode == WXK_RETURN:
				command = self.inputField.GetValue()
				if self.echoInput:
					self.console.AppendText(command+"\n")
				self.inputField.Clear()
				self.inputBuffer.append(command)
				if len(self.inputBuffer) > self.inputBufferMaxSize:
					self.inputBuffer = self.inputBuffer[-self.inputBufferMaxSize:]
				self.inputBufferIndex = len(self.inputBuffer) - 1
				self.client.handleLocalCommand(command)
			elif keycode == WXK_UP:
				if len(self.inputBuffer) > 0:
					cmdText = self.inputBuffer[self.inputBufferIndex]
					self.inputField.Clear()
					self.inputField.AppendText(cmdText)
					if self.inputBufferIndex >= 1:
						self.inputBufferIndex -= 1
			elif keycode == WXK_DOWN:
				if 0 <= self.inputBufferIndex < len(self.inputBuffer)-1:
					self.inputBufferIndex += 1
					self.inputField.Clear()
					self.inputField.AppendText(self.inputBuffer[self.inputBufferIndex])
			event.Skip()

	def onClose(self, event):
		self.root.Hide()
		if self.wantFullscreen:
			self.setFullscreen(True)
			
	def statusMessage(self, text):
		"""
		Displays status message in console.
		"""
		if not self.fullscreen:
			self.console.AppendText('%s\n' % text)
	
	def errorMessage(self, text):
		"""
		Displays error message in console.
		"""
		if not self.fullscreen:
			self.console.AppendText('ERROR: %s\n' % text)

	def destroy(self, args=None):
		"""
		Closes and destroys interface components.
		"""
		self.client.disconnect()

	
	def update(self, timestamp):
		"""
		Updates view components.

		This should be overridden by extending classes.
		"""
		if not self.lastTime:
			self.lastTime = timestamp
		
		if timestamp - self.switchTime >= self.lastTime:
				self.updateSlogan()
				self.lastTime = timestamp
				self.switchTime = randint(60,600)
		
		#pass
	
	def updateSlogan(self):
		"""
		display a random slogan in bar at bottom
		"""
		if self.showInfo:
			index = randint(0, len(slogans)-1)
			self.sloganLabel.text = slogans[index]
	
	##############################
	### soya related functions ###
	##############################
	
	def addTerrainView(self, data):
		self.terrain_view = TerrainView(data, parent=None, context=self)
	
	def addSimObject(self, simObj):
		"""
		Draws individual object on display.
		"""
		self.object_views[simObj._uid] = ObjectView(simObj, parent=None, context=self)
		
	def addAgent(self, agent):
		"""
		Draws individual agent on display.
		"""
		self.agent_views[agent._uid] = AgentView(agent, parent=None, context=self)
	
	def updateViews(self, data):
		for entry in data:
			update_uid=entry['source']
			if self.agent_views.has_key(update_uid):
				self.agent_views[update_uid].update(entry['data'])
			elif self.object_views.has_key(update_uid):
				self.object_views[update_uid].update(entry['data'])
				
	def infoMessage(self, text):
		self.helpText.text = text
		


class SoyaGUIClient(PlayerClient, ScriptableClient):
	"""
	SVS client with graphical interface.
	"""
	def __init__(self, name, passwd, draw3D=False, fullscreen=False, showInfo=True, showCursor=True, kiosk=True,  tickDuration=1000):
		PlayerClient.__init__(self, tickDuration=tickDuration)
		ScriptableClient.__init__(self, name, passwd)
		
		self.isRunning = False
		self.tickDelay = 0.030
		
		data_path = os.path.join(os.path.abspath(os.path.dirname(sys.argv[0])), "data")
		soya.path.append(data_path)
		self.idler = soya.Idler()
		
		self.gui = SoyaGUI(self, draw3D=draw3D, fullscreen=fullscreen, showInfo=showInfo, showCursor=showCursor)
		self.gui.kiosk = kiosk
		self.gui.build()
		
	def startClock(self):
		if not self.isRunning: self.isRunning = True
		reactor.callLater(self.tickDelay, self.tick)
		
	def stopClock(self):
		self.isRunning = False
		#self.idler.stop()
		self.disconnect()

	def tick(self):
		self.idler.begin_round()
		self.idler.advance_time(1)
		self.idler.end_round()
		self.idler.render()
		reactor.callLater(self.tickDelay, self.tick)

	#######################
	# UTILITY
	#######################
	def statusMessage(self, text):
		"""
		Handles status messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		self.gui.statusMessage(text)

	def logMessage(self, text):
		"""
		Handles log messages for client. Log messages can be turned on and off.

		This should be overridden by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		if self.logging: self.gui.statusMessage(text)

	def errorMessage(self, text):
		"""
		Handles error messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		self.gui.errorMessage(text)

	#########################
	# REMOTE METHODS
	#########################
	def remote_update(self, time):
		"""
		Handles update sent from client proxy on server.
		
		@type 	time: list
		@param 	time: current simulation time
		"""
		#self.user.update(time)
		#print "update:", time
		self.gui.update(time)
		
	#############################
	# PROCESS HANDLERS
	#############################
	def handleUpdateWorld(self, data):
		"""
		Handles update data from world.
		"""
		#self.world.handleUpdateWorld(data)
		#print data
		self.gui.updateViews(data)
		
		
	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_quit(self, cmd):
		"""
		Disconnect from server and quit client application.
		"""
		self.gui.destroy()
		return makeCommandResult(cmd, message="quitting", status=svs_const.OK)
	
	
	def setupTerrain(self, model):
		self.world.setupTerrain(model, self.lod)
		self.gui.addTerrainView(self.world.terrain)

		for partitionName in self.world.terrain.partitions.keys():
			self.getPartitionModel(partitionName)
		# get agents
		self.getAgentGroups()
		# get objects
		self.getObjectGroups()
		
	def setupPartition(self, model):
		
		partition = self.world.setupPartition(model)
		self.gui.terrain_view.addPartitionView(partition)
		
	def setupAgentGroups(self, groups):
		"""
		Handles received list of agent groups.
		"""
		self.world.setupAgentGroups(groups)
		agentGroups = self.world.getAgentGroups() 
		for group in agentGroups.keys():
			self.getAgentsInGroup(group)

	def setupAgentGroup(self, groupData):
		"""
		Handles received data for agent group.
		"""
		self.world.setupAgentGroup(groupData)
		agentGroup = self.world.getAgentGroup(groupData[sim_const.LABEL_NAME])
		agents = agentGroup.getMembers()
		for agent in agents:
			self.gui.addAgent(agent)
		
	def setupObjectGroups(self, groups):
		"""
		Handles received list of object groups.
		"""
		self.world.setupObjectGroups(groups)
		objectGroups = self.world.getObjectGroups() 
		for group in objectGroups.keys():
			self.getObjectsInGroup(group)
		
	def setupObjectGroup(self, groupData):
		"""
		Handles received data for object group.
		"""
		self.world.setupObjectGroup(groupData)
		objectGroup = self.world.getObjectGroup(groupData[sim_const.LABEL_NAME])
		simObjects = objectGroup.getMembers()
		for obj in simObjects:
			self.gui.addSimObject(obj)
		
