#!/usr/bin/env python

# tutorials.creating_clients.timeuser

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


"""
'Creating Clients' tutorial source code.

Basic user client for requesting time.

@author:	Simon Yuill
@copyright:	Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# svs imports
from svs_core.commands.scriptcommands import *
from svs_core.network.clientuser import GraphicalClient
from svs_core.network.packets import *
from svs_core.utilities.constants import svs_const

#############################
# CONSTANTS
#############################
daysOfWeek = ["Monday", 
		"Tuesday", 
		"Wednesday", 
		"Thursday", 
		"Friday", 
		"Saturday",
		"Sunday"]


#############################
# CLIENT
#############################
class TimeUser(GraphicalClient):
	"""
	Client accessing time service.
	"""
	def __init__(self, name, passwd, guiClass=None):
		GraphicalClient.__init__(self, name, passwd, guiClass=guiClass)
		self.timeService = None

	#######################
	# TIME
	#######################
	def getTime(self):
		"""
		Requests time from tiem service.
		"""
		dataRequest = makeDataRequest(self, recipient=self.timeService, label='time')
		self.getData(dataRequest)
		
	def handleTime(self, timeData):
		"""
		Displays time returned from service.
		"""
		dateStr = "date: %s, %d/%d/%d" % (daysOfWeek[timeData[6]], timeData[2], timeData[1], timeData[0])
		self.statusMessage(dateStr)
		timeStr = "time: %d:%d:%d" % (timeData[3], timeData[4], timeData[5])
		self.statusMessage(timeStr)

	def getTimeLog(self):
		"""
		Requests time log from time service.
		"""
		dataRequest = makeDataRequest(self, recipient=self.timeService, label='time_log')
		self.getData(dataRequest)

	def handleTimeLog(self, logData):
		"""
		Displays log returned from service.
		"""
		lStr = "time log:\n"
		clients = logData.keys()
		clients.sort()
		for clientname in clients:
			lStr += "  %s: %d\n" % (clientname, logData[clientname])
		self.statusMessage(lStr)

	#######################
	# CLIENTS
	#######################
	def handleClientJoined(self, data):
		"""
		Responds to new client joining the network.
		"""
		self.statusMessage("'%s' has joined '%s'." % (data['client_name'], data['client_group']))

	def handleClientDeparted(self, data):
		"""
		Responds to new client leaving the network.
		"""
		self.statusMessage("'%s' has left '%s'." % (data['client_name'], data['client_group']))
		


	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.

		This should be overidden by extending classes.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# time
		if dataLabel == 'time':
			self.handleTime(dataPacket.content)
		# timelog
		if dataLabel == 'time_log':
			self.handleTimeLog(dataPacket.content)
		# client joined network
		elif dataLabel == 'client_joined':
			self.handleClientJoined(dataPacket.content)
		# client departed network
		elif dataLabel == 'client_departed':
			self.handleClientDeparted(dataPacket.content)
		else:
			try:self.logMessage("data packet received: <%s>" % dataPacket.content)
			except:pass


	#######################
	# COMMANDS
	#######################
	def cmdprivate_time(self, command):
		"""
		Gets current time from service.
		"""
		self.getTime()
		return makeCommandResult(command, message="requesting time ...", status=svs_const.OK)

	def cmdprivate_timelog(self, command):
		"""
		Gets log of those who have requested time from service.
		"""
		self.getTimeLog()
		return makeCommandResult(command, message="requesting time log ...", status=svs_const.OK)
		


#########################
# MAIN
#########################
if __name__ == '__main__':
	import sys
	timeuser = TimeUser(sys.argv[1], sys.argv[2])
	timeuser.timeService = 'time_service'
	timeuser.connect("time_group", "localhost", 9797)
