;; spark-fix-error.el --- utilities for automatically fixing
;; errors reported by the SPARK Examiner.

;; Copyright (C) 2010, 2011  Gaétan Allaert

;; Author: Gaétan Allaert <gaetan.allaert@belgacom.net>
;; Maintainer: Gaétan Allaert <gaetan.allaert@belgacom.net>
;; Keywords: languages SPARK ada

;; This file is not part of GNU Emacs.

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.


;;; Usage:
;; - SPARK the source code
;; - Click on the error and put the cursor inside the global/derives annotation that contains the error.
;; - M-x spark-fix-error

(require 'spark-mode)

(defconst identifier-name-regexp
  "\\([a-zA-Z0-9_.']+\\)"
  "regexp to extract the identifier in error messages")

(defconst identifiers-name-regexp
  "\\([a-zA-Z0-9 _.,']+\\)"
  "regexp to extract the identifiers in error messages")

;; Fix flow error
(defun spark-fix-flow-error (state1 state2 spark-derives-fixed)
  (let ((begin nil)
	(end nil)
	(paramlist nil)
	(paramlist-fixed nil)
	(paramlist-reformatted nil))

    ;; find start of current SPARK derives annotation
    (ada-search-ignore-string-comment "derives\\>" t nil nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (forward-word 1)
    (ada-goto-next-non-ws nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (setq begin (point))

    ;; find end of the SPARK derives annotation
    (ada-search-ignore-string-comment ";" nil nil nil nil t)
    (delete-char -1)
    (insert "\n")
    (setq end (point))

    ;; build a list of all elements of the SPARK derives annotation
    (setq paramlist (spark-scan-derives begin (1- end)))

    ;; delete the original SPARK derives annotation
    (delete-region begin end)

    ;; fix the SPARK derives list
    (setq paramlist-fixed (funcall spark-derives-fixed paramlist state1 state2))

    ;; reformat the SPARK derives list
    (setq paramlist-reformatted (spark-derives-reformat paramlist-fixed))

    ;; insert the new SPARK derives list
    (goto-char begin)
    (spark-insert-derives paramlist-reformatted)
    (goto-char begin)
    (forward-word 1)))

(defun add-missing-derive-state (paramlist state missing-state)
  (let ((i (length paramlist))
	state-list
	(paramlist-fixed nil))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (setq state-list (member-ignore-case state (nth 0 (nth i paramlist))))
      (if (not state-list)
	  ;; STATE not in "derives A, B from ...;"
	  (setq paramlist-fixed (append (list (nth i paramlist))
					paramlist-fixed))
	;; STATE in "derives STATE, A from ...;"
	(if (member-ignore-case missing-state (nth 1 (nth i paramlist)))
	    ;; MISSING-STATE in "derives STATE, A from MISSING-STATE, X;"
	    (setq paramlist-fixed (append (list (nth i paramlist))
					paramlist-fixed))
	  ;; MISSING-STATE not in "derives STATE, A from X, Y;"
	  (if (eq (length (nth 0 (nth i paramlist))) 1)
	      ;; "derives STATE from X, Y;" -> "derives STATE from MISSING-STATE, X, Y;"
	      (setq paramlist-fixed (append (list (list (nth 0 (nth i paramlist)) (sort (append (nth 1 (nth i paramlist)) (list missing-state)) 'spark-sort-no-case)))
					    paramlist-fixed))
	    ;; "derives STATE, A from X, Y;" -> "derives STATE from MISSING-STATE, X, Y & 
	    ;;                                           A     from X, Y;"
	    (setq paramlist-fixed (append (list (list (remq (car state-list) (nth 0 (nth i paramlist))) (nth 1 (nth i paramlist))))
					  (list (list (list (car state-list)) (sort (append (nth 1 (nth i paramlist)) (list missing-state)) 'spark-sort-no-case)))
					  paramlist-fixed))))))
    paramlist-fixed))

(defun remove-derive-state (paramlist state removing-state)
  (let ((i (length paramlist))
	state-list
	removing-state-list
	(paramlist-fixed nil))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (setq state-list (member-ignore-case state (nth 0 (nth i paramlist))))
      (if (not state-list)
	  ;; STATE not in "derives A, B from ...;"
	  (setq paramlist-fixed (append (list (nth i paramlist))
					paramlist-fixed))
	;; STATE in "derives STATE, A from ...;"
	(setq removing-state-list (member-ignore-case removing-state (nth 1 (nth i paramlist))))
	(if (not removing-state-list)
	    ;; REMOVING-STATE not in "derives STATE, A from X, Y;"
	    (setq paramlist-fixed (append (list (nth i paramlist))
					paramlist-fixed))
	  ;; REMOVING-STATE in "derives STATE, A from REMOVING-STATE, X;"
	  (if (eq (length (nth 0 (nth i paramlist))) 1)
	      ;; "derives STATE from REMOVING-STATE, X;" -> "derives STATE from X;"
	      (setq paramlist-fixed (append (list (list (nth 0 (nth i paramlist)) (remq (car removing-state-list) (nth 1 (nth i paramlist)))))
					    paramlist-fixed))
	    ;; "derives STATE, A from REMOVING-STATE, X;" -> "derives STATE from X & 
	    ;;                                                        A     from REMOVING-STATE, X;"
	    (setq paramlist-fixed (append (list (list (remq (car state-list) (nth 0 (nth i paramlist))) (nth 1 (nth i paramlist))))
					  (list (list (list (car state-list)) (remq (car removing-state-list) (nth 1 (nth i paramlist)))))
					  paramlist-fixed))))))
    paramlist-fixed))

(defun spark-add-missing-derive-state (state missing-states)
  (let (missing-states-list)
    (if (ada-check-emacs-version 22 0)
	(setq missing-states-list (split-string missing-states ", " t))
      (setq missing-states-list (split-string missing-states ", ")))
    (while missing-states-list
      (spark-fix-flow-error state (car missing-states-list) 'add-missing-derive-state)
      (setq missing-states-list (cdr missing-states-list)))))

(defun spark-remove-derive-state (state removing-states)
  (let (removing-states-list)
    (if (ada-check-emacs-version 22 0)
	(setq removing-states-list (split-string removing-states ", " t))
      (setq removing-states-list (split-string removing-states ", ")))
    (while removing-states-list
      (spark-fix-flow-error state (car removing-states-list) 'remove-derive-state)
      (setq removing-states-list (cdr removing-states-list)))))

;; Fix semantic error
(defun spark-fix-sem-error (state spark-own-or-global-fixed spark-global-varlist)
  (let ((begin nil)
	(end nil)
	(paramlist nil)
	(paramlist-fixed nil)
	(paramlist-reformatted nil))

    ;; find start of current variable-list
    (ada-search-ignore-string-comment "\\<own\\|global\\>" t nil nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (forward-word 1)
    (ada-goto-next-non-ws nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (setq begin (point))

    ;; find end of variable-list
    (ada-search-ignore-string-comment ";" nil nil nil nil t)
    (let ((p (point)))
      (while (save-excursion (and
			      (ada-goto-next-non-ws nil nil t)
			      (spark-in-annotation-p)
			      (or (re-search-forward "protected[ \t]" (+ (point) 10) t)
				  (re-search-forward "in[ \t]" (+ (point) 3) t)
				  (re-search-forward "out[ \t]" (+ (point) 4) t)
				  (save-excursion (skip-syntax-forward "w")
						  (not (spark-after-keyword-p))))
			      (setq p (point))))
	(progn (goto-char p)
	       (ada-search-ignore-string-comment ";" nil nil nil nil t))))
    (delete-char -1)
    (insert "\n")

    ;; find end of last variable-declaration
    (setq end (point))

    ;; build a list of all elements of the variable-list
    (setq paramlist (ada-scan-paramlist begin (1- end) t t))

    ;; delete the original variable-list
    (delete-region begin end)

    ;; fix the variable-list
    (setq paramlist-fixed (funcall spark-own-or-global-fixed paramlist state))

    ;; reformat the variable-list
    (setq paramlist-reformatted
	  (if (spark-own-varlist-no-type-p paramlist)
	      (spark-own-or-global-varlist-reformat paramlist-fixed spark-global-varlist)
	    paramlist-fixed))

    ;; insert the new variable-list
    (goto-char begin)
    (ada-insert-paramlist paramlist-reformatted t t)
    (goto-char begin)
    (forward-word 1)))

(defun add-missing-global-state (paramlist state in out)
  (let ((i (length paramlist))
	state-list
	(found nil)
	(paramlist-fixed nil))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (setq state-list (eq t (compare-strings state 0 nil (nth 4 (nth i paramlist)) 0 nil t)))
      (if (not state-list)
	  ;; STATE not in "global X;"
	  (setq paramlist-fixed (append (list (nth i paramlist))
					paramlist-fixed))
	;; STATE in "global STATE;"
	(setq found t)
	(setq paramlist-fixed (append (list (list (nth 0 (nth i paramlist))
						  (or in
						      (nth 1 (nth i paramlist)))
						  (or out
						      (nth 2 (nth i paramlist)))
						  (nth 3 (nth i paramlist))
						  (nth 4 (nth i paramlist))
						  (nth 5 (nth i paramlist))))
				      paramlist-fixed))))
    (unless found
      (setq paramlist-fixed (append (list (list "" in out nil state nil))
				    paramlist-fixed)))
    paramlist-fixed))

(defun remove-global-state (paramlist state in out)
  (let ((i (length paramlist))
	state-list
	(paramlist-fixed nil))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (setq state-list (eq t (compare-strings state 0 nil (nth 4 (nth i paramlist)) 0 nil t)))
      (if (not state-list)
	  ;; STATE not in "global X;"
	  (setq paramlist-fixed (append (list (nth i paramlist))
					paramlist-fixed))
	;; STATE in "global STATE;"
	(setq paramlist-fixed (append (list (list (nth 0 (nth i paramlist))
						  (and (not in)
						       (nth 1 (nth i paramlist)))
						  (and (not out)
						       (nth 2 (nth i paramlist)))
						  (nth 3 (nth i paramlist))
						  (nth 4 (nth i paramlist))
						  (nth 5 (nth i paramlist))))
				      paramlist-fixed))))
    paramlist-fixed))

(defun add-missing-global-in-state (paramlist state)
  (add-missing-global-state paramlist state t nil))

(defun add-missing-global-out-state (paramlist state)
  (add-missing-global-state paramlist state nil t))

(defun remove-global-in-state (paramlist state)
  (remove-global-state paramlist state t nil))

(defun remove-global-out-state (paramlist state)
  (remove-global-state paramlist state nil t))

(defun spark-add-missing-global-in-state (state)
  (spark-fix-sem-error state 'add-missing-global-in-state t))

(defun spark-add-missing-global-out-state (state)
  (spark-fix-sem-error state 'add-missing-global-out-state t))

(defun spark-remove-global-in-state (state)
  (spark-fix-sem-error state 'remove-global-in-state t))

(defun spark-remove-global-out-state (state)
  (spark-fix-sem-error state 'remove-global-out-state t))

;; Fix
(defun spark-fix-one-error (source-buffer error-type)
  ;; No error       => ERROR-TYPE = 0
  ;; Semantic error => ERROR-TYPE = 1
  ;; Flow error     => ERROR-TYPE = 2
  ;;
  ;; No more error  => ERROR-TYPE = 9
  (let ((new-error-type error-type))
    (cond
     ((or (looking-at (concat "Semantic Error  *1 - The identifier " identifier-name-regexp)) ;; is either undeclared or not visible at this point.
	  (looking-at (concat "Semantic Error  *25 - The identifier " identifier-name-regexp)) ;; (imported by called subprogram) is not visible at this point.
	  (looking-at (concat "Semantic Error  *85 - " identifier-name-regexp)) ;; is not a constituent of any abstract own variable appearing in the earlier global definition for this subprogram.
	  (looking-at (concat "Semantic Error  *144 - The dotted name " identifier-name-regexp)) ;; is either undeclared or not visible at this point.
	  (looking-at (concat "Semantic Error  *752 - The identifier " identifier-name-regexp)) ;; is either undeclared or not visible at this point. This identifier must appear in a preceding legal global annotation or formal parameter list.
	  (looking-at (concat "Semantic Error  *753 - The identifier " identifier-name-regexp))) ;; is either undeclared or not visible at this point. This identifier must appear in a preceding legal global annotation or formal parameter list.
      (let ((state (match-string 1)))
	(pop-to-buffer source-buffer)
	(when (or (= error-type 0)
		  (= error-type 1))
	  (spark-add-missing-global-in-state state)
	  (setq new-error-type 1))))
     ((or (looking-at (concat "Semantic Error  *24 - The identifier " identifier-name-regexp)) ;; (exported by called subprogram) is not visible at this point.
	  (looking-at (concat "Flow Error  *2 - The updating of " identifier-name-regexp)) ;; has not been previously stated.
	  (looking-at (concat "Flow Error  *34 - The imported, non-exported variable " identifier-name-regexp)) ;; may be redefined.
	  (looking-at (concat "Semantic Error  *754 - The identifier " identifier-name-regexp))) ;; is either undeclared or not visible at this point. This identifier must appear in a preceding legal global annotation or formal parameter list.
      (let ((state (match-string 1)))
	(pop-to-buffer source-buffer)
	(when (or (= error-type 0)
		  (= error-type 1))
	  (spark-add-missing-global-out-state state)
	  (setq new-error-type 1))))
     ((looking-at (concat "Flow Error  *30 - The variable " identifier-name-regexp)) ;; is imported but neither referenced nor exported.
      (let ((state (match-string 1)))
	(pop-to-buffer source-buffer)
	(when (or (= error-type 0)
		  (= error-type 1))
	  (spark-remove-global-in-state state)
	  (setq new-error-type 1))))
     ((looking-at (concat "Flow Error  *31 - The variable " identifier-name-regexp)) ;; is exported but not (internally) defined.
      (let ((state (match-string 1)))
	(pop-to-buffer source-buffer)
	(when (or (= error-type 0)
		  (= error-type 1))
	  (spark-remove-global-out-state state)
	  (setq new-error-type 1))))
     ((or (looking-at (concat "Flow Error  *4 - The dependency of the exported value of " identifier-name-regexp " on the imported value of " identifiers-name-regexp " has not been previously stated\."))
	  (looking-at (concat "Flow Error  *601 - " identifier-name-regexp " may be derived from the imported value(s) of " identifiers-name-regexp "\.")))
      (let ((state (match-string 1))
	    (missing-states (match-string 2)))
	(pop-to-buffer source-buffer)
	(when (or (= error-type 0)
		  (= error-type 2))
	  (spark-add-missing-derive-state state missing-states)
	  (setq new-error-type 2))))
     ((or (looking-at (concat "Flow Error  *3 - The previously stated dependency of the exported value of " identifier-name-regexp " on the imported value of " identifiers-name-regexp " has been omitted\."))
	  (looking-at (concat "Flow Error  *50 - " identifier-name-regexp " is not derived from the imported value(s) of " identifiers-name-regexp "\.")))
      (let ((state (match-string 1))
	    (removing-states (match-string 2)))
	(pop-to-buffer source-buffer)
	(when (or (= error-type 0)
		  (= error-type 2))
	  (spark-remove-derive-state state removing-states)
	  (setq new-error-type 2))))
     ((looking-at "Semantic Error  *156 - Entire variable expected\.")
      (let (end)
	(pop-to-buffer source-buffer)
	(when (= error-type 0)
	  (save-excursion (setq end (search-forward "\.")))
	  (delete-region (point) end)
	  (ada-format-spark-derives)
	  (setq new-error-type 9))))
     ((looking-at (concat "Semantic Error  *337 - Package " identifier-name-regexp " may not be used as a prefix here\."))
      (let ((state (match-string 1))
	    end)
	(pop-to-buffer source-buffer)
	(when (= error-type 0)
	  (setq begin (point))
	  (looking-at (concat state "\."))
	  (delete-region begin (match-end 0))
	  (setq new-error-type 9))))
     ((looking-at (concat "Warning  *380 - Casing inconsistent with declaration. Expected casing is " identifier-name-regexp "\."))
      (let ((correct (match-string 1))
	    begin)
	(pop-to-buffer source-buffer)
	(when (= error-type 0)
	  (setq begin (point))
	  (looking-at correct)
	  (delete-region begin (match-end 0))
	  (insert correct)
	  (setq new-error-type 9))))
     ((or (looking-at (concat "Warning  *430 - SLI generation abandoned owing to syntax or semantic errors or multiple units in a single source file\."))
	  (looking-at (concat "Warning  *399 - The called subprogram has semantic errors in its interface (parameters and/or annotations) which prevent flow analysis of this call\.")))
      (pop-to-buffer source-buffer)))
    new-error-type))

(defun spark-fix-error ()
  (interactive)

  (let ((source-buffer (current-buffer))
	(source-buffer-start (point))
	(source-window (selected-window))
        compilation-buffer-start
	fix-grouped-errors
	(error-type 0))

    ;; Goto the error message. Sometimes (the very first time the
    ;; compilation buffer is used?) point is not at the right place in
    ;; the compilation buffer. This sequence seems to fix it.
    (pop-to-buffer compilation-last-buffer)
    (pop-to-buffer source-buffer)
    (set-buffer compilation-last-buffer)
    (if (not (eq source-window (selected-window)))
        (error "two windows open on source buffer; please close one"))

    ;; Save starting point in compilation buffer, in case we need to
    ;; repeat operation. We don't use save-excursion, because we want
    ;; point to change in source-buffer
    (setq compilation-buffer-start (point))
    (if (or (search-forward "Semantic Error " (line-end-position) t)
	    (search-forward "Flow Error " (line-end-position) t)
	    (search-forward "Warning " (line-end-position) t))
	(progn (beginning-of-line)
	       (looking-at "[^:]*:[1-9][0-9]*:[1-9][0-9]*: *")
	       (setq fix-grouped-errors (match-string 0))
	       (beginning-of-line))
      (error "spark-fix-error: error not found from point %d" (point)))
    (while (looking-at fix-grouped-errors)
      (goto-char (match-end 0))
      (setq error-type (spark-fix-one-error source-buffer error-type))
      (goto-char source-buffer-start)
      (pop-to-buffer compilation-last-buffer)
      (forward-line))
    (pop-to-buffer source-buffer)))

(provide 'spark-fix-error)
