;;; spark-mode.el --- addon to the ada-mode for editing SPARK Ada
;;; sources

;; Copyright (C) 2010, 2011  Gaétan Allaert

;; Author: Gaétan Allaert <gaetan.allaert@belgacom.net>
;; Maintainer: Gaétan Allaert <gaetan.allaert@belgacom.net>
;; Keywords: languages SPARK ada

;; This file is not part of GNU Emacs.

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:
;; This file contains the code for editing SPARK Ada source code. It
;; requires the full ada-mode for GNU Emacs and modifications in the
;; ada-mode.el and in the ada-xref.el.

;;; Usage:
;; The SPARK Ada mode provides the following functionalities:
;; - TAB                              : indent SPARK annotations;
;; - ENTER                            : add the SPARK annotation "--#" when you insert a newline inside the SPARK annotations +
;;                                      indent the new line;
;; - TYPE                             : adjust casing for identifier in SPARK annotations;
;; - ESC-/ or META-/                  : auto-completion in SPARK annotation;
;; - C-c C-f                          : reformat proof functions;
;; - C-c C-i                          : reformat/sort inherit annotations;
;; - C-c C-g                          : reformat/sort own/global annotations;
;; - C-c C-h                          : reformat/compress/sort derives annotation;
;; - Ada -> SPARK -> SPARK current file
;;   (spark-current-file)             : SPARK the current file;
;; - Ada -> SPARK -> SPARK metafile
;;   (spark-metafile)                 : SPARK the metafile;
;; - Ada -> SPARK -> SPARK/Proof current body
;;   (spark-proof-current-body)       : SPARK and Simplify the current body (without the nested subprograms) +
;;                                      open the related SIV file;
;; - Ada -> SPARK -> SPARK/Proof current subprogram
;;   (spark-proof-current-subprogram) : SPARK, Simplify and POGS the current subprogram (with the nested subprograms) +
;;                                      open the related SIV file and SUM file if nested subprograms;
;; - Ada -> SPARK -> SPARK/Proof current file
;;   (spark-proof-current-file)       : SPARK, Simplify and POGS the current file +
;;                                      open the top level SUM file;
;; - Ada -> SPARK -> SPARK/Proof metafile
;;   (spark-proof-metafile)           : SPARK, Simplify and POGS the metafile +
;;                                      open the top level SUM file;
;; - Ada -> SPARK -> Proof all
;;   (spark-proof)                    : Simplify all and POGS +
;;                                      open the top level SUM file;
;; - (spark-fix-error)                : automatic fix of SPARK semantic, flow and capitalisation errors;
;; - C-c C-d                          : "Goto Declaration/Body" cross navigation in SPARK annotations;
;; - (ada-find-any-references)        : "List References" cross navigation in SPARK annotations using gnatfind or gps;
;; - AUTO                             : syntax highlighting of SPARK keywords in annotation;
;; - (ada-rename-identifier)          : renames identifier in Ada source code and in SPARK annotations;
;; - (spark-pretty-print)             : pretty print the SPARK source code;
;; - (ada-format-call-paramlist)      : reformat subprogram call or aggregate.

(defcustom ada-fill-spark-annotation "#"
  "*Text inserted in the first columns when filling a SPARK annotation paragraph.
Note: if you modify this variable, you will have to invoke `ada-mode'
again to take account of the new value."
  :type 'string :group 'ada)

(defcustom ada-spark-examiner-command "spark"
  "*Name of the SPARK Examiner and the options to use."
  :type 'string :group 'ada)

(defcustom ada-spark-sparksimp-command "sparksimp -t"
  "*Name of the SPARKSimp and the options to use.
Don't provide the '-p=' option."
  :type 'string :group 'ada)

(defcustom ada-spark-simplifier-command "spadesimp"
  "*Name of the Simplifier and the options to use."
  :type 'string :group 'ada)

(defcustom ada-spark-pogs-command "pogs"
  "*Name of the POGS and the options to use."
  :type 'string :group 'ada)

(defcustom ada-spark-mode nil
  "*Set SPARK Ada mode."
  :type 'boolean :group 'ada)

(defcustom ada-spark-number-of-jobs 2
  "*Specifies the number of jobs (commands) to run simultaneously."
  :type 'integer :group 'ada)

(defvar ada-mode-spark-syntax-table nil
  "Syntax table for SPARK, where `--#' is not a comment.")

(defsubst spark-in-annotation-p (&optional parse-result)
  "Return t if inside a SPARK annotation.
If PARSE-RESULT is non-nil, use it instead of calling `parse-partial-sexp'."
  (if (and ada-spark-mode
	   (ada-in-comment-p parse-result))
      (save-excursion
	(goto-char (nth 8 (or parse-result
			      (parse-partial-sexp
			       (line-beginning-position) (point)))))
	(looking-at ada-spark-annotation-start))
    nil))

(defsubst spark-in-annotation-not-in-string-p (&optional parse-result)
  "Return t if inside a SPARK annotation but not in a string inside the SPARK annotation.
If PARSE-RESULT is non-nil, use it instead of calling `parse-partial-sexp'."
  (if (spark-in-annotation-p parse-result)
      (let ((previous-syntax-table (syntax-table)))
	(unwind-protect
	    (progn
	      (set-syntax-table ada-mode-spark-syntax-table)
	      (not (ada-in-string-p)))
	  (set-syntax-table previous-syntax-table)))
    nil))

(defun spark-after-keyword-p (&optional parse-result)
  "Return t if cursor is after a SPARK annotation keyword that is not an attribute."
  (if (spark-in-annotation-p parse-result)
      (save-excursion
	(forward-word -1)
	(and (not (and (char-before)
		       (or (= (char-before) ?_)
			   (= (char-before) ?'))));; unless we have a _ or '
	     (looking-at (concat ada-spark-keywords "[^_]"))))
    nil))

(defun spark-sort-no-case (el1 el2)
  "Sort SPARK variable list casing is ignored."
  (string< (upcase el1) (upcase el2)))

;;-----------------------------
;;-- SPARK format own/global --
;;-----------------------------
(defun spark-in-global-varlist-p ()
  "Return t if point is inside a SPARK global variable-list."
  (if (spark-in-annotation-p)
      (let ((orgpoint (point)))
	(save-excursion
	  (ada-goto-stmt-start nil t)
	  (re-search-forward "\\<global\\>[ \t\n]" orgpoint t)))
    nil))

(defun spark-in-own-p ()
  "Return t if point is inside a SPARK own variable-list."
  (if (spark-in-annotation-p)
      (let ((orgpoint (point)))
	(save-excursion
	  (ada-goto-stmt-start nil t)
	  (while (save-excursion
		   (backward-word 1)
		   (looking-at "\\<is\\>[ \t\n]"))
	    (backward-word 1)
	    (ada-goto-stmt-start nil t))
	  (re-search-forward "\\<own\\>[ \t\n]" orgpoint t)))
    nil))

(defun spark-in-own-varlist-p ()
  "Return t if point is inside a SPARK own variable-list but not in a refinement."
  (if (spark-in-annotation-p)
      (let ((orgpoint (point))
	    endpoint)
	(save-excursion
	  (ada-goto-stmt-start nil t)
	  (if (re-search-forward "\\<own\\>[ \t\n]" orgpoint t)
	      (progn (setq endpoint (save-excursion (search-forward ";") (point)))
		     (not (re-search-forward "is[ \t\n]" endpoint t)))
	    nil)))
    nil))

(defun spark-own-varlist-no-type-p (varlist)
  "Check if SPARK own variable list contains types."
  (let ((i (length varlist))
	(type-found nil))
    ;; loop until last parameter or type is found
    (while (not (or (zerop i) type-found))
      (setq i (1- i))
      (setq type-found (nth 5 (nth i varlist))))
    (not type-found)))

(defun spark-sort-own-refinement (el1 el2)
  "Sort SPARK own/global variable list."
  (string< (upcase (nth 0 el1)) (upcase (nth 0 el2))))

(defun spark-format-own-refinement ()
  "Reformat SPARK own refinement annotation point is in."
  (let ((notend t)
	i
	j
	j-in
	j-out
	total-j
	in-out
	(match-cons nil)
	(begin nil)
	(end nil)
	(apos nil)
	(epos nil)
	(semipos nil)
	own-state
	(own-list nil)
	(own-list-in nil)
	(own-list-out nil)
	(paramlist (list))
	(paramlist-reformatted nil)
	paramlist-len
	own-list-len)

    ;; find start of current SPARK own annotation
    (ada-search-ignore-string-comment "own\\>" t nil nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (forward-word 1) ;; skip OWN
    (ada-goto-next-non-ws nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (setq begin (point))

    ;; find end of the SPARK own annotation
    (ada-search-ignore-string-comment ";" nil nil nil nil t)
    (delete-char -1)
    (insert "\n")
    (setq end (point))

    ;; build a list of all elements of the SPARK own annotation
    (goto-char begin)

    ;; loop until end of the SPARK derives annotation
    (while notend

      ;; find first character of parameter-declaration
      (ada-goto-next-non-ws nil nil t)
      (setq apos (point))

      (if (setq match-cons
		(ada-search-ignore-string-comment "[ \t\n]*[&;]" nil end nil nil t))
	  (progn
	    (setq epos (car match-cons))
	    (setq semipos (cdr match-cons)))
	(setq epos end))

      (goto-char apos)

      (looking-at "\\(\\sw\\|[_.\\*]\\)+")
      (setq own-state (match-string 0))
      (goto-char (match-end 0)) ;; skip OWN-STATE
      (ada-goto-next-non-ws nil nil t)
      (forward-word 1) ;; skip IS
      (ada-goto-next-non-ws nil nil t)

      (setq own-list nil)
      (setq own-list-in nil)
      (setq own-list-out nil)
      (while (progn (ada-goto-next-non-ws nil nil t)
		    (< (point) epos))
	(setq in-out nil)
	(when (looking-at "in[ \t][ \t]*")
	  (goto-char (match-end 0)) ;; skip IN
	  (looking-at "\\(\\sw\\|[_.\\*]\\)+")
	  (setq own-list-in (append own-list-in
				    (list (match-string 0))))
	  (setq in-out t))
	(when (looking-at "out[ \t][ \t]*")
	  (goto-char (match-end 0)) ;; skip OUT
	  (looking-at "\\(\\sw\\|[_.\\*]\\)+")
	  (setq own-list-out (append own-list-out
				     (list (match-string 0))))
	  (setq in-out t))
	(unless in-out
	  (looking-at "\\(\\sw\\|[_.\\*]\\)+")
	  (setq own-list (append own-list 
				 (list (match-string 0)))))
	(ada-search-ignore-string-comment "[ \t\n]*[,&]" nil epos nil nil t))

      ;; add this parameter-declaration to the list
      (setq paramlist (append paramlist (list (list own-state
						    (sort own-list     'spark-sort-no-case)
						    (sort own-list-in  'spark-sort-no-case)
						    (sort own-list-out 'spark-sort-no-case)))))

      ;; check if it was the last parameter
      (if (eq epos end)
	  (setq notend nil)
	(goto-char semipos)))

    ;; delete the original SPARK own annotation
    (delete-region begin end)

    ;; reformat the SPARK own list
    (setq paramlist-reformatted (sort paramlist 'spark-sort-own-refinement))

    ;; insert the new SPARK own list
    (goto-char begin)

    (setq i (length paramlist-reformatted))
    (setq paramlist-len (1- i))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))

      (insert (nth 0 (nth (- paramlist-len i) paramlist-reformatted)))
      (insert " is ")

      (setq j     (length (nth 1 (nth (- paramlist-len i) paramlist-reformatted))))
      (setq j-in  (length (nth 2 (nth (- paramlist-len i) paramlist-reformatted))))
      (setq j-out (length (nth 3 (nth (- paramlist-len i) paramlist-reformatted))))
      (setq total-j (+ j j-in j-out))

      (setq own-list-len (1- j))
      ;; loop until last parameter
      (while (not (zerop j))
	(setq j (1- j))
	(setq total-j (1- total-j))
	(insert (nth (- own-list-len j) (nth 1 (nth (- paramlist-len i) paramlist-reformatted))))
	(if (zerop total-j)
	    (if (zerop i)
		(insert ";")
	      (insert " &")
	      (newline)
	      (insert ada-spark-annotation-start)
	      (backward-char 3)
	      (ada-indent-current)
	      (forward-char 3)
	      (ada-indent-current))
	  (insert ",")
	  (newline)
	  (insert ada-spark-annotation-start)
	  (backward-char 3)
	  (ada-indent-current)
	  (forward-char 3)
	  (ada-indent-current)))

      (setq own-list-len (1- j-in))
      ;; loop until last parameter
      (while (not (zerop j-in))
	(setq j-in (1- j-in))
	(setq total-j (1- total-j))
	(insert "in ")
	(insert (nth (- own-list-len j-in) (nth 2 (nth (- paramlist-len i) paramlist-reformatted))))
	(if (zerop total-j)
	    (if (zerop i)
		(insert ";")
	      (insert " &")
	      (newline)
	      (insert ada-spark-annotation-start)
	      (backward-char 3)
	      (ada-indent-current)
	      (forward-char 3)
	      (ada-indent-current))
	  (insert ",")
	  (newline)
	  (insert ada-spark-annotation-start)
	  (backward-char 3)
	  (ada-indent-current)
	  (forward-char 3)
	  (ada-indent-current)))

      (setq own-list-len (1- j-out))
      ;; loop until last parameter
      (while (not (zerop j-out))
	(setq j-out (1- j-out))
	(setq total-j (1- total-j))
	(insert "out ")
	(insert (nth (- own-list-len j-out) (nth 3 (nth (- paramlist-len i) paramlist-reformatted))))
	(if (zerop total-j)
	    (if (zerop i)
		(insert ";")
	      (insert " &")
	      (newline)
	      (insert ada-spark-annotation-start)
	      (backward-char 3)
	      (ada-indent-current)
	      (forward-char 3)
	      (ada-indent-current))
	  (insert ",")
	  (newline)
	  (insert ada-spark-annotation-start)
	  (backward-char 3)
	  (ada-indent-current)
	  (forward-char 3)
	  (ada-indent-current))))))

(defun spark-sort-own-or-global-varlist (el1 el2)
  "Sort SPARK own/global variable list."
  (string< (upcase (nth 4 el1)) (upcase (nth 4 el2))))

(defun spark-own-or-global-varlist-reformat (varlist spark-global-varlist)
  "Reformat the SPARK own/global variable list point is in."
  (let ((varlist-in nil)
	(varlist-in-sorted nil)
	(varlist-out nil)
	(varlist-out-sorted nil)
	(varlist-in-out nil)
	(varlist-in-out-sorted nil)
	(varlist-protected nil)
	(varlist-protected-sorted nil)
	(varlist-unmoded nil)
	(varlist-unmoded-sorted nil)
	(previous-varlist nil)
	(i (length varlist)))
    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (cond
       ((and (nth 1 (nth i varlist))
	     (not (nth 2 (nth i varlist))))
	(setq previous-varlist (if spark-global-varlist 'varlist-in nil))
	(setq varlist-in (append varlist-in (list (list "" t nil nil (nth 4 (nth i varlist)) nil)))))

       ((and (not (nth 1 (nth i varlist)))
	     (nth 2 (nth i varlist)))
	(setq previous-varlist (if spark-global-varlist 'varlist-out nil))
	(setq varlist-out (append varlist-out (list (list "" nil t nil (nth 4 (nth i varlist)) nil)))))

       ((and (nth 1 (nth i varlist))
	     (nth 2 (nth i varlist)))
	(setq previous-varlist (if spark-global-varlist 'varlist-in-out nil))
	(setq varlist-in-out (append varlist-in-out (list (list "" t t nil (nth 4 (nth i varlist)) nil)))))

       ((nth 3 (nth i varlist))
	(setq previous-varlist (if spark-global-varlist 'varlist-protected nil))
	(setq varlist-protected (append varlist-protected (list (list "" nil nil t (nth 4 (nth i varlist)) nil)))))

       ((and (not (nth 1 (nth i varlist)))
	     (not (nth 2 (nth i varlist)))
	     (not (nth 3 (nth i varlist))))
	(cond
	 ((equal previous-varlist 'varlist-in)
	  (setq varlist-in (append varlist-in (list (list "" t nil nil (nth 4 (nth i varlist)) nil)))))

	 ((equal previous-varlist 'varlist-out)
	  (setq varlist-out (append varlist-out (list (list "" nil t nil (nth 4 (nth i varlist)) nil)))))

	 ((equal previous-varlist 'varlist-in-out)
	  (setq varlist-in-out (append varlist-in-out (list (list "" t t nil (nth 4 (nth i varlist)) nil)))))

	 ((equal previous-varlist 'varlist-protected)
	  (setq varlist-protected (append varlist-protected (list (list "" nil nil t (nth 4 (nth i varlist)) nil)))))

	 ((not previous-varlist)
	  (setq varlist-unmoded (append varlist-unmoded (list (list "" nil nil nil (nth 4 (nth i varlist)) nil)))))))))

    (setq varlist-in-sorted        (sort varlist-in        'spark-sort-own-or-global-varlist))
    (setq varlist-out-sorted       (sort varlist-out       'spark-sort-own-or-global-varlist))
    (setq varlist-in-out-sorted    (sort varlist-in-out    'spark-sort-own-or-global-varlist))
    (setq varlist-protected-sorted (sort varlist-protected 'spark-sort-own-or-global-varlist))
    (setq varlist-unmoded-sorted   (sort varlist-unmoded   'spark-sort-own-or-global-varlist))
    (append (if varlist-unmoded-sorted
		(reverse varlist-unmoded-sorted))
	    (append (if varlist-protected-sorted
			(reverse (append (list (list "" nil nil t (nth 4 (car varlist-protected-sorted)) nil))
					 (cdr varlist-protected-sorted))))
		    (append (if varlist-out-sorted
				(reverse (append (list (list "" nil t nil (nth 4 (car varlist-out-sorted)) nil))
						 (cdr varlist-out-sorted))))
			    (append (if varlist-in-out-sorted
					(reverse (append (list (list "" t t nil (nth 4 (car varlist-in-out-sorted)) nil))
							 (cdr varlist-in-out-sorted))))
				    (if varlist-in-sorted
					(reverse (append (list (list "" t nil nil (nth 4 (car varlist-in-sorted))  nil))
							 (cdr varlist-in-sorted))))))))))

(defun spark-format-own-or-global ()
  "Reformat the SPARK own/global variable list point is in."
  (interactive)
  (let ((begin nil)
	(end nil)
	(paramlist nil)
	(paramlist-reformatted nil)
	spark-global-varlist
	(previous-syntax-table (syntax-table)))

    (setq spark-global-varlist (spark-in-global-varlist-p))

    ;; check if really inside a own/global variable list
    (or (spark-in-own-p)
	spark-global-varlist
	(error "Not in SPARK own/global variable list"))

    (if (or (spark-in-own-varlist-p)
	    spark-global-varlist)
	(unwind-protect
	    (progn
	      (set-syntax-table ada-mode-symbol-syntax-table)

	      ;; find start of current variable-list
	      (ada-search-ignore-string-comment "\\<own\\|global\\>" t nil nil nil t)
	      (save-excursion (beginning-of-line)
			      (ada-indent-current))
	      (forward-word 1)
	      (ada-goto-next-non-ws nil nil t)
	      (save-excursion (beginning-of-line)
			      (ada-indent-current))
	      (setq begin (point))

	      ;; find end of variable-list
	      (ada-search-ignore-string-comment ";" nil nil nil nil t)
	      (let ((p (point)))
		(while (save-excursion (and
					(ada-goto-next-non-ws nil nil t)
					(spark-in-annotation-p)
					(or (re-search-forward "protected[ \t]" (+ (point) 10) t)
					    (re-search-forward "in[ \t]" (+ (point) 3) t)
					    (re-search-forward "out[ \t]" (+ (point) 4) t)
					    (save-excursion (skip-syntax-forward "w")
							    (not (spark-after-keyword-p))))
					(setq p (point))))
		  (progn (goto-char p)
			 (ada-search-ignore-string-comment ";" nil nil nil nil t))))
	      (delete-char -1)
	      (insert "\n")

	      ;; find end of last variable-declaration
	      (setq end (point))

	      ;; build a list of all elements of the variable-list
	      (setq paramlist (ada-scan-paramlist begin (1- end) t t))

	      ;; delete the original variable-list
	      (delete-region begin end)

	      ;; reformat the variable-list
	      (setq paramlist-reformatted
		    (if (spark-own-varlist-no-type-p paramlist)
			(spark-own-or-global-varlist-reformat paramlist spark-global-varlist)
		      paramlist))

	      ;; insert the new variable-list
	      (goto-char begin)
	      (ada-insert-paramlist paramlist-reformatted t t))

	  ;; restore syntax-table
	  (set-syntax-table previous-syntax-table))
      (spark-format-own-refinement))))

;;--------------------------
;;-- SPARK format derives --
;;--------------------------

;; Compare A and B (involve A, B, L3 with A not in L3 and B not in L3)

;;      derives A  from *           & B  from *               => derives A, B   from *
;;                                    B  from A               => derives A, B   from A
;;                                    B  from *, A            => derives A, B   from *, A
;;
;;      derives A  from B           & B  from *               => derives A, B   from B
;;
;;      derives A  from L3          & B  from L3              => derives A, B   from L3
;;
;;      derives A  from *, B        & B  from *               => derives A, B   from *, B
;;                                    B  from *, A            => derives A, B   from A, B
;;
;;      derives A  from *, L3       & B  from *, L3           => derives A, B   from *, L3
;;                                    B  from A, L3           => derives A, B   from A, L3
;;                                    B  from *, A, L3        => derives A, B   from *, A, L3
;;
;;      derives A  from B, L3       & B  from *, L3           => derives A, B   from B, L3
;;
;;      derives A  from *, B, L3    & B  from *, L3           => derives A, B   from *, B, L3
;;                                    B  from *, A, L3        => derives A, B   from A, B, L3

;; Compare A and L1 (involve A, L1 and L3 with A not in L1, A not in L3 and L1 may be equal to L3)

;; [01] derives A  from *           & L1 from *               => derives A, L1  from *
;; [02]                               L1 from A               => derives A, L1  from A
;; [03]                               L1 from *, A            => derives A, L1  from *, A
;;
;; [04] derives A  from L1          & L1 from L1              => derives A, L1  from L1 (derives A from L3 & L1 from L3 => derives A, L1 from L3 with L1 = L3)
;; [05]                               L1 from *, L1           => derives A, L1  from L1
;;
;; [06] derives A  from L3          & L1 from L3              => derives A, L1  from L3
;;
;; [07] derives A  from *, L1       & L1 from L1              => derives A, L1  from *, L1
;; [08]                               L1 from *, L1           => derives A, L1  from *, L1 (derives A from *, L3 & L1 from *, L3 => derives A, L1 from *, L3 with L1 = L3)
;; [09]                               L1 from A, L1           => derives A, L1  from A, L1 (derives A from *, L3 & L1 from A, L3 => derives A, L1 from A, L3 with L1 = L3)
;; [10]                               L1 from *, A, L1        => derives A, L1  from A, L1
;;
;; [11] derives A  from *, L3       & L1 from *, L3           => derives A, L1  from *, L3
;; [12]                               L1 from A, L3           => derives A, L1  from A, L3
;; [13]                               L1 from *, A, L3        => derives A, L1  from *, A, L3
;;
;; [14] derives A from L1, L3       & L1 from L1, L3          => derives A, L1  from L1, L3 (derives A from L3 & L1 from L3 => derives A, L1 from L3 with L1 = L3)
;; [15]                               L1 from *, L1, L3       => derives A, L1  from L1, L3
;;
;; [16] derives A  from *, L1, L3   & L1 from L1, L3          => derives A, L1  from *, L1, L3
;; [17]                               L1 from *, L1, L3       => derives A, L1  from *, L1, L3 (derives A from *, L3 & L1 from *, L3 => derives A, L1 from *, L3 with L1 = L3)
;; [18]                               L1 from A, L1, L3       => derives A, L1  from A, L1, L3 (derives A from *, L3 & L1 from A, L3 => derives A, L1 from A, L3 with L1 = L3)
;; [19]                               L1 from *, A, L1, L3    => derives A, L1  from A, L1, L3

;; Compare L1 and A (involve A, L1 and L3 with A not in L1, A not in L3 and L1 may be equal to L3)

;; [01] derives L1 from *            & A  from *              => derives A, L1  from *
;;
;; [02] derives L1 from A            & A  from *              => derives A, L1  from A
;;
;; [04] derives L1 from L1           & A  from L1             => derives A, L1  from L1
;; [07]                                A  from *, L1          => derives A, L1  from *, L1
;;
;; [06] derives L1 from L3           & A  from L3             => derives A, L1  from L3
;;
;; [03] derives L1 from *, A         & A  from *              => derives A, L1  from *, A
;;
;; [05] derives L1 from *, L1        & A  from L1             => derives A, L1  from L1
;; [08]                                A  from *, L1          => derives A, L1  from *, L1
;;
;; [11] derives L1 from *, L3        & A  from *, L3          => derives A, L1  from *, L3
;;
;; [09] derives L1 from A, L1        & A  from *, L1          => derives A, L1  from A, L1
;;
;; [12] derives L1 from A, L3        & A  from *, L3          => derives A, L1  from A, L3
;;
;; [14] derives L1 from L1, L3       & A  from L1, L3         => derives A, L1  from L1, L3
;; [16]                                A  from *, L1, L3      => derives A, L1  from *, L1, L3
;;
;; [10] derives L1 from *, A, L1     & A  from *, L1          => derives A, L1  from A, L1
;;
;; [13] derives L1 from *, A, L3     & A  from *, L3          => derives A, L1  from *, A, L3
;;
;; [15] derives L1 from *, L1, L3    & A  from L1, L3         => derives A, L1  from L1, L3
;; [17]                                A  from *, L1, L3      => derives A, L1  from *, L1, L3
;;
;; [18] derives L1 from A, L1, L3    & A  from *, L1, L3      => derives A, L1  from A, L1, L3
;;
;; [19] derives L1 from *, A, L1, L3 & A  from *, L1, L3      => derives A, L1  from A, L1, L3

;; Compare L1 and L2 (involve L1, L2 and L3)

;;      derives L1 from *             & L2 from *             => derives L1, L2 from *
;;
;;      derives L1 from L1            & L2 from L1            => derives L1, L2 from L1 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L1 = L3)
;;                                      L2 from *, L1         => derives L1, L2 from *, L1
;;
;;      derives L1 from L2            & L2 from L2            => derives L1, L2 from L2 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L2 = L3)
;;                                      L2 from *, L2         => derives L1, L2 from L2
;;
;;      derives L1 from L3            & L2 from L3            => derives L1, L2 from L3
;;
;;      derives L1 from *, L1         & L2 from L1            => derives L1, L2 from L1
;;                                      L2 from *, L1         => derives L1, L2 from *, L1 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L1 = L3)
;;
;;      derives L1 from *, L2         & L2 from L2            => derives L1, L2 from *, L2
;;                                      L2 from *, L2         => derives L1, L2 from *, L2 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L2 = L3)
;;
;;      derives L1 from *, L3         & L2 from *, L3         => derives L1, L2 from *, L3
;;
;;      derives L1 from L1, L2        & L2 from L1, L2        => derives L1, L2 from L1, L2
;;                                      L2 from *, L1, L2     => derives L1, L2 from L1, L2
;;
;;      derives L1 from L1, L3        & L2 from L1, L3        => derives L1, L2 from L1, L3 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L1 = L3)
;;                                      L2 from *, L1, L3     => derives L1, L2 from *, L1, L3
;;
;;      derives L1 from L2, L3        & L2 from L2, L3        => derives L1, L2 from L2, L3 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L2 = L3)
;;                                      L2 from *, L2, L3     => derives L1, L2 from L2, L3
;;
;;      derives L1 from *, L1, L2     & L2 from L1, L2        => derives L1, L2 from L1, L2
;;                                      L2 from *, L1, L2     => derives L1, L2 from L1, L2
;;
;;      derives L1 from *, L1, L3     & L2 from L1, L3        => derives L1, L2 from L1, L3
;;                                      L2 from *, L1, L3     => derives L1, L2 from *, L1, L3 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L1 = L3)
;;
;;      derives L1 from *, L2, L3     & L2 from L2, L3        => derives L1, L2 from *, L2, L3
;;                                      L2 from *, L2, L3     => derives L1, L2 from *, L2, L3 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L2 = L3)
;;
;;      derives L1 from L1, L2, L3    & L2 from L1, L2, L3    => derives L1, L2 from L1, L2, L3
;;                                      L2 from *, L1, L2, L3 => derives L1, L2 from L1, L2, L3
;;
;;      derives L1 from *, L1, L2, L3 & L2 from L1, L2, L3    => derives L1, L2 from L1, L2, L3
;;                                      L2 from *, L1, L2, L3 => derives L1, L2 from L1, L2, L3

(defun spark-sort-derives-varlist (el1 el2)
  "Sort SPARK derives variable list."
  (cond
   ((string= (upcase (nth 0 (nth 0 el1))) "NULL")
    nil)
   ((string= (upcase (nth 0 (nth 0 el2))) "NULL")
    t)
   (t
    (string< (upcase (nth 0 (nth 0 el1))) (upcase (nth 0 (nth 0 el2)))))))

(defun spark-in-spark-derives-p ()
  "Return t if point is inside a SPARK derives annotation."
  (if (spark-in-annotation-p)
      (let ((orgpoint (point)))
	(save-excursion
	  (ada-goto-stmt-start nil t)
	  (or (re-search-forward "derives\\>[ \t\n]" orgpoint t)
	      (save-excursion
		(backward-word 1)
		(re-search-forward "from\\>[ \t\n]" orgpoint t)))))
    nil))

(defun spark-compare-string-list (list1 list2)
  "Return TRUE if string list LIST1 and string list LIST2 are equal.
The comparison is not case sensitive."
  (let ((i (length list1))
	(j (length list2))
	(is-equal 't))
    (if (not (= i j))
	(setq is-equal nil)
      (while (and (not (zerop i))
		  is-equal)
	(setq i (1- i))
	(unless (eq t (compare-strings (nth i list1) 0 nil (nth i list2) 0 nil t))
	  (setq is-equal nil))))
    is-equal))

(defun spark-string-search (var list)
  "Return TRUE if the string VAR is in the list of string LIST."
  (not (eq (member-ignore-case var list) nil)))

(defun spark-delete-string (var list)
  "Delete all occurences of the string VAR in the list of string LIST."
  (let ((result-list nil)
	(i (length list)))
    (while (not (zerop i))
      (setq i (1- i))
      (if (not (string= (upcase var) (upcase (nth i list))))
	  (setq result-list (append (list (nth i list)) result-list))))
    result-list))

(defun spark-include (varlist1 varlist2)
  "Return TRUE if VARLIST1 is included in VARLIST2."
  (let ((i (length varlist1))
	(include 't))
    (while (and (not (zerop i))
		include)
      (setq i (1- i))
      (if (not (spark-string-search (nth i varlist1) varlist2))
	  (setq include nil)))
    include))

(defun spark-non-canonical-format (varlist from-varlist)
  "Return a list of all the equivalent lists of FROM-VARLIST.
Assuming that element A is not in list L.
Assuming that list L1 is not part of list L2.

derives A from L

derives A from *, A, L <=>
derives A from A, L    <=>
derives A from *, L

derives L1 from L2

derives L1 from L1, L2 <=>
derives L1 from *, L1, L2"
  (if from-varlist
      (if (= (length varlist) 1)
	  (cond
	   ((and (not (spark-string-search (nth 0 varlist) from-varlist))
		 (not (spark-string-search "*" from-varlist)))
	    (list from-varlist))
	   ((and (not (spark-string-search (nth 0 varlist) from-varlist))
		 (spark-string-search "*" from-varlist))
	    (list (sort (copy-sequence (append (list (nth 0 varlist)) from-varlist)) 'spark-sort-no-case)
		  (sort (copy-sequence (append (list (nth 0 varlist)) (spark-delete-string "*" from-varlist))) 'spark-sort-no-case)
		  from-varlist))
	   ((and (spark-string-search (nth 0 varlist) from-varlist)
		 (not (spark-string-search "*" from-varlist)))
	    (list (sort (copy-sequence (append (list "*") from-varlist)) 'spark-sort-no-case)
		  from-varlist
		  (sort (copy-sequence (append (list "*") (spark-delete-string (nth 0 varlist) from-varlist))) 'spark-sort-no-case)))
	   ((and (spark-string-search (nth 0 varlist) from-varlist)
		 (spark-string-search "*" from-varlist))
	    (list from-varlist
		  (sort (copy-sequence (spark-delete-string "*" from-varlist)) 'spark-sort-no-case)
		  (sort (copy-sequence (spark-delete-string (nth 0 varlist) from-varlist)) 'spark-sort-no-case))))
	(if (spark-include varlist from-varlist)
	    (cond
	     ((spark-string-search "*" from-varlist)
	      (list from-varlist
		    (spark-delete-string "*" from-varlist)))
	     ((not (spark-string-search "*" from-varlist))
	      (list (sort (copy-sequence (append (list "*") from-varlist)) 'spark-sort-no-case)
		    from-varlist)))
	  (list from-varlist)))
    nil))

(defun spark-compare-from-varlist (varlist1 varlist2 from-varlist1 from-varlist2)
  "Compare 2 lists of variables to check if they are compatible.
Return 'nil' if they are not compatible or
the new lists of variables if they are compatible."
  (let ((from-varlist1-non-canonical (spark-non-canonical-format varlist1 from-varlist1))
	(from-varlist2-non-canonical (spark-non-canonical-format varlist2 from-varlist2))
	i
	j
	(found nil))
    (setq i (length from-varlist1-non-canonical))
    (while (and (not (zerop i))
		(not found))
      (setq i (1- i))
      (setq j (length from-varlist2-non-canonical))
      (while (and (not (zerop j))
		  (not found))
	(setq j (1- j))
	(if (spark-compare-string-list (nth i from-varlist1-non-canonical) (nth j from-varlist2-non-canonical))
	    (setq found (nth i from-varlist1-non-canonical)))))
    found))

(defun spark-derives-reformat (paramlist)
  "Reorganize and refactorize the lists of variables."
  (let ((i (length paramlist))
	j
	from-varlist
	continue
	(paramlist-reformatted nil))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))

      (setq j (length paramlist-reformatted))
      (setq continue 't)
      ;; loop until last parameter
      (while (and (not (zerop j))
		  continue)
	(setq j (1- j))
	(if (and (not (nth 1 (nth i paramlist)))
		 (not (nth 1 (nth j paramlist-reformatted))))
	    (progn (setq from-varlist nil)
		   (setq continue nil))
	  (setq from-varlist (spark-compare-from-varlist (nth 0 (nth i paramlist))
							 (nth 0 (nth j paramlist-reformatted))
							 (nth 1 (nth i paramlist))
							 (nth 1 (nth j paramlist-reformatted))))
	  (if from-varlist
	      (setq continue nil))))

      ;; add this parameter-declaration to the list
      (setq paramlist-reformatted (append paramlist-reformatted (list
								 (if continue
								     (nth i paramlist)
								   (list (sort (copy-sequence (append (nth 0 (nth i paramlist)) (nth 0 (nth j paramlist-reformatted)))) 'spark-sort-no-case)
									 from-varlist)))))
      (if (not continue)
	  (setq paramlist-reformatted (delete (nth j paramlist-reformatted) paramlist-reformatted))))
    (sort paramlist-reformatted 'spark-sort-derives-varlist)))

(defun spark-scan-derives (begin end)
  "Scan the SPARK derives annotation found in between BEGIN and END.
Return the equivalent internal parameter list."
  (let ((notend t)
	(match-cons nil)
	(apos nil)
	(epos nil)
	(fpos nil)
	(semipos nil)
	star
	(paramlist (list))
	derives-varlist
	derives-from-varlist)

    (goto-char begin)

    ;; loop until end of the SPARK derives annotation
    (while notend

      ;; find first character of parameter-declaration
      (ada-goto-next-non-ws nil nil t)
      (setq apos (point))

      ;; find last character of parameter-declaration
      (if (setq match-cons
		(ada-search-ignore-string-comment "from\\>" nil end nil nil t))
	  (setq fpos (car match-cons))
	(setq fpos end))

      (if (setq match-cons
		(ada-search-ignore-string-comment "[ \t\n]*[&;]" nil end nil nil t))
	  (progn
	    (setq epos (car match-cons))
	    (setq semipos (cdr match-cons)))
	(setq epos end))

      (goto-char apos)

      (setq derives-varlist nil)
      (while (progn (ada-goto-next-non-ws nil nil t)
		    (< (point) fpos))

	(looking-at "\\(\\sw\\|[_.]\\)+")
	(setq derives-varlist
	      (append derives-varlist
		      (list (match-string 0))))

	(ada-search-ignore-string-comment ",\\|from\\>" nil epos nil nil t))

      (setq derives-from-varlist nil)
      (setq star nil)
      (while (progn (ada-goto-next-non-ws nil nil t)
		    (< (point) epos))
      
	(looking-at "\\(\\sw\\|[_.\\*]\\)+")
	(if (and (eq (length derives-varlist) 1)
		 (or (string= (upcase (match-string 0)) (upcase (car derives-varlist)))
		     (string= (match-string 0) "*")))
	    (if (not star)
		(progn (setq star 't)
		       (setq derives-from-varlist
			     (append derives-from-varlist
				     (list "*")))))
	  (if (not (spark-string-search (match-string 0) derives-from-varlist))
	      (setq derives-from-varlist
		    (append derives-from-varlist
			    (list (match-string 0))))))

	(ada-search-ignore-string-comment "[ \t\n]*[,&]" nil epos nil nil t))

      (if (and (> (length derives-varlist) 1)
	       (spark-include derives-varlist derives-from-varlist)
	       (spark-string-search "*" derives-from-varlist))
	  (setq derives-from-varlist (spark-delete-string "*" derives-from-varlist)))

      ;; add this parameter-declaration to the list
      (setq paramlist (when derives-varlist
			(append paramlist (list (list (sort derives-varlist 'spark-sort-no-case)
						      (sort derives-from-varlist 'spark-sort-no-case))))))

      ;; check if it was the last parameter
      (if (eq epos end)
	  (setq notend nil)
	(goto-char semipos)))
    paramlist))

(defun spark-insert-derives (paramlist)
  "Insert a formatted PARAMLIST in the buffer."
  (let (i
	j
	derives-col
	fromcol
	derives-from-col
	derives-list-len
	paramlist-len
	(parlen 0))

    (setq i (length paramlist))
    (setq derives-list-len (1- i))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      
      (setq j (length (nth 0 (nth (- derives-list-len i) paramlist))))

      ;; loop until last parameter
      (while (not (zerop j))
	(setq j (1- j))
      
	;; get max length of parameter-name
	(setq parlen (max parlen (length (nth j (nth 0 (nth (- derives-list-len i) paramlist))))))))

    (setq derives-col (current-column))
    (setq fromcol (+ (current-column) parlen))
    (setq derives-from-col (+ fromcol 6))
    (setq i (length paramlist))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))

      (setq j (length (nth 0 (nth (- derives-list-len i) paramlist))))
      (setq paramlist-len (1- j))

      ;; loop until last parameter
      (while (not (zerop j))
	(setq j (1- j))
	(indent-to derives-col)
	(insert (nth (- paramlist-len j) (nth 0 (nth (- derives-list-len i) paramlist))))
	(if (not (zerop j))
	    (progn (insert ",")
		   (newline)
		   (insert ada-spark-annotation-start)
		   (backward-char 3)
		   (ada-indent-current)
		   (forward-char 3))))

      (indent-to fromcol)
      (insert " from ")

      (setq j (length (nth 1 (nth (- derives-list-len i) paramlist))))
      (setq paramlist-len (1- j))

      ;; loop until last parameter
      (while (not (zerop j))
	(setq j (1- j))
	(indent-to derives-from-col)
	(insert (nth (- paramlist-len j) (nth 1 (nth (- derives-list-len i) paramlist))))
	(if (not (zerop j))
	    (progn (insert ",")
		   (newline)
		   (insert ada-spark-annotation-start)
		   (backward-char 3)
		   (ada-indent-current)
		   (forward-char 3))))
      (if (zerop i)
	  (insert ";")
	(insert " &")
	(newline)
	(insert ada-spark-annotation-start)
	(backward-char 3)
	(ada-indent-current)
	(forward-char 3)))))

(defun spark-format-derives ()
  "Reformat the SPARK derives annotation point is in."
  (interactive)
  (let ((begin nil)
	(end nil)
	(paramlist nil)
	(paramlist-reformatted nil))

    ;; check if really inside a SPARK derives annotation
    (or (spark-in-spark-derives-p)
	(error "Not in SPARK derives annotation"))

    ;; find start of current SPARK derives annotation
    (ada-search-ignore-string-comment "derives\\>" t nil nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (forward-word 1)
    (setq begin (1+ (point)))
    (ada-goto-next-non-ws nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))

    ;; find end of the SPARK derives annotation
    (ada-search-ignore-string-comment ";" nil nil nil nil t)
    (delete-char -1)
    (insert "\n")
    (setq end (point))

    ;; build a list of all elements of the SPARK derives annotation
    (setq paramlist (spark-scan-derives begin (1- end)))

    ;; delete the original SPARK derives annotation
    (delete-region begin end)

    (when paramlist
      ;; reformat the SPARK derives list
      (setq paramlist-reformatted (spark-derives-reformat paramlist)))

    ;; insert the new SPARK derives list
    (goto-char begin)
    (if paramlist
	(spark-insert-derives paramlist-reformatted)
      (insert ";"))))

;;--------------------------
;;-- SPARK format inherit --
;;--------------------------
(defun spark-in-spark-inherit-p ()
  "Return t if point is inside a SPARK inherit annotation."
  (if (spark-in-annotation-p)
      (let ((orgpoint (point)))
	(save-excursion
	  (ada-goto-stmt-start nil t)
	  (re-search-forward "inherit\\>[ \t\n]" orgpoint t)))
    nil))

(defun spark-format-inherit ()
  "Reformat SPARK inherit annotation point is in."
  (interactive)
  (let (i
	inherit-list-len
	(begin nil)
	(end nil)
	(inherit-list nil)
	(inherit-list-reformatted nil))

    ;; check if really inside a SPARK inherit annotation
    (or (spark-in-spark-inherit-p)
	(error "Not in SPARK inherit annotation"))

    ;; find start of current SPARK inherit annotation
    (ada-search-ignore-string-comment "inherit\\>" t nil nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (forward-word 1) ;; skip INHERIT
    (ada-goto-next-non-ws nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (setq begin (point))

    ;; find end of the SPARK inherit annotation
    (ada-search-ignore-string-comment ";" nil nil nil nil t)
    (delete-char -1)
    (insert "\n")
    (setq end (point))

    ;; build a list of all elements of the SPARK inherit annotation
    (goto-char begin)
    (while (progn (ada-goto-next-non-ws nil nil t)
		  (< (point) end))
      (looking-at "\\(\\sw\\|[_.\\*]\\)+")
      (setq inherit-list (append inherit-list 
				 (list (match-string 0))))
      (ada-search-ignore-string-comment "[ \t\n]*," nil end nil nil t))
    
    ;; delete the original SPARK inherit annotation
    (delete-region begin end)

    ;; reformat the SPARK inherit list
    (setq inherit-list-reformatted (sort inherit-list 'spark-sort-no-case))

    ;; insert the new SPARK inherit list
    (goto-char begin)
    (setq i (length inherit-list-reformatted))
    (setq inherit-list-len (1- i))
    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (insert (nth (- inherit-list-len i) inherit-list-reformatted))
      (if (zerop i)
	  (insert ";")
	(insert ",")
	(newline)
	(insert ada-spark-annotation-start)
	(backward-char 3)
	(ada-indent-current)
	(forward-char 3)
	(ada-indent-current)))))

;;----------------
;;-- SPARK menu --
;;----------------
(defun spark-find-switch (the-switch the-command in-switch-file)
  "Find a specific switch in the SPARK switch file (if in-switch-file is not nil) or in the command line."
  (if (string-match the-switch the-command)
      (match-string 0)
    (when in-switch-file
      (let ((spark-source-buffer (current-buffer)))
	(set-buffer (find-file-noselect in-switch-file))
	(prog1 (save-excursion
		 (goto-char (point-min))
		 (if (re-search-forward the-switch nil t)
		     (match-string 0)
		   nil))
	  (switch-to-buffer spark-source-buffer))))))

(defun spark-brief-switch ()
  "Return the '-brief' switch for SPARK Examiner command line."
  (if (spark-find-switch "-brief" ada-spark-examiner-command "spark.sw")
      ""
    " -brief=fullpath"))

(defun spark-current-file (&optional arg)
  "SPARK the current file."
  (interactive "P")
  (compile (concat ada-spark-examiner-command (spark-brief-switch) " " (file-name-nondirectory (buffer-file-name))))) ;; SPARK

(defun spark-metafile (meta-filename &optional arg)
  "SPARK the metafile."
  (interactive "FSPARK Metafile: \np")
  (compile (concat ada-spark-examiner-command (spark-brief-switch) " @" (file-name-nondirectory meta-filename)))) ;; SPARK

(defun spark-vcg-switch ()
  "Return '-vcg' switch for SPARK Examiner command line."
  (if (spark-find-switch "-vcg" ada-spark-examiner-command "spark.sw")
      ""
    " -vcg"))

(defvar spark-start-re
  (eval-when-compile
    (concat "\\<" (regexp-opt '("accept" "end" "entry" "function" "overriding" "package" "procedure"
				"protected" "separate" "task")) "\\>"))
  "Regexp for the start/end of a subprogram, a package or a separate")

(defun spark-find-full-name ()
  "Find the full name of the subprogram/package. Detect if there are nested subprograms/packages.
Return a list:
- nested    : the subprogram/package contains nested subprograms/packages;
- full-name : the full name of the subprogram/package. Terminate with a '/' for a package."
  (let ((full-name nil)
	(nested nil)
	(bound (point))
	next-bound
	(subprogram-stack nil)
	subprogram-name)
    (save-excursion
      (while (not (bobp))
	(ada-search-ignore-string-comment spark-start-re t)
	(setq next-bound (point))
	(beginning-of-line)
	(unless (or (re-search-forward "^[ \t]*end[ \t][ \t]*if[ \t]*;" bound t) ;; end if
		    (re-search-forward "^[ \t]*end[ \t][ \t]*loop[ \t]*;" bound t) ;; end loop
		    (re-search-forward "^[ \t]*end[ \t][ \t]*case[ \t]*;" bound t) ;; end case
		    (re-search-forward "^[ \t]*end[ \t][ \t]*record[ \t]*;" bound t)) ;; end record
	  (if (re-search-forward (concat "^[ \t]*separate[ \t]*(" ada-name-regexp ")") bound t) ;; separate (...)
	      (unless (car subprogram-stack)
		(setq full-name (if full-name
				    (concat (downcase (match-string 1)) "/" full-name)
				  (downcase (match-string 1))))
		(setq next-bound (point-min))) ;; stop the loop
	    (if (or (re-search-forward (concat "^[ \t]*end[ \t][ \t]*" ada-name-regexp "[ \t]*;") bound t) ;; end subprogram/package
		    (re-search-forward "[ \t][ \t]*is[ \t][ \t]*\\(separate\\)[ \t]*;" bound t)) ;; is separate
		(progn (unless full-name
			 (setq nested t)) ;; nested subprogram/package
		       (push (downcase (match-string 1)) subprogram-stack)) ;; push to the stack
	      (if (re-search-forward ada-procedure-start-regexp bound t) ;; subprogram
		  (progn (setq subprogram-name (downcase (match-string 5)))
			 (if (car subprogram-stack)
			     (when (or (string= "separate" (car subprogram-stack))
				       (string= subprogram-name (car subprogram-stack)))
			       (pop subprogram-stack)) ;; match end subprogram -> pop from the stack
			   (setq full-name (if full-name ;; add subprogram to the full-name
					       (concat subprogram-name "/" full-name)
					     subprogram-name))))
		(if (re-search-forward ada-package-start-regexp bound t) ;; package
		    (progn (setq subprogram-name (downcase (match-string 4)))
			   (if (car subprogram-stack)
			       (when (or (string= "separate" (car subprogram-stack))
					 (string= subprogram-name (car subprogram-stack)))
				 (pop subprogram-stack)) ;; match end package -> pop from the stack
			     (setq full-name (if full-name ;; add package to the full-name
						 (concat subprogram-name "/" full-name)
					       (concat subprogram-name "/"))))))))))
	(goto-char next-bound)
	(setq bound next-bound)))
    (list nested (subst-char-in-string ?. ?/ full-name))))

(defun spark-sparksimp-p-switch ()
  "Return '-p=jobs' switch within the SPARKSimp command line."
  (if (spark-find-switch "-p=" ada-spark-sparksimp-command nil)
      ada-spark-sparksimp-command
    (if (spark-find-switch " --*[a-zA-Z0-9]" ada-spark-sparksimp-command nil)
	(replace-regexp-in-string "\\( -\\).*\\'" (concat " -p=" (int-to-string ada-spark-number-of-jobs) " -") ada-spark-sparksimp-command t t 1)
      (concat ada-spark-sparksimp-command " -p=" (int-to-string ada-spark-number-of-jobs)))))

(defun spark-output-dir ()
  "Return the output directory of a SPARK command followed by a '/'."
  (let ((output-dir-str (spark-find-switch "-output_directory=[^ ]*" ada-spark-examiner-command "spark.sw")))
    (expand-file-name (if output-dir-str
			  (concat (substring output-dir-str 18) "/")
			"./"))))

(defun spark-simplify-pogs (&optional nested-file-or-directory)
  "Take the list produced by 'spark-find-full-name'.
Return a list:
- the-command : the command to Simplify and POGS a file or a directory;
- the-files   : a list a files (max 2) that will be produced by the command (SIV by spadesimp + SUM by POGS)."
  (let (the-command
	(the-files nil)
	(the-output-dir (spark-output-dir)))
    (setq the-command
	  (if nested-file-or-directory
	      (let ((nested (nth 0 nested-file-or-directory))
		    (file-or-directory (nth 1 nested-file-or-directory)))
		(if (file-name-nondirectory file-or-directory)
		    ;; Simplify a subprogram
		    (prog1 (concat "cd " (file-name-directory file-or-directory) " && " ada-spark-simplifier-command " " (file-name-nondirectory file-or-directory) ".vcg"
				   (if nested
				       ;; Simplify nested subprograms
				       (prog1 (concat " && cd " (file-name-nondirectory file-or-directory) " && " (spark-sparksimp-p-switch) " && " ada-spark-pogs-command)
					 (push (concat the-output-dir file-or-directory "/" (file-name-nondirectory file-or-directory) ".sum") the-files))
				     ;; No nested subprograms to simplify
				     ""))
		      (push (concat the-output-dir file-or-directory ".siv") the-files))
		  ;; Simplify a set of subprograms
		  (if nested
		      (prog1 (concat "cd " file-or-directory " && " (spark-sparksimp-p-switch) " && " ada-spark-pogs-command)
			(push (concat the-output-dir file-or-directory (file-name-nondirectory (substring file-or-directory 0 (1- (length file-or-directory)))) ".sum") the-files))
		    (error "Nothing to SPARK."))))
	    (prog1 (concat (spark-sparksimp-p-switch) " && " ada-spark-pogs-command)
	      (push (concat the-output-dir (file-name-nondirectory (substring the-output-dir 0 (1- (length the-output-dir)))) ".sum") the-files))))
    (list (concat "cd " the-output-dir " && " the-command) the-files)))

(defvar spark-proc-siv-sum-files nil
  "List made up of '(PROC [SIV] [SUM])' but at least one SIV or SUM file.")

(defun spark-open-siv-sum-files-sentinel (proc msg)
  "Open the SIV and SUM files when 'spark-simplify-pogs' command finished."
  (compilation-sentinel proc msg)
  (let* ((the-files spark-proc-siv-sum-files)
	 (the-proc (pop the-files)))
    (when (and the-proc
	       (string= (process-name proc) (process-name the-proc)))
      (when (string= msg "finished\n")
	(while the-files
	  (let ((buf (find-file-noselect (pop the-files) t)))
	    (with-current-buffer buf
	      (revert-buffer nil t))
	    (switch-to-buffer-other-window buf))))
      (setq spark-proc-siv-sum-files nil))))

(defun spark-proof-current-body (&optional arg)
  "SPARK and Simplify the current body (without the nested subprograms).
Open the SIV file."
  (interactive "P")
  (let* ((the-command-the-files (spark-simplify-pogs (list nil (nth 1 (spark-find-full-name)))))
	 (the-command (nth 0 the-command-the-files))
	 (the-files   (nth 1 the-command-the-files))) ;; (SIV)
    (push (get-buffer-process (compile (concat ada-spark-examiner-command (spark-vcg-switch) (spark-brief-switch) " " (file-name-nondirectory (buffer-file-name)) " && " ;; SPARK
					       the-command))) ;; Simplify
	  the-files) ;; (PROC SIV)
    (setq spark-proc-siv-sum-files the-files)
    (set-process-sentinel (car the-files) 'spark-open-siv-sum-files-sentinel)))

(defun spark-proof-current-subprogram (&optional arg)
  "SPARK and Simplify the current subprogram (with the nested subprograms).
Run POGS and open the SIV file and the POGS summary file."
  (interactive "P")
  (let* ((the-command-the-files (spark-simplify-pogs (spark-find-full-name)))
	 (the-command (nth 0 the-command-the-files))
	 (the-files   (nth 1 the-command-the-files))) ;; (SIV [SUM])
    (push (get-buffer-process (compile (concat ada-spark-examiner-command (spark-vcg-switch) (spark-brief-switch) " " (file-name-nondirectory (buffer-file-name)) " && " ;; SPARK
					       the-command))) ;; Simplify and POGS
	  the-files) ;; (PROC SIV [SUM])
    (setq spark-proc-siv-sum-files the-files)
    (set-process-sentinel (car the-files) 'spark-open-siv-sum-files-sentinel)))

(defun spark-proof-current-file (&optional arg)
  "SPARK and Simplify the current file.
Run POGS and open the POGS summary file."
  (interactive "P")
  (let* ((the-command-the-files (spark-simplify-pogs))
	 (the-command (nth 0 the-command-the-files))
	 (the-files   (nth 1 the-command-the-files))) ;; (SUM)
    (push (get-buffer-process (compile (concat ada-spark-examiner-command (spark-vcg-switch) (spark-brief-switch) " " (file-name-nondirectory (buffer-file-name)) " && " ;; SPARK
					       the-command))) ;; Simplify and POGS
	  the-files) ;; (PROC SUM)
    (setq spark-proc-siv-sum-files the-files)
    (set-process-sentinel (car the-files) 'spark-open-siv-sum-files-sentinel)))

(defun spark-proof-metafile (meta-filename &optional arg)
  "SPARK and Simplify the metafile.
Run POGS and open the POGS summary file."
  (interactive "FSPARK Metafile: \np")
  (let* ((the-command-the-files (spark-simplify-pogs))
	 (the-command (nth 0 the-command-the-files))
	 (the-files   (nth 1 the-command-the-files))) ;; (SUM)
    (push (get-buffer-process (compile (concat ada-spark-examiner-command (spark-brief-switch) " @" (file-name-nondirectory meta-filename) " && " ;; SPARK
					       the-command))) ;; Simplify and POGS
	  the-files) ;; (PROC SUM)
    (setq spark-proc-siv-sum-files the-files)
    (set-process-sentinel (car the-files) 'spark-open-siv-sum-files-sentinel)))

(defun spark-proof ()
  "Simplify all.
Run POGS and open the POGS summary file."
  (interactive)
  (let* ((the-command-the-files (spark-simplify-pogs))
	 (the-command (nth 0 the-command-the-files))
	 (the-files   (nth 1 the-command-the-files))) ;; (SUM)
    (push (get-buffer-process (compile the-command)) ;; Simplify and POGS
	  the-files) ;; (PROC SUM)
    (setq spark-proc-siv-sum-files the-files)
    (set-process-sentinel (car the-files) 'spark-open-siv-sum-files-sentinel)))

;;------------------------
;;-- SPARK pretty print --
;;------------------------
(defun spark-pretty-print-reformat (re-pattern process)
  "Local function for spark-pretty-print."
  (let (search-pos)
    (goto-char (point-min))
    (while (re-search-forward re-pattern nil t)
      (setq search-pos (1+ (point)))
      (funcall process)
      (goto-char search-pos))))

(defun spark-pretty-print ()
  "Pretty print Ada code and SPARK annotations."
  (interactive)
  (ada-indent-region (point-min) (point-max))
  (spark-pretty-print-reformat "\\<\\(procedure\\|function\\)[ \t\n]+[a-zA-Z0-9_]+[ \t\n]+("
			       (lambda ()
				 (let ((previous-syntax-table (syntax-table)))
				   (forward-char 1)
				   (when (unwind-protect
					     (progn
					       (set-syntax-table ada-mode-symbol-syntax-table)
					       (ada-in-paramlist-p (spark-in-annotation-p)))
					   (set-syntax-table previous-syntax-table))
				     (ada-format-paramlist)))))
  (spark-pretty-print-reformat "([ \t]*[a-zA-Z0-9][a-zA-Z0-9_\. |]*[ \t]*=>" 
  			       (lambda ()
  				 (when (not (ada-in-string-or-comment-p))
  				   ;; !!! FIXME !!! (when (or (not (ada-in-string-or-comment-p))
  				   ;; !!! FIXME	!!!           (spark-in-annotation-not-in-string-p))
  				   (ada-format-call-paramlist))))
  (when ada-spark-mode
    (spark-pretty-print-reformat (concat "^[ \t]*" ada-spark-annotation-start "[ \t]*inherit[ \t\n]") 'spark-format-inherit)
    (spark-pretty-print-reformat (concat "^[ \t]*" ada-spark-annotation-start "[ \t]*own[ \t\n]")     'spark-format-own-or-global)
    (spark-pretty-print-reformat (concat "^[ \t]*" ada-spark-annotation-start "[ \t]*global[ \t\n]")  'spark-format-own-or-global)
    (spark-pretty-print-reformat (concat "^[ \t]*" ada-spark-annotation-start "[ \t]*derives[ \t\n]") 'spark-format-derives)))

(provide 'spark-mode)
