;; 2d and 1d vector displaying library

;; Copyright (C) 2004  Sylvain Beucler
;; Copyright (C) 2004  Julien Charles
;; Copyright (C) 2004  Pierre Chtel
;; Copyright (C) 2004  Cyril Rodas

;; This file is part of SpcialK.

;; SpcialK is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; SpcialK is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SpcialK; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

(module display-vector
  mzscheme
  (provide new-tab-viewer)
  (require (lib "mred.ss" "mred")(lib "class.ss" "mzlib")
	   (lib "string.ss" "mzlib") "l10n.ss")
(define (new-tab-viewer vector)
  (let* ((tab vector)
         (max-cell-char (get-max-length vector))
         (min-cell-width 30)
         (cell-width (let ((tmp (+ (* max-cell-char 7) 5)))
                           (if (< tmp min-cell-width)
                               min-cell-width
                               tmp)))
         (required-width (* (vector-length vector) cell-width))
         (required-height (* (get-max-height vector) cell-width))
         ($frame (instantiate frame% () (label (localized-message 'display-vector-title))))
         (face-bitmap (instantiate bitmap% ((* required-width 2) (* required-height 2))))
         (bm-dc (instantiate bitmap-dc% (face-bitmap)))
         (brushes `(,(instantiate brush% ("BLUE" 'solid))
                     ,(instantiate brush% ("YELLOW" 'solid))
                     ,(instantiate brush% ("GREEN" 'solid))))
         (associations '())
         (canvas (instantiate canvas% ()
                   (parent $frame)
                   (min-width required-width)
                   (min-height required-height)
                   (stretchable-width #t)
                   (stretchable-height #t)
                   (paint-callback
                    (lambda (c dc)
                      (send bm-dc clear) 
                      ;(send dc clear)
                      (if (null? associations)
                          (let ((tmp (display-line tab bm-dc 0 0 cell-width brushes '())))
                            (set! brushes (car tmp))
                            (set! associations (cadr tmp)))
                          (display-line tab bm-dc 0 0 cell-width brushes associations))
                      (send dc draw-bitmap face-bitmap 0 0)
                      )))))
    (define (tab-viewer action . args)
      (case action
        ;change the vector that will be displayed (but no refresh)
        ((set!) (if (not (null? args))
                    (set! tab (car args))))
        ;show viewer's frame
        ((display) (send $frame show #t))
        ;refresh the viewer
        ((refresh) (send canvas on-paint))
        ;refresh the viewer with a new vector
        ((refresh-with) (if (not (null? args))
                           (begin (set! tab (car args))
                                  (send canvas on-paint))))
        ;associate new brushes to elements and refresh
        ((colors) (if (not (null? args))
                      (begin (set! brushes (car args))
                             (tab-viewer 'reset))))
        ;recalculate colors associations
        ((reset) (set! associations '())
                 (tab-viewer 'refresh))
        (else (error (localized-message 'unknown-action) action))))
    tab-viewer))
                    
(define (display-line content dc x y width free-colors associations)
  (if (vector? content)
      (do ((i 0 (+ i 1))
           (x x (+ x width -1)))
        ((= i (vector-length content)))
        (let* ((element (vector-ref content i))
               (res (if (vector? element)
                        (display-row element dc x y width free-colors associations)
                        (display-cell element dc x y width free-colors associations))))
          (set! free-colors (car res))
          (set! associations (cadr res)))))
  (list free-colors associations))

(define (display-row content dc x y width free-colors associations)
   (if (vector? content)
      (do ((i 0 (+ i 1))
           (y y (+ y width -1)))
        ((= i (vector-length content)))
        (let* ((element (vector-ref content i))
               (res (display-cell element dc x y width free-colors associations)))
          (set! free-colors (car res))
          (set! associations (cadr res)))))
  (list free-colors associations))
  
(define (display-cell content dc x y width free-colors associations)
  (let ((used? (assoc content associations)))
    (if used?
        (send dc set-brush (cadr used?))
        (if (null? free-colors)
            (send dc set-brush (instantiate brush% ("WHITE" 'solid)))
            (begin (send dc set-brush (car free-colors))
                   (set! associations (cons (list content (car free-colors)) associations))
                   (set! free-colors (cdr free-colors)))))
    (send dc draw-rectangle x y width width)
    (let* ((text (if (string? content)
                    content
                    (expr->string content)))
           (char-largeur 3)
           (char-hauteur 10)
           (position-x (- (+ x (/ width 2)) (* (string-length text) char-largeur)))
           (position-y (- (+ y (/ width 2)) char-hauteur)))
      (send dc draw-text text position-x position-y)))
      (list free-colors associations))

;returns the longest element number of character
(define (get-max-length content)
  (define (internal vector i)
    (if (< i (- (vector-length vector) 1))
        (max (get-max-length (vector-ref vector i)) (internal vector (+ i 1)))
        (get-max-length (vector-ref vector i))))
  (cond ((vector? content) (internal content 0))
        ((string? content) (string-length content))
        (else (string-length (expr->string content)))))

;returns the height number of cells 
(define (get-max-height content)
  (define (internal vector i)
    (if (< i (vector-length vector)) 
        (if (vector? (vector-ref vector i))
            (max (vector-length (vector-ref vector i)) (internal vector (+ i 1)))
            (max 1 (internal vector (+ i 1))))
        0))
  (if (vector? content)
      (internal content 0)
      1)))

;(display-vector '#4(a #3(1 (1 prilililili) a) #4(1 (1 2) #2(a b) "foo") "SpcialK is fun"))
;(define test (new-tab-viewer '#4(a #3(1 (1 prilililili) a) #4(1 (1 2) #2(a b) "foo") "SpcialK is fun")))
;(define test (new-tab-viewer #(#(0 1 2 3) #(3 2 1 0) #(2 1 0 3) #(0 3 1 2))))
;(test 'display)
;(do ((i 0 (+ i 1)))
;  ((= i 100))
;  (begin (test 'refresh-with `#(#(,(random 4) ,(random 4) ,(random 4) ,(random 4))
;                                #(,(random 4) ,(random 4) ,(random 4) ,(random 4))
;                                #(,(random 4) ,(random 4) ,(random 4) ,(random 4))
;                                #(,(random 4) ,(random 4) ,(random 4) ,(random 4))))
;         (sleep 0.1)))
