;; Grammar - Parser

;; Copyright (C) 2004  Sylvain Beucler
;; Copyright (C) 2004  Julien Charles
;; Copyright (C) 2004  Pierre Chtel
;; Copyright (C) 2004  Cyril Rodas

;; This file is part of SpcialK.

;; SpcialK is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; SpcialK is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SpcialK; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

(module special-k-grammar
  mzscheme
  (require "special-k-structs.ss"
	   "l10n.ss"
           (lib "tool.ss" "drscheme")
           (lib "class.ss")
           (lib "lex.ss" "parser-tools")
           (lib "yacc.ss" "parser-tools")
           (lib "readerr.ss" "syntax")
           (lib "embed.ss" "compiler")
           (lib "mred.ss" "mred")
           (lib "errortrace.ss" "errortrace"))
  (provide special-k-parser)
  
  
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;             Analyseur             ;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  
  (define global-dict (new_dict))
  (define sourcefile '?)
  
  (define-tokens value-tokens (ID INTEGER REAL BOOL SCHEMECODE STRING))
  (define-empty-tokens op-tokens 
                       (-> \= \( \) \; \. \, \: \< \>  ! == <> <= >= <-  <-> + - * / mod div \| \{ \} \[ \] \&  NEG eof nil true false))
  
  (define-lex-abbrevs (letter (: (- #\a #\z) (- #\A #\Z)))
                      (alphanum (:(- #\a #\z) (- #\A #\Z) (- #\0 #\9) #\_ #\- #\?))
                      (ascii (- #\000 #\177))
                      (digit (- #\0 #\9)))
  (define get-kappa
    (lexer-src-pos
     ((eof) 'EOF)
     ((: #\tab #\space #\newline (@ "/*" (~ (@ (&) "*/" (&))) "*/"))
         (return-without-pos (get-kappa input-port)))
     ((: "->" "=" "(" ")" ";" "." "," ":" "<" ">"  "!" "==" "<>" "<=" ">=" "<-" "<->" "+" "-" "*" "/" "mod" "div" "|" "{" "}" "[" "]" "&" "nil")
      (string->symbol lexeme))
      ((: "true" "false") (token-BOOL (string->symbol lexeme)))
      ((@ (+ letter) (* alphanum)) (token-ID (string->symbol lexeme)))
      ((+ digit) (token-INTEGER (string->number lexeme)))
      ((@ "<%%" (* (: (@ #\% #\% (^ #\>))
                      (@ #\% (^ #\%))
                      (@ (^ #\%)))) "%%>") (token-SCHEMECODE lexeme))
      ((@ #\" (* (^ #\")) #\") (token-STRING lexeme))
      ((: (@ (* digit) #\. (+ digit)) (@ (+ digit) (: #\e #\E) (? #\-)  (+ digit)))  (token-REAL (string->number lexeme)))
      
      ((- #\000 #\377) (raise-read-error (localized-message 'unauthorized_char_error)
                                         sourcefile (position-line start-pos) (position-col start-pos) 
                                         (position-offset start-pos) 
                                         (- (position-offset end-pos) (position-offset start-pos))))))
    
    
    (if (file-exists? "debug-lalr.txt")
        (delete-file "debug-lalr.txt"))
    (define (addfun f L) 
      (if (null? L)
          (list f)
          (if (equal? (car f) (caar L))
              (cons (append  (car L)(cdr f)) (cdr L))
              (cons (car L) (addfun f (cdr L))))))
    
    (define special_k->scheme
      (parser
       (start go)
       (end EOF)
       (src-pos)
       (suppress)
       (tokens value-tokens op-tokens)
       (error (lambda (token-ok token-name token-value start-pos end-pos)
                (raise-read-error
                 (localized-message 'syntaxic_error token-name)
                 sourcefile 
                 (position-line start-pos) (position-col start-pos)
                 (position-offset start-pos)
                 (- (position-offset end-pos) (position-offset start-pos)))))
       
       (precs (nonassoc <-)
              (left \& \|)
              (nonassoc == <= >= < > <>)
              (right :)
              (left + -)
              (left * / mod div)
              (nonassoc ! NEG))
       
       (grammar
         (go ((prog)   #t))
        (prog ((line prog) #t)
              (() #f))
        
        (line ((list_clause \.) (test_fonc $1 $1-start-pos $2-end-pos)) 
              ; On verifie que toutes les parties-gauches ont le mme identifiant et nombre d'arguments.
              
              ((SCHEMECODE) (set! global-dict (ajouter_code_scheme global-dict $1)))
              ((expr_t \.) (test_expr_t $1 $1-start-pos $2-end-pos)))
        
        (list_clause ((clause \; list_clause) (cons $1 $3))
                     ((clause) (list $1)))
        
        (clause ((partiegauche = expr)
                 (cons (car $1) (test_clause_1 $1 $3 $1-start-pos $3-end-pos)))
                ; On verifie que les arguments de la partie gauche ne contiennent que
                ; des litteraux, variables ou expressions ne contenant que l'operateur ':
                
                ((expr -> partiegauche = expr)
                 (cons (car $3) (test_clause_2 $1 $3 $5 $1-start-pos $5-end-pos))))
                ; On verifie en plus que la garde est bien de type BOOL ou VAR.
        
        
        (partiegauche ((ID \( list_expr \)) (list $1 $3))
                      ((ID \( \)) (list $1 '())))
        
        ; Pout tous les oprateurs si l'appel arbre_bin, arbre_un, arbre_ter renvoi #f
        ; alors on lve une exception exn:read contenant le message d'erreur.
        (expr ((expr * expr) (test_op (arbre_bin '* $1 $3) $1-start-pos $3-end-pos)) 
              ((expr / expr) (test_op (arbre_bin '/ $1 $3) $1-start-pos $3-end-pos))
              ((expr mod expr) (test_op (arbre_bin 'mod $1 $3) $1-start-pos $3-end-pos))
              ((expr div expr) (test_op (arbre_bin 'div $1 $3) $1-start-pos $3-end-pos))
              ((expr + expr) (test_op (arbre_bin '+ $1 $3) $1-start-pos $3-end-pos)) 
              ((expr - expr) (test_op (arbre_bin '- $1 $3) $1-start-pos $3-end-pos)) 
              ((expr > expr) (test_op (arbre_bin '> $1 $3) $1-start-pos $3-end-pos)) 
              ((expr < expr) (test_op (arbre_bin '< $1 $3) $1-start-pos $3-end-pos)) 
              ((expr >= expr) (test_op (arbre_bin '>= $1 $3) $1-start-pos $3-end-pos)) 
              ((expr <= expr) (test_op (arbre_bin '<= $1 $3) $1-start-pos $3-end-pos)) 
              ((expr <> expr) (test_op (arbre_bin '<> $1 $3) $1-start-pos $3-end-pos)) 
              ((expr == expr) (test_op (arbre_bin '== $1 $3) $1-start-pos $3-end-pos))
              ((expr & expr) (test_op (arbre_bin '& $1 $3) $1-start-pos $3-end-pos)) 
              ((expr \| expr) (test_op (arbre_bin '\| $1 $3) $1-start-pos $3-end-pos))  
              ((expr : expr) (test_op (arbre_bin ': $1 $3) $1-start-pos $3-end-pos)) 
              ((! expr) (test_op (arbre_un '! $2) $1-start-pos $2-end-pos)) 
              ((- expr) (prec NEG) (test_op (arbre_un '- $2) $2-start-pos $2-end-pos)) 
              (( \( expr \) ) $2)
             
              (( \{ list_expr \} ) (test_tab_decl $2)) 
              (( \{ \}) (arbre_bin '\{\} 0 '()))
              
              ; On teste si tous les lments de list_expr sont de type INT ou NUMERIC ou VAR.
              ((expr \[ list_expr \]) (test_tab_access $1 $3 $1-start-pos $4-end-pos))
              ((expr \[ list_expr <-> list_expr \]) (test_tab_swap $1 $3 $5 $1-start-pos $6-end-pos))
              ((expr \[ list_expr \] <- expr) (test_tab_affect $1 $3 $6 $1-start-pos $6-end-pos))
              
              
              ((ID \( \)) (arbre_bin 'call $1 '()))
              ((ID \( list_expr \)) (test_call $1 $3))
              
              ((STRING) (feuille $1 K-STRING))
              ((BOOL) (feuille $1 BOOL))
              ((INTEGER) (feuille $1 INT))
              ((REAL) (feuille $1 REAL))
              ((ID) (feuille $1 VAR)) 
              ((nil) (feuille 'nil LISTE)))
        
        (expr_t ((expr) $1)
                ((ID <- expr) (test_affect $1 $3 $1-start-pos $3-end-pos)))
        
        (list_expr ((expr \, list_expr) (cons $1 $3))
                   ((expr) (list $1)))
       
        
        )))
    
    
    
    
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;             Semantique             ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
    (define semantic_errors #f)
    
    (define (error_sem_mess str start_pos end_pos)
      (raise-read-error str sourcefile (position-line start_pos) (position-col start_pos) 
                        (position-offset start_pos) 
                        (- (position-offset end_pos) (position-offset start_pos))))
    
    ; On verifie que toutes les clauses aient bien le meme nombre d'arguments et que nom soit toujours 
    ; identique
    ; List_clauses est de la forme ((nom . clause1) ...)
    (define (test_fonc list_clauses start_pos end_pos)
      (let ((type_error #f) (nom_fonc (caar list_clauses)) (nb_args (length (get_args (cdar list_clauses)))))
        (define (iter l res)
          (cond ((null? l) res)
                ((not (= nb_args (length (get_args (cdar l))))) (begin (set! type_error 'args) #f))
                ((not (equal? (caar l) nom_fonc)) (begin (set! type_error 'nom) #f))
                (else (iter (cdr l) (cons (cdar l) res)))))
        (let ((tmp (iter (reverse list_clauses) '())))
          (cond ((and (not tmp) (equal? type_error 'args))
                 (begin (set! semantic_errors #t)
                        (error_sem_mess (localized-message 'fonc_args_error) start_pos end_pos)))
                ((and (not tmp) (equal? type_error 'nom))
                 (begin (set! semantic_errors #t)
                        (error_sem_mess (localized-message 'fonc_name_error) start_pos end_pos)))
                (tmp (set! global-dict (ajouter_fonc global-dict nom_fonc tmp)))
                (else (begin (set! semantic_errors #t)
                             (error_sem_mess (localized-message 'internal_error) start_pos end_pos)))))))
    
    
    ; si a = #f alors affiche un message d'erreur
    (define (test_op a start_pos end_pos)
      (cond ((not a) (begin (set! semantic_errors #t) (error_sem_mess error_str start_pos end_pos) 'err_expr))
            (else a)))
    
    ; test chaque element e de (cadr partiegauche)
    ; verifie:
    ; e != 'err_mess 
    ; e ne contient pas d'autre operateurs que ':
    ; sinon affiche un message d'erreur
    ; renvoi #t si pas d'erreurs
    
    (define (test_partiegauche partiegauche start_pos end_pos)
      (define (rec_parc liste)
        (cond ((null? liste) #t)
              ((equal? (car liste) 'err_expr) #f)
              ((not (verif_ops (car liste))) 
               (begin (error_sem_mess (localized-message 'invalid_pg_op_error) start_pos end_pos) #f))
              (else (rec_parc (cdr liste)))))
      (rec_parc (cadr partiegauche)))
    
    
    
    ; verifie que arbre ne contient pas d'autres operateurs que ':
    (define (verif_ops arbre)
      (cond ((feuille? arbre) #t)
            ((or (unaire? arbre) (ternaire? arbre)) #f)
            (else (and (list_op? (get_op arbre)) 
                       (verif_ops (get_fg arbre)) 
                       (verif_ops (get_fd arbre))))))
    
    ; verifie partiegauche et expr 
    ; expr != 'err_mess 
    ; sinon semantic_error = #t
    ; met a jour le global-dict
    (define (test_clause_1 partiegauche expr start_pos end_pos)
      (if (or (not (test_partiegauche partiegauche start_pos end_pos))
              (equal? expr 'err_expr))
          (set! semantic_errors #t))
      
      (new_clause start_pos end_pos (cadr partiegauche) #t expr))
    
    
    ; verifie garde partiegauche expr
    ; garde doit etre de type BOOL ou VAR
    ; sinon erreur 
    ; met a jour le global-dict
    (define (test_clause_2 garde partiegauche expr start_pos end_pos)
      (if (or (not (test_partiegauche partiegauche start_pos end_pos))
              (equal? expr 'err_expr))
          (set! semantic_errors #t))
      
      (if (equal? garde 'err_expr) 
          (set! semantic_errors #t)
          (if (and (not (bool? garde)) (not (var? garde)))
              (begin (error_sem_mess (localized-message 'guard_type_error) start_pos end_pos)
                     (set! semantic_errors #t))))
      
      (new_clause start_pos end_pos (cadr partiegauche) garde expr))
    
    
    ; verifie expr_t != err_expr
    ; sinon semantic_error = #t
    (define (test_expr_t expr_t start_pos end_pos)
      (if (equal? expr_t 'err_expr)
          (set! semantic_errors #t))
      (set! global-dict (ajouter_expr global-dict expr_t start_pos end_pos)))
    
    ; Test la liste des elements du tableau
    ; Renvoi 'err_expr en cas d'erreur
    (define (test_tab_decl l)
      (define (parc_rec l)
        (cond ((null? l) #t)
              ((equal? (car l) 'err_expr) #f)
              (else (parc_rec (cdr l)))))
      (if (not (parc_rec l)) (set! semantic_errors #t))
      (arbre_bin '\{\} (length l) l))
    
    ; Verifie expr et list_expr
    ; chaque element de list_expr est de type INT ou NUMERIC ou VAR
    (define (test_tab_access expr list_expr start_pos end_pos)
      (if (not (verif_list_tab list_expr start_pos end_pos)) (set! semantic_errors #t))
      (let ((a (arbre_bin '\[\] expr list_expr)))
        (if a a (begin (error_sem_mess error_str start_pos end_pos) 'err_expr))))
    
    
    
    (define (verif_list_tab list_expr start_pos end_pos)
      (cond ((null? list_expr) #t)
            ((equal? (car list_expr) 'err_expr) #f)
            ((and (not (int? (car list_expr)))
                  (not (numeric? (car list_expr)))
                  (not (var? (car list_expr))))
             (begin (error_sem_mess (localized-message 'index_type_error) start_pos end_pos) #f))
            (else (verif_list_tab (cdr list_expr) start_pos end_pos))))
    
    (define (test_tab_affect expr1 list_expr expr2 start_pos end_pos)
      (if (not (verif_list_tab list_expr start_pos end_pos)) (set! semantic_errors #t))
      (let ((a (arbre_ter '<- expr1 list_expr expr2)))
        (if a a (begin (error_sem_mess error_str start_pos end_pos)  'err_expr))))
    
    (define (test_tab_swap expr list_expr1 list_expr2 start_pos end_pos)
      (if (not (verif_list_tab list_expr1 start_pos end_pos)) (set! semantic_errors #t))
      (if (not (verif_list_tab list_expr2 start_pos end_pos)) (set! semantic_errors #t))
      (let ((a (arbre_ter '<-> expr list_expr1 list_expr2)))
        (if a a (begin (error_sem_mess error_str start_pos end_pos) 'err_expr))))
    
    (define (test_call id list_expr)
      (define (parc_rec l)
        (cond ((null? l) #t)
              ((equal? (car l) 'err_expr) #f)
              (else (parc_rec (cdr l)))))
      (if (not (parc_rec list_expr)) (set! semantic_errors #t))
      (arbre_bin 'call id list_expr))
    
    
    (define (test_affect id expr start_pos end_pos)
      (arbre_bin ':= (feuille id VAR) expr))
    
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;             Execution              ;
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
    (define (scan ip)
      (lambda () (get-kappa ip)))
    (define line 0)
    
    (define (safe-read ip str)
      (set! global-dict '())
      ;  (with-handlers 
      ;      ([exn:read? 
      ;       (lambda (exn) 
      ;         (raise (make-exn:read (exn-message exn) (datum->syntax-object #f 
      ;                                                                       (exn-continuation-marks exn)
      ;                                                                       (list str (exn:read-line exn) (exn:read-column exn) 
      ;                                                                             (exn:read-position exn) (exn:read-span exn)) #f)
      ;                                      str (exn:read-line exn) (exn:read-column exn) 
      ;                                      (exn:read-position exn) (exn:read-span exn))
      ;                               ))]) 
      ;(error-display-handler 
      ; (drscheme:debug:make-debug-error-display-handler (error-display-handler)))
      (set! sourcefile str)
      (special_k->scheme (scan ip))
      global-dict)
    
    
    
    (define (special-k-parser fd filename)
      ;(error-print-source-location #f)
      (safe-read fd filename)))
  ;  
  ;(require special-k-grammaire)
  ;(define truc
  ;  (string-append 
  ;   ;"/* commentaire */ \n"
  ;   "fac(0) = 1;\n"
  ;   "n > 0 -> fac(n) = n * fac().\n"
  ;   ;"<%% (define x 0) %%>\n"
  ;   ;"sort(nil) = *nil;\n"
  ;   ;"L <> nil -> sort(c:L) = insert(c,sort(L)).\n"
  ;   ;"truc() =   x.\n"
  ;   ;"f(\"aazdzd\").\n"
  ;   ))
  ;
  ;
  ;(display truc)
  ;
  ;
  ;(pretty-print-columns 10)
  ;
  ;(with-handlers ([exn:read? (lambda(exn) (printf "~a : ~a~n" (exn:read-source exn) (exn-message exn)))])
  ;  (special-k-parser (open-input-string truc) "truc.spk"))
  ;
  
  ;semantic_errors
  
