#!/usr/bin/python -O
# -*- coding: iso-8859-15 -*-

##    Copyright 2012, Momme Winkelnkemper <specmate@posteo.de>
##
##    This file is part of SpecMate.
##
##    SpecMate is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    Specmate is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with SpecMate.  If not, see <http://www.gnu.org/licenses/>.

#----------------------------------------------------------------------------
# packages to import
#----------------------------------------------------------------------------
import sys,os
import copy
import warnings
#import time
from math import sqrt,pi

try:
	import numpy as na
except:
	print "Please install numpy!"
	sys.exit(1)
try:
	import scipy.interpolate as interp
	from scipy.special import wofz
except:
	print "Please install scipy!"
	sys.exit(1)

from specmate_globals import *
#----------------------------------------------------------------------------
#----------------------------------------------------------------------------

#----------------------------------------------------------------------------
#----------------------------------------------------------------------------
def german2Float(s):
	splitCommas=s.split(',')
	if len(splitCommas)==1:#no commas found, treat as regular float
		return(float(s))
	splitPoints=s.split('.')
	if len(splitPoints)==1:#no points found
		return(float(s.replace(",",".")))
	if len(splitCommas[-1])>len(splitPoints[-1]):
		s=""
		for each in splitCommas:
			s+=each
		return(s)
	s=""
	for each in splitPoints:
		s+=each
	return(float(s.replace(",",".")))
#----------------------------------------------------------------------------
#----------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
def voigt(x,x0,y0,FWHM,shape):
	"""
	Helper function, implements the voigt profile
	
	The FWHM of the voigt profile is determined by the FWHM of the underlying lorentzian (FWHM_l) and that of the underlying gaussian (FWHM_g):
	FWHM=0.5346*FWHM_l+sqrt(0.2166*FWHM_l**2+FWHM_g**2)
	
	The shape factor is currently implemented as follows:
	FWHM_l=shape*FWHM
	FWHM_g=sqrt(((FWHM-0.5346*shape*FWHM)**2)-0.2166*(shape*FWHM)**2)

	@param x: X-axis to calculate the profile on
	@type x: 1D-numpy-array(Float)
	@param x0: Peak position
	@type x0: float
	@param y0: Peak amplitude
	@type y0: float
	@param FWHM: FWHM of the peak
	@type FWHM: float
	@param shape: shape factor of the peak
	@type shape: float
	@return: voigt peak with the given parameters.
	@rtype: 1D-numpy-array(Float)
	"""

	#if shape>0.9999969:
	#	FWHM_l=FWHM
	#	FWHM_g
	FWHM_l=shape*FWHM
	FWHM_g=sqrt(((FWHM-0.5346*shape*FWHM)**2)-0.2166*(shape*FWHM)**2)
	#print FWHM_l,FWHM_g
	#gamma=
	#sigma=

	z = na.sqrt(na.log(2))*(2.0*(x-x0)+1.0j*FWHM_l)/FWHM_g
    	z0 = na.sqrt(na.log(2)) * 1.j * FWHM_l / FWHM_g
    	return (y0*wofz(z).real/wofz(z0).real)
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# begin class "dataset"
#-----------------------------------------------------------------------------------------
class specmate_singleDataset():
	"""
	Default dataset for Specmate. Represent a single spectrum dataset.

	@ivar data: holds the original data as read by dataset.read()
	@type data: 2D-Numpy-Array

	@ivar interpData: holds interpolated spectral data generated by dataset.interpolate()
	@type interpData: 2D-Numpy-Array

	@ivar hasData: True if "data" is present
	@type hasData: Boolean

	@ivar hasInterpData: True if "interpData" is present
	@type hasInterpData: Boolean

	@ivar filename: Name of the data-file read, default is "None"
	@type filename: String

	@ivar name: Name of dataset, default is self.filename
	@type name: String

	@ivar interpTypes: Interpolation types that are supported
	@type interpTypes: String

	@ivar interpType: Interpolation type that is used
	@type interpType: String

	@ivar fitParam: Dict to store parameters for fit routine (Default: 'None')
	@type fitParam: dict
	@ivar linestyle: linestyle in matplotlib-notation (Default: '-')
	@type linestyle: string
	@ivar pointstyle: pointstyle in matplotlib-notation (Default: 'o')
	@type pointstyle: string
	@ivar linewidth: linewidth in matplotlib-notation (Default: 1.0)
	@type linewidth: float
	@ivar pointsize: Pointsize in matplotlib-notation (Default: 2.0)
	@type pointsize: float
	"""
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def read(self,filename):
		"""
		Read spetrum from file, store it in dataset.data
		
		@param filename: Name of txt-file to read
		@type filename: String

		@return: Success(0)/Fail(1)
		@rtype: int
		"""
		try:
			#self.data=na.loadtxt(filename)
			#self.data=na.loadtxt(filename,converters={0 : lambda s: float(s.replace(",",".")),1: lambda s: float(s.replace(",","."))})
			self.clear()
			self.data=na.loadtxt(filename,converters={0: german2Float,1: german2Float})
			self.maxValue=na.max(self.data[:,1])
			self.minValue=na.min(self.data[:,1])
			self.maxX=na.max(self.data[:,0])
			self.minX=na.min(self.data[:,0])
		except:
			warnings.warn("Couldn't read file: "+filename,UserWarning)
			#print "singleDataset.read(): Couldn't read file: "+filename+"."
			return(1)
		if ((len(self.data[:,0])<1) or (len(self.data[:,1])<1)):
			warnings.warn("No data found in: "+filename,UserWarning)
			#print "No data found in: "+filename+"."
			return(1)
		self.hasData=True
		self.hasInterpData=False
		self.filename=filename
		self.name=os.path.basename(self.filename)
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def write(self,filename,which='interpolated'):
		"""
		Write spectrum to file. Writes 'interpolated' or 'original' data, depending on parameter 'which'.  'original data is written in any case if no 'interpolated' data is present. 
		
		@param filename: Name of txt-file to write to.
		@type filename: String
		@param which: whether to save the interpolated or original data.
		@type which: string('interpolated' or 'original') 
		@return: Success(0)/Fail(1)
		@rtype: int
		"""
		data=na.transpose(na.array([self.X(which),self.Y(which)]))
		#if (which=='interpolated'):
		#	if self.hasInterpData==True:
		#		data=self.interpData
		#	else:
		#		print "dataset.write(%s,%s): Warning, Object has no interpolated data. Trying to use original data instead!" % (filename,which) 
		#		which='original'
		#if(which=='original'):
		#	if (self.hasData==True):
		#		data=self.data
		#	else:
		#		print "Couldn't write file, object has no data: "+filename+"."
		#		return(1)
		try:
			na.savetxt(filename,data,fmt='%.5e')
			return(0)
		except:
			warnings.warn("Couldn't write file: "+filename,UserWarning)
			#print "Couldn't write file: "+filename+"."
			return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def interpolate(self,new_x,kind='linear'):#xmin,xmax,dx):
		"""
		Interpolates self.data to the new X-Axis "new_x". Interpolated data is stored in self.interpData  
		@param new_x: New X-Axis 
		@type new_x: 1-D Numpy array (monotonous)
		@param kind: Interpolation scheme to be used 
		@type kind: string ('linear' / 'cubic splines')
		@return: Success(0)/Fail(1)
		@rtype: int
		"""
		if len(self.data)<1:
			warnings.warn("No data present to interpolate in: "+self.name,UserWarning)
			#print "No data present to interpolate!"
			#print self.name
			return(1)
		elif len(self.data)==1:
			new_y=na.interp(new_x,self.data[:,0],self.data[:,1])
			self.interpData=na.transpose(na.array([new_x,new_y]))
			self.hasInterpData=True
			return(0)
		elif len(self.data)>1:
			ind=na.lexsort((self.data[:,1],self.data[:,0]))    
			data=self.data[ind]
			if self.interpType=='linear':
				new_y=na.interp(new_x,data[:,0],data[:,1])
				self.interpData=na.transpose(na.array([new_x,new_y]))
			elif self.interpType=='cubic splines':
				k=3
				if len(data)<=k:
					k=len(data)-1
				f=interp.splrep(data[:,0],data[:,1],s=0.0,k=k)
				new_y=interp.splev(new_x,f)
				self.interpData=na.transpose(na.array([new_x,new_y]))
			else:
				warnings.warn('Unknown interpolation type: %d' %(self.interpType),UserWarning)
				#print 'Unknown interpolation type: %d' %(self.interpType)
				return(1)
			self.hasInterpData=True
			return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def X(self,which='interpolated'):
		"""
		Return the X-Axis for the data to be plotted. Interpolated data is prefered over original data. 
		@return: X-Axis
		@rtype: numpy-array
		"""
		if (which=='interpolated' and self.hasInterpData):
			return(self.interpData[:,0])
		elif (self.hasData):
			return(self.data[:,0])
		else:
			return(na.array([]))
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	##--------------------------------------------------------------------------------
	##--------------------------------------------------------------------------------
	#def calculate(self):
	#	"""
	#	Return the data to be plotted.
	#	@return: data to plot
	#	@rtype: numpy-array
	#	"""
	#	if self.hasInterpData:
	#		return(self.interpData[:,1])
	#	elif self.hasData:
	#		return(self.data[:,1])
	#	else:
	#		return(na.array([]))
	##--------------------------------------------------------------------------------
	##--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def Y(self,which='interpolated'):
		"""
		Return the Y-Axis for the data to be plotted. Interpolated data is prefered over original data. 
		@return: Y-Axis
		@rtype: numpy-array
		"""
		if (which=='interpolated' and self.hasInterpData):
			return(self.interpData[:,1])
		elif (self.hasData):
			return(self.data[:,1])
		else:
			return(na.array([]))
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def clear(self):
		"""
		set up empty dataset / clear dataset.
		"""
		self.hasData=False
		self.hasInterpData=False
		self.filename='None'
		self.name='None'
		self.data=na.transpose(na.array([[],[]]))
		self.interpData=na.transpose(na.array([[],[]]))
		self.interpTypes=['linear', 'cubic splines']
		self.interpType='linear'
		self.fitParam={}
		self.linestyle='-'
		self.pointstyle='o'
		self.linewidth=1.0
		self.pointsize=2.0
		self.xShift=0.0
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def exportDataAsDict(self):
		"""
		Export dataset as dictionary. Intended for use in "save project" routine.
		@return: Dictionary that contains all data of this dataset.
		@rtype: dict
		"""
		return({'name': self.name,'filename': self.filename,'hasData': self.hasData, 'hasInterpData': self.hasInterpData, 'data': self.data, 'interpData': self.interpData,'interpType': self.interpType, 'fitParam': self.fitParam,'linestyle':self.linestyle,'linewidth':self.linewidth,'pointstyle':self.pointstyle,'pointsize':self.pointsize,'xShift':self.xShift})
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def importDataFromDict(self,data):
		""" 
		Import dataset from a dictionary. Intended for use in "open project" routine.
		@param data: dictionary containing a dataset to import
		@type data: dict
		"""
		keys=data.keys()
		self.name=data['name']
		self.filename=data['filename']
		self.hasData=data['hasData']
		self.hasInterpData=data['hasInterpData']
		self.data=data['data']
		self.interpData=data['interpData']
		if 'interpType' in keys:
			self.interpType=data['interpType']
		if 'fitParam' in keys:
			self.fitParam=data['fitParam']
		if 'linestyle' in keys:
			self.linestyle=data['linestyle']
		if 'linewidth' in keys:
			self.linewidth=data['linewidth']
		if 'pointstyle' in keys:
			self.pointstyle=data['pointstyle']
		if 'pointsize' in keys:
			self.pointsize=data['pointsize']
		if 'xShift' in keys:
			self.xShift=data['xShift']
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def shiftData(self,shift):
		"""
		shift x-axis of self.data by (shift-self.xShift) 
		@param shift: new xShift value
		@type shift: float
		"""
		if self.hasData:
			self.data[:,0]+=(shift-self.xShift)
			self.xShift=shift
		if self.hasInterpData:
			self.interpolate(self.data[:,0])
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def clipData(self,xMin,xMax):
		"""
		Clip self.data and self.interpData to given x-range
		@param xMin: lower limit for new x-axis
		@type xMin: float
		@param xMax: upper limit for new x-axis
		@type xMax: float

		"""
		if self.hasData:
			newStart=0
			newEnd=len(self.data[:,0])
			if xMin > self.data[0,0]:
				newStart=na.argmin(na.abs(self.data[:,0]-xMin))
			if xMax < self.data[-1,0]:
				newEnd=na.argmin(na.abs(self.data[:,0]-xMax))
			self.data=self.data[newStart:newEnd,:]
		if self.hasInterpData:
			newStart=0
			newEnd=len(self.interpData[:,0])
			if xMin > self.interpData[0,0]:
				newStart=na.argmin(na.abs(self.interpData[:,0]-xMin))
			if xMax < self.interpData[-1,0]:
				newEnd=na.argmin(na.abs(self.interpData[:,0]-xMax))
			self.interpData=self.interpData[newStart:newEnd,:]
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def __init__(self,filename="None"):
		"""
		If 'filename' is given, dataset.read(filename) is called automatically.
		@param filename: Name of file to open.
		@type filename: String
		@default filename: "None"
		"""
		self.clear()
		if filename!="None":
			if self.read(filename)!=0:
				raise IOError("Couldn't open file %s" % (filename))	
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
# end class "dataset"
#-----------------------------------------------------------------------------------------


#-----------------------------------------------------------------------------------------
# begin class "specmate_refDataset"
#-----------------------------------------------------------------------------------------
class specmate_refDataset(specmate_singleDataset):
	"""
	Dataset for reference spectra in Specmate.

	@ivar coeff: Coefficient to scale the reference spectrum with.
	@type coeff: float
	@ivar maxCoeff: Maximum value for "coeff". Default is 1.0
	@type maxCoeff: float
	@ivar minCoeff: Minimum value for "coeff". Default is 0.0
	@type minCoeff: float
	"""

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def exportDataAsDict(self):
		"""
		Export dataset as dictionary. Intended for use in "save project" routine.
		@return: Dictionary that contains all data of this dataset.
		@rtype: dict
		"""
		data=specmate_singleDataset.exportDataAsDict(self)
		data['coeff']= self.coeff
		data['maxCoeff']= self.maxCoeff
		data['minCoeff']= self.minCoeff
		return(data)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def importDataFromDict(self,data):
		""" 
		Import dataset from a dictionary. Intended for use in "open project" routine.
		@param data: dictionary containing a dataset to import
		@type data: dict
		"""
		specmate_singleDataset.importDataFromDict(self,data)
		self.coeff=data['coeff']
		self.maxCoeff=data['maxCoeff']
		self.minCoeff=data['minCoeff']
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def Y(self,which='interpolated'):
		"""
		Return the data to be plotted (Y-Axis). Return value is scaled by "self.coeff". Interpolated Data is prefered over original data.
		@return: Y-Axis of data to plot
		@rtype: numpy-array
		"""
		return(specmate_singleDataset.Y(self,which)*self.coeff)
		#if self.hasInterpData:
		#	return(self.interpData[:,1]*self.coeff)
		#elif self.hasData:
		#	return(self.data[:,1]*self.coeff)
		#else:
		#	return(na.array([]))
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	##--------------------------------------------------------------------------------
	##--------------------------------------------------------------------------------
	#def calculate(self):
	#	"""
	#	Return the data to be plotted.
	#	@return: data to plot
	#	@rtype: numpy-array
	#	"""
	#	if self.hasInterpData:
	#		return(self.interpData[:,1]*self.coeff)
	#	else:
	#		return(self.data[:,1]*self.coeff)
	##--------------------------------------------------------------------------------
	##--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def clear(self,coeff=0.5):
		"""
		set up empty dataset / clear dataset.
		@param coeff: initial value for self.coeff.
		@type coeff: float
		@default coeff: 0.5
		"""
		specmate_singleDataset.clear(self)
		self.coeff=coeff
		self.maxCoeff=1.0
		self.minCoeff=0.0
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def __init__(self, filename="None",coeff=0.5):
		"""
		If "filename is given, dataset.read(filename) is called automatically.
		@param filename: Name of file to open.
		@type filename: String
		@default filename: "None"
		"""
		self.clear(coeff)
		if filename!="None":
			if self.read(filename)!=0:
				raise IOError("Couldn't open file %s" % (filename))	
		#specmate_singleDataset.__init__(self,filename)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
# end class "specmate_refDataset"
#-----------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# begin class "specmate_peakDataset"
#-----------------------------------------------------------------------------------------
class specmate_peakDataset(specmate_singleDataset):
	"""
	Dataset for single peak spectra in Specmate.

	@ivar x0: Peak position
	@type x0: float
	@ivar y0: Peak amplitude
	@type y0: float
	@ivar FWHM: FWHM of the peak
	@type FWHM: float
	@ivar shape: shape factor (use for voigt peaks only). See doc-string of function "voigt" for details.
	@type shape: float(0..1)
	@ivar peakType: Peak type, "gaussian", "lorentzian", or "voigt"
	@type peakType: string ("gaussian"/"lorentzian"/"voigt")
	@ivar peakTypes: list of supported peak types
	@type peakTypes: list of strings ["gaussian","lorentzian","voigt"]
	"""

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def exportDataAsDict(self):
		"""
		Export dataset as dictionary. Intended for use in "save project" routine.
		@return: Dictionary that contains all data of this dataset.
		@rtype: dict
		"""
		data=specmate_singleDataset.exportDataAsDict(self)
		data['x0']= self.x0
		data['y0']= self.y0
		data['FWHM']= self.FWHM
		data['shape']=self.shape
		data['peakType']=self.peakType
		return(data)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def importDataFromDict(self,data):
		""" 
		Import dataset from a dictionary. Intended for use in "open project" routine.
		@param data: dictionary containing a dataset to import
		@type data: dict
		"""
		specmate_singleDataset.importDataFromDict(self,data)
		self.x0=data['x0'] 
		self.y0=data['y0'] 
		self.FWHM= data['FWHM'] 
		self.shape=data['shape']
		self.peakType=data['peakType']
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def calculate(self):
		"""
		Calculate the Y-axis of the peak data based on the self.XXX parameters and store it in self.interpData. Works only if an X-axis is present in self.interpData. 
		"""
		if self.hasInterpData:
			if ((self.peakType=='gaussian') or ((self.peakType=='voigt') and (self.shape<=0.0))):
				sigma=self.FWHM/2.35482
				self.interpData[:,1]=self.y0*na.exp(-na.square(self.interpData[:,0]-self.x0)/(2.0*sigma**2))
			elif ((self.peakType=='lorentzian') or ((self.peakType=='voigt') and (self.shape>0.99999))):
				gamma=0.5*self.FWHM
				self.interpData[:,1]=self.y0*gamma**2/(na.square(self.interpData[:,0]-self.x0)+gamma**2)
			elif self.peakType=='voigt':
				self.interpData[:,1]=voigt(self.interpData[:,0],self.x0,self.y0,self.FWHM,self.shape)
		else:
			self.interpData*=0.0
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	def interpolate(self,new_x):#xmin,xmax,dx):
		"""
		Set new X-Axis "new_x" and calculate peak-data based on self.XXX parameters (use self.calculate()), store it in self.interpData  
		@param new_x: New X-Axis 
		@type new_x: 1-D Numpy array (monotonous)
		"""
		self.interpData=na.transpose(na.array([new_x,copy.deepcopy(new_x)]))
		#print self.interpData.shape#[:,1]*=0.0
		self.hasInterpData=True
		self.calculate()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def clear(self,peakType='gaussian',x0=0.0,y0=1.0,FWHM=10.0,shape=0.0):
		"""
		set up empty dataset / clear dataset.

		@param peakType: Peak type, "gaussian", "lorentzian", or "voigt"
		@type peakType: string ("gaussian"/"lorentzian"/"voigt")
		@default peakType: 'gaussian'
		@param x0: Peak position
		@type x0: float
		@default x0: 0.0
		@param y0: Peak amplitude
		@type y0: float
		@default y0: 1.0
		@param FWHM: FWHM of the peak
		@type FWHM: float
		@default FWHM: 10.0
		@param shape: shape factor (use for voigt peaks only). See doc-string of function "voigt" for details.
		@type shape: float(0..1)
		@default shape: 0.0
		"""
		specmate_singleDataset.clear(self)
		self.x0=x0
		self.y0=y0
		self.FWHM=FWHM
		self.peakType=peakType
		self.peakTypes=['gaussian','lorentzian','voigt']
		self.shape=shape
		#self.data=na.array([[]])
		#self.hasData=False
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def __init__(self,new_x=na.array([]),peakType='gaussian',x0=0.0,y0=1.0,FWHM=10.0,shape=0.0):
		"""
		@param new_x: X-Axis to be used
		@type new_x: numpy-array(float)
		@param peakType: Peak type, "gaussian", "lorentzian", or "voigt"
		@type peakType: string ("gaussian"/"lorentzian"/"voigt")
		@default peakType: 'gaussian'
		@param x0: Peak position
		@type x0: float
		@default x0: 0.0
		@param y0: Peak amplitude
		@type y0: float
		@default y0: 1.0
		@param FWHM: FWHM of the peak
		@type FWHM: float
		@default FWHM: 10.0
		@param shape: shape factor (use for voigt peaks only). See doc-string of function "voigt" for details.
		@type shape: float(0..1)
		@default shape: 0.0
		"""
		self.clear(peakType,x0,y0,FWHM,shape)
		if len(new_x)>0:
			self.interpolate(new_x)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
# end class "specmate_peakDataset"
#-----------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# begin class "specmate_functionDataset"
#-----------------------------------------------------------------------------------------
class specmate_functionDataset(specmate_singleDataset):
	"""
	Dataset for single function in Specmate.

	@ivar a: function parameters
	@type a: numpy-array((5),'Float64') 
	@ivar functionString:
	@type functionString: string
	@ivar functionName:
	@type functionName: string
	@ivar functionDescription:
	@type functionDescription: string

	"""

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def exportDataAsDict(self):
		"""
		Export dataset as dictionary. Intended for use in "save project" routine.
		@return: Dictionary that contains all data of this dataset.
		@rtype: dict
		"""
		data=specmate_singleDataset.exportDataAsDict(self)
		data['a']= self.a
		data['function']= self.function
		data['functionString']= self.functionString
		data['functionName']= self.functionName
		data['functionDescription']= self.functionDescription
		data['nParams']=self.nParams 
		return(data)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def importDataFromDict(self,data):
		""" 
		Import dataset from a dictionary. Intended for use in "open project" routine.
		@param data: dictionary containing a dataset to import
		@type data: dict
		"""
		specmate_singleDataset.importDataFromDict(self,data)
		self.a=data['a'] 
		self.function=data['function'] 
		self.functionString= data['functionString'] 
		self.functionName= data['functionName'] 
		self.functionDecription= data['functionDescription'] 
		self.nParams= data['nParams'] 
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def calculate(self):
		"""
		Calculate the Y-axis of the peak data based on the self.XXX parameters and store it in self.interpData. Works only if an X-axis is present in self.interpData. 
		"""
		if self.hasInterpData and self.function!=None:
			self.interpData[:,1]=self.function(self.interpData[:,0],self.a)
		else:
			self.interpData*=0.0
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	def interpolate(self,new_x):#xmin,xmax,dx):
		"""
		Set new X-Axis "new_x" and calculate fuction-data based on self.a parameters (use self.calculate()), store it in self.interpData  
		@param new_x: New X-Axis 
		@type new_x: 1-D Numpy array (monotonous)
		"""
		self.interpData=na.transpose(na.array([new_x,copy.deepcopy(new_x)]))
		#print self.interpData.shape#[:,1]*=0.0
		self.hasInterpData=True
		self.calculate()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def defineFunction(self,functionString):
		"""
		Compile a user function from a string
		"""
		self.functionString=functionString
		try:
			exec(self.functionString)
			self.function=function
		except:
			warnings.warn("Couldn't compile user function: \n"+functionString,UserWarning)
			self.functionString=None
			return(1)
		self.calculate()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def read(self,filename):
		try:
			f=open(filename)
		except IOError:
			warnings.warn("Couldn't open file: "+filename)
			return(1)

		self.filename=filename
		self.name=filename
		lines=f.readlines()
		func=""
		for each in lines:
			if len(each)<2: 
				continue
			if each.strip()[0]=='#':
				continue
			if each.split(":")[0].strip().lower()=="name":
				self.functionName=each.split(":")[1].strip()
			elif each.split(":")[0].strip().lower()=="description":
				self.functionDescription=each.split(":")[1].strip()
			elif each.split(":")[0].strip().lower()=="parameters":
				try:
					self.nParams=float(each.split(":")[1].strip())
				except:
					warnings.warn("Couldn't read number of parameters.")
			else:
				func+=each
		f.close()
		return(self.defineFunction(func))
	#dataset=specmate_functionDataset()
	#dataset.defineFunction(func)
		
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def clear(self,functionString="Empty",functionName="Empty",functionDescription="Empty",a=na.zeros((10),'Float64')):
		"""
		set up empty dataset / clear dataset.

		"""
		specmate_singleDataset.clear(self)
		self.a=a
		self.function=None
		self.functionString=functionString
		self.functionName=functionName
		self.functionDescription=functionDescription
		self.nParams=10
		if self.functionString!="Empty":
			self.defineFunction(functionString)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def __init__(self,new_x=na.array([]),functionString="Empty",functionName="Empty",functionDescription="Empty",a=na.zeros((10),'Float64')):
		"""
		"""
		self.clear(functionString,functionName,functionDescription,a)
		if len(new_x)>0:
			self.interpolate(new_x)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# end class "specmate_functionDataset"
#-----------------------------------------------------------------------------------------
