/* Sprite32 - small, cross-platform sprite library
 * Copyright (c) 1996-2003 Jeffrey T. Read
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <GL/gl.h>
#include <GL/glx.h>
#include <sprite32/sprite.h>
#include <sprite32/svppm.h>
#include <stdio.h>
#include <sys/time.h>
#include <unistd.h>				
#include <string.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>

#ifdef __USE_JOYSTICK
#ifdef HAVE_LINUX_JOYSTICK_H
#define __USE_LINUX_JOYSTICK
#include <linux/joystick.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#endif
#define IMAGE(x) ((RGBA_Image *)x)
Window wnd;
Display *display;
Window root;
int screen;
Colormap cmap;

typedef struct tagRGBA_Image {
  int width;
  int height;
  char *data;
} RGBA_Image;

int S32_SpecialKeyMap[] = {
  XK_F1,S32_Key_F1,
  XK_F2,S32_Key_F2,
  XK_F3,S32_Key_F3,
  XK_F4,S32_Key_F4,
  XK_F5,S32_Key_F5,
  XK_F6,S32_Key_F6,
  XK_F7,S32_Key_F7,
  XK_F8,S32_Key_F8,
  XK_F9,S32_Key_F9,
  XK_F10,S32_Key_F10,
  XK_F11,S32_Key_F11,
  XK_F12,S32_Key_F12,
  XK_Return,S32_Key_Enter,
  XK_Tab,S32_Key_Tab,
  XK_BackSpace,S32_Key_BS,
  XK_Up,S32_Key_Up,
  XK_Down,S32_Key_Down,
  XK_Left,S32_Key_Left,
  XK_Right,S32_Key_Right,
  XK_Home,S32_Key_Home,
  XK_End,S32_Key_End,
  XK_Page_Up,S32_Key_PgUp,
  XK_Page_Down,S32_Key_PgDn,
  XK_Insert,S32_Key_Ins,
  XK_Delete,S32_Key_Del,
  XK_Shift_L,S32_Key_Shift,
  XK_Control_L,S32_Key_Ctrl,
  XK_Alt_L,S32_Key_Alt,
  XK_Meta_L,S32_Key_Feature,
  XK_Escape,S32_Key_Escape,
  0,0
};

unsigned int nearestTexSize(int p) {
  int i = 0;
  while(p >> i) {
    i++;
  }
  return 1 << i;
}

#ifdef __USE_LINUX_JOYSTICK
int joyfd = -1;

SpriteEvent __LinuxGameHandler() {
  struct js_event je;
  SpriteEvent se;
  if(joyfd > 0) {
    if(read(joyfd,&je,sizeof(struct js_event)) >= 0) {
      switch(je.type & ~JS_EVENT_INIT) {
      case JS_EVENT_BUTTON:
	if(je.value) {
	  se.type = S32_KeyPressEvent;
	}
	else {
	  se.type = S32_KeyReleaseEvent;
	}
	se.param.code = je.number + 0x3000;
	break;
      case JS_EVENT_AXIS:
	se.type = S32_GameMoveEvent;
	se.x = je.value;
	se.y = je.number;
	break;
      }
    }
    else {
      se.type = S32_NullEvent;
    }
  }
  else {
    se.type = S32_NullEvent;
  }
  return se;
}
#endif

  
extern "C" void AsmImageCopy(void *,void *,int,int,int,int,int,int,int,int,int,int);


void BuildSI(SpriteImage *si) {
  si->cx = IMAGE(si->img)->width;
  si->cy = IMAGE(si->img)->height;
  si->depth = 32;
  si->scan_length = IMAGE(si->img)->width * 4;
  si->endian = 0;
  si->bits = IMAGE(si->img)->data;
}


Sprite  *SpriteAppWin::lastSprite() {
  Sprite  *ds = first;
  if(ds == NULL)
    return(NULL);
  else
    while (ds->next != NULL) {
      ds=ds->next;
    }
  return(ds);
}

void SpriteAppWin::addSprite(Sprite  *ds) {
  Sprite  *last;
  if (first == NULL) {
    first = ds;
    ds->prev = NULL;
  }
  else
    {
      last = lastSprite();
      ds->prev = last;
      last->next = ds;
    }
  ds->next = NULL;
}

void SpriteAppWin::deleteSprite(Sprite  *ds) {
  Sprite  *last,*next;
  if(ds->prev == NULL) {
    first = ds->next;
    if(ds->next != NULL)
      (ds->next)->prev = NULL;
  }
  else
    {
      last = ds->prev;
      last->next = ds->next;
      next = ds->next;
      if(next != NULL) {next->prev = last;}
    }
}

void SpriteAppWin::placeSpriteBehind(Sprite  *ds1,Sprite  *ds2) {
  if(ds1 != ds2) {
    Sprite  *p;
    deleteSprite(ds1);
    if(ds2 == NULL) {addSprite(ds1);}
    else
      {
	p = ds2->prev;
	ds1->prev = p;
	ds1->next = ds2;
	ds2->prev = ds1;
	if(p != NULL) {p->next = ds1;} else {first = ds1;}
      }
  }
}
SpriteAppWin::SpriteAppWin(char *_title,int _cx,int _cy) {
  XGCValues gcv;
  XVisualInfo *vi;
  GLXContext ctx;
  XTextProperty tp;
  XSetWindowAttributes attr;
  unsigned long mask;
  char *envptr,*envptr2,*envptr3;
  int attrib[] = {
    GLX_RGBA,
      GLX_RED_SIZE,1,
      GLX_GREEN_SIZE,1,
      GLX_BLUE_SIZE,1,
      GLX_DEPTH_SIZE,2,
      GLX_DOUBLEBUFFER,
      None };
  display = XOpenDisplay(NULL);
#ifdef __USE_LINUX_JOYSTICK
  if(joyfd < 0) joyfd = open("/dev/js0",O_RDONLY | O_NONBLOCK);
#endif
  int maj,min,pix;
  screen = DefaultScreen(display);
  vi = glXChooseVisual(display,screen,attrib);
  cx = _cx;
  cy = _cy;
  extra = NULL;
  title = _title;
  first = NULL;
  iter = NULL; iter2 = NULL;
  rentype = RENDER_IMAGE;
  quitting = 0;
  frameskip = 1;
  dyn_frameskip = 0;
  clock_delay = 30;
  cmap = DefaultColormap(display,screen);
  root = DefaultRootWindow(display);
  attr.background_pixel = 0;
  attr.border_pixel = 0;
  /* TODO: share root colormap if possible */
  attr.colormap = XCreateColormap( display, root, vi->visual, AllocNone);
  attr.event_mask = StructureNotifyMask | ExposureMask | KeyPressMask | KeyReleaseMask | PointerMotionMask | ButtonPressMask | ButtonReleaseMask;
  mask = CWBackPixel | CWBorderPixel | CWColormap | CWEventMask;
  
  wnd = XCreateWindow( display, root, 0, 0, _cx, _cy,
		       0, vi->depth, InputOutput,
		       vi->visual, mask, &attr );
  XStringListToTextProperty(&title,1,&tp);
  XSetWMName(display,wnd,&tp);
  XStoreName(display,wnd,_title);
  XMapRaised(display,wnd);
  XAutoRepeatOff(display);
  XSelectInput(display,wnd,attr.event_mask);
  gcv.function = GXcopy;
  ctx = glXCreateContext( display, vi, NULL, True );
  glXMakeCurrent(display,wnd,ctx);
  glClearColor( 0.0,0.0,0.0, 1.0 );
  glClear( GL_COLOR_BUFFER_BIT );
  glViewport( 0, 0, _cx,_cy );
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho( 0, _cx, _cy,0, -1.0, 1.0 );
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glEnable(GL_TEXTURE_2D);
  glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_DECAL);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_FASTEST);
  glShadeModel( GL_FLAT );
  envptr = getenv("SPRITE32_CLOCK_DELAY");
  envptr2 = getenv("SPRITE32_FRAME_SKIP");
  envptr3 = getenv("SPRITE32_DYN_FRAME_SKIP");
  if(envptr != NULL) {
    clock_delay = atoi(envptr);
  }
  if(envptr2 != NULL) {
    frameskip = atoi(envptr2);
  }
  if(envptr3 != NULL) {
    dyn_frameskip = atoi(envptr3);
  }
  Back.img = malloc(sizeof(RGBA_Image));
  IMAGE(Back.img)->width = nearestTexSize(_cx);
  IMAGE(Back.img)->height = nearestTexSize(_cy);
  IMAGE(Back.img)->data = (char *)malloc(nearestTexSize(_cx) * nearestTexSize(_cy) * 4);
  Buf.img = malloc(sizeof(RGBA_Image));
  IMAGE(Buf.img)->width = nearestTexSize(_cx);
  IMAGE(Buf.img)->height = nearestTexSize(_cy);
  IMAGE(Buf.img)->data = (char *)malloc(nearestTexSize(_cx) * nearestTexSize(_cy) * 4);
  BuildSI(&Back);
  BuildSI(&Buf);
  gcv.graphics_exposures = False;
}

SpriteAppWin::~SpriteAppWin() {
  quit();
  XDestroyWindow(display,wnd);
  XAutoRepeatOn(display);
  XCloseDisplay(display);
}

void SpriteAppWin::quit() {
  quitting = 1;
}


void SpriteAppWin::run() {
XEvent evt;
quitting = 0;
#if 0
  while(!quitting) {
    XNextEvent(display,&evt);
      if(evt.xany.window != wnd) continue;
      if((evt.type == Expose)) {
        break;
    }
  }
#endif
while(!quitting) {
    SpriteEvent se,se2;
    while(QLength(display) > 0) {
      if(XCheckWindowEvent(display,wnd,0xffffffff,&evt)) {
	_massageEvent(&evt);
      }
    }
#ifdef __USE_LINUX_JOYSTICK
    se2.type = S32_NullEvent;
    do {
      se2 = __LinuxGameHandler();
      handleEvent(&se2);
    } while(se2.type != S32_NullEvent);
#endif
    se.type = S32_RefreshEvent;
    handleEvent(&se);
    _spriteLoop();
    glEnable(GL_TEXTURE_2D);
    glTexImage2D(GL_TEXTURE_2D,0,4,(GLsizei)Buf.cx,(GLsizei)Buf.cy,0,GL_RGBA,GL_UNSIGNED_BYTE,(GLvoid *)(IMAGE(Buf.img)->data));
    glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_DECAL);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_FASTEST);
    glBegin(GL_QUADS);
    glNormal3f(0,0,-1);
    glTexCoord2f(0.0,0.0);
    glVertex2i(0,0);
    glTexCoord2f(1.0,0.0);
    glVertex2i(Buf.cx,0);
    glTexCoord2f(1.0,1.0);
    glVertex2i(Buf.cx,Buf.cy);
    glTexCoord2f(0.0,1.0);
    glVertex2i(0,Buf.cy);
    glEnd();
    glFlush();
    glXSwapBuffers(display,wnd);
}
}

int SpriteAppWin::loadSpriteImage(char *data,SpriteImage *si) {

RGBA_Image *theimg = (RGBA_Image *)malloc(sizeof(RGBA_Image));

ReadPpmAsRgba(data,&(theimg->width),&(theimg->height),(unsigned char **)&(theimg->data));
si->img = (void *)theimg;
BuildSI(si);
}

void SpriteAppWin::destroySpriteImage(SpriteImage *si) {
  free(IMAGE(si->img)->data);
  free(IMAGE(si->img));
}


//Main sprite loop.
void SpriteAppWin::_spriteLoop() {
  struct timeval t,tv1,tv2;
  long tim;
  int q;
  int msec;
  gettimeofday(&tv1,NULL);
  for(int i=0;i<Buf.cy;i++) {
    memcpy(IMAGE(Buf.img)->data + i * Buf.scan_length,
	   IMAGE(Back.img)->data + i * Back.scan_length,
	   Buf.scan_length);
  }
  //Loop through all the available sprites, move, and render. The iterators are
  //stored in the object so ~Sprite() can massage them out of the SIGSEGV
  //danger zone.
  if(frameskip <= 0) {
    frameskip = 1;
  }
  for(q=0;q<frameskip;q++) {
    for(iter = first;iter != NULL;iter = iter->next) {
      iter->move();
      if(iter->autoColTest()) {
	for(iter2 = first;iter2 != NULL;iter2 = iter2->next) {
	  if(iter2 != iter) {
	    if(iter->collisionTest(iter2)) iter->onCollision(iter2);
	  }
	}
      }
    }
  }
  for(iter = first;iter != NULL;iter = iter->next) {
      iter->render(&Buf);
  }
  gettimeofday(&tv2,NULL);
  tim = (tv2.tv_sec - tv1.tv_sec) * 1000;
  if(tv2.tv_usec < tv1.tv_usec) {
    tim += 1000 - ((tv1.tv_usec - tv2.tv_usec) / 1000);
  }
  else {
    tim += (tv2.tv_usec - tv1.tv_usec) / 1000;
  }
  msec = tim / 1000;
  if(msec < clock_delay) {
    t.tv_sec = 0;
    t.tv_usec = (clock_delay - msec) * 1000;
    //select(0,NULL,NULL,NULL,&t);
    usleep((clock_delay - msec) * 1000);
  }
  else 
    if(dyn_frameskip && clock_delay != 0) {
      frameskip = (msec  / clock_delay);
    }

}

void SpriteAppWin::_massageEvent(void *e) {
  XEvent *evt = (XEvent *)e;
  SpriteEvent se,se2;
  KeySym ks;
  switch(evt->type) {
//  case Expose:
//    if(rentype == RENDER_IMAGE) {
//      for(int i=0;i<IMAGE(Buf.img)->height;i++) {
//	memcpy(IMAGE(Buf.img)->data + i * IMAGE(Buf.img)->bytes_per_line,
//	       IMAGE(Back.img)->data + i * IMAGE(Back.img)->bytes_per_line,
//	       IMAGE(Buf.img)->bytes_per_line);
//      }
//    }
//    break;
  case KeyPress:
    ks = XLookupKeysym((XKeyEvent *)evt,0);
    se.x = 0;
    se.y = 0;
    if(ks >= 0x20 && ks <= 0x7f) {
      se.param.code = ks;
    }
    else {
      for(int i=0;S32_SpecialKeyMap[i] != 0;i += 2) {
	if(ks == S32_SpecialKeyMap[i]) {
	  se.param.code = S32_SpecialKeyMap[i + 1];
	}
      }
    }
    se.type = S32_KeyPressEvent;
    handleEvent(&se);
    break;
  case KeyRelease:
    ks = XLookupKeysym((XKeyEvent *)evt,0);
    se.x = 0;
    se.y = 0;
    if(ks >= 0x20 && ks <= 0x7f) {
      se.param.code = ks;
    }
    else {
      for(int i=0;S32_SpecialKeyMap[i] != 0;i += 2) {
	if(ks == S32_SpecialKeyMap[i]) {
	  se.param.code = S32_SpecialKeyMap[i + 1];
	}
      }
    }
    se.type = S32_KeyReleaseEvent;
    handleEvent(&se);
    break;
  case MotionNotify:
    se.type = S32_MouseMoveEvent;
    se.x = evt->xmotion.x;
    se.y = evt->xmotion.y;
    se.param.code = evt->xmotion.state >> 8;
    handleEvent(&se);
    break;
  case ButtonPress:
    se.type = S32_KeyPressEvent;
    se.x = evt->xbutton.x;
    se.y = evt->xbutton.y;
    se.param.code = evt->xbutton.button + S32_Key_Button1 - Button1;
    handleEvent(&se);
    break;
  case ButtonRelease:
    se.type = S32_KeyReleaseEvent;
    se.x = evt->xbutton.x;
    se.y = evt->xbutton.y;
    se.param.code = evt->xbutton.button + S32_Key_Button1 - Button1;
    handleEvent(&se);
    break;
  }
}

void SpriteAppWin::handleEvent(SpriteEvent *se) {
}


void Sprite::render(SpriteImage *si) {
  if(simg != NULL) {
    if(maxframes < 2) {
      ImageCopy(simg,si,0,0,x-hx,y-hy,cx,cy,SIMG_USE_KEY,key);
    }
    else {
      ImageCopy(simg,si,0,cy * frame,x-hx,y-hy,cx,cy,SIMG_USE_KEY,key);
      if(auto_animate) {
	count++; if(count >= delay) {
	  count = 0;
	  frame++; frame %= maxframes;
	}
      }
    }
  }
}

Sprite::Sprite(SpriteAppWin *sw,float _x,float _y,SpriteImage *si,unsigned int mf,unsigned int d) {
  host = sw;
  x = _x;
  y = _y;
  vx = 0; vy = 0;
  hx = 0; hy = 0;
  maxframes = mf;
  frame = 0;
  delay = d;
  unsigned long dummy1;
  int dummy2;
  unsigned int dummy3,_cx,_cy;
  simg = si;
    _cx = IMAGE(simg->img)->width;
    _cy = IMAGE(simg->img)->height;
    int b = 4;
    key = 0;
    void *q = &key;
    switch(b) {
    case 1:
      *((char *)q) = *((char *)(IMAGE(si->img)->data));
      break;
    case 2:
      *((short *)q) = *((short *)(IMAGE(si->img)->data));
      break;
    case 4:
      *((long *)q) = *((long *)(IMAGE(si->img)->data));
      break;
    }
  cx = (int)_cx;
  cy = (int)_cy;
  calculateSize();
  host->addSprite(this);
  auto_hittest = False;
  auto_animate=0;
}


void Sprite::calculateSize() {
    cy = simg->cy;
    cy /= maxframes;
}

Sprite::~Sprite() {
  //XFreePixmap(host->display,pxm);
  //XFreePixmap(host->display,maskpxm);
  host->deleteSprite(this);
  if(host->iter == this) {
    host->iter = nextSprite();
  }
  if(host->iter2 == this) {
    host->iter2 = nextSprite();
  }
  if(host->iter == NULL) {
    host->iter = host->first;
  }
  if(host->iter2 == NULL) {
    host->iter2 = host->first;
  }

}

void Sprite::moveTo(float newx,float newy) {
  x = newx; y = newy;
}
void Sprite::move() {
  x += vx; y += vy;
}

void Sprite::setMotion(float _vx,float _vy) {
  vx = _vx; vy = _vy;
}


void Sprite::changeShape(SpriteImage *si) {
  simg = si;
  if(si == NULL) {
    cx = 0; cy = 0; return;
  }
  cx = si->cx;
  cy = si->cy;
  int b = 4;
  key = 0;
  void *q = &key;
  switch(b) {
  case 1:
    *((char *)q) = *((char *)(IMAGE(si->img)->data));
    break;
  case 2:
    *((short *)q) = *((short *)(IMAGE(si->img)->data));
    break;
  case 4:
    *((long *)q) = *((long *)(IMAGE(si->img)->data));
    break;
  }
  calculateSize();
};


//Do some simple collision checking by comparing the rectangles of
//the source and destination sprites for overlap.
int Sprite::collisionTest(Sprite *s) {
  int step1,step2; //Intermediate x and y proximity values.
  int x1 = x - hx;
  int y1 = y - hy;
  int x2 = s->xPos() - s->xHotspot();
  int y2 = s->yPos() - s->yHotspot();
  //Step 1: Check for horizontal proximity.
  if(x1 <= x2) {
    if(x2 - x1 < cx) step1 = True; else step1 = False;
  }
  else {
    if(x2 + s->xSize () > x1) step1 = True; else step1 = False;
  }
  //Step 2: Check for vertical proximity.
  if(y1 <= y2) {
    if(y2 - y1 < cy) step2 = True; else step2 = False;
  }
  else {
    if(y2 + s->ySize() > y1) step2 = True; else step2 = False;
  }
  //Return true if we are overlapping horizontally AND vertically.
  return(step1 && step2);
}

//Default, do-nothing procedure for collision notification.
void Sprite::onCollision(Sprite *spr) {
}

//Nuke. Replaces the Win32 delete this functionality. DO NOT use delete this
//in a MoveSprite function under unix!

//NOTE: This has changed as delete this is now safe due to a newly implemented
//iterator loop. So Nuke() simply calls delete this.
void Sprite::nuke() {
  delete this;
}
