/* Sprite32 - small, cross-platform sprite library
 * Copyright (c) 1996-2003 Jeffrey T. Read
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sprite32/sprite.h>
#include <stdio.h>
#include <sys/time.h>
#include <unistd.h>				
#include <string.h>

#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#include <sprite32/svppm.h>

#ifdef __USE_JOYSTICK
#ifdef HAVE_LINUX_JOYSTICK_H
#define __USE_LINUX_JOYSTICK
#include <linux/joystick.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#endif
#define IMAGE(x) ((XImage *)x)
Window wnd;
Display *display;
Window root;
int screen;
Colormap cmap;

#ifdef __SPRITE_DGA
#include <X11/extensions/xf86dga.h>
#include <X11/extensions/xf86vmode.h>

char *MemBase;
char *SoftwareBuffer;
int BankSize;
int BufferOffset;
int SecondPageStart;
int Ram;
int PhysWidth;
int hw = 0;
int scx,scy;



void HardwareDoubleBuffer(SpriteAppWin *sw)

{
        if(BufferOffset)
        {
	  BufferOffset = 0;
   	  XF86DGASetViewPort(display,
			     DefaultScreen(display),0,scy);
        }
        else
        {  
	  BufferOffset = SecondPageStart;
   	  XF86DGASetViewPort(display,
			     DefaultScreen(display),0,0);
       }
}

void SoftwareDoubleBuffer(SpriteAppWin *sw)
{
  for(int i=0;i<(sw->Buffer().cy);i++) {
    memcpy(MemBase + i * IMAGE(sw->Buffer().img)->bytes_per_line,
	   SoftwareBuffer + i * IMAGE(sw->Buffer().img)->bytes_per_line,
	   IMAGE(sw->Buffer().img)->bytes_per_line);
  }
}

#endif /* __SPRITE_DGA */
int S32_SpecialKeyMap[] = {
  XK_F1,S32_Key_F1,
  XK_F2,S32_Key_F2,
  XK_F3,S32_Key_F3,
  XK_F4,S32_Key_F4,
  XK_F5,S32_Key_F5,
  XK_F6,S32_Key_F6,
  XK_F7,S32_Key_F7,
  XK_F8,S32_Key_F8,
  XK_F9,S32_Key_F9,
  XK_F10,S32_Key_F10,
  XK_F11,S32_Key_F11,
  XK_F12,S32_Key_F12,
  XK_Return,S32_Key_Enter,
  XK_Tab,S32_Key_Tab,
  XK_BackSpace,S32_Key_BS,
  XK_Up,S32_Key_Up,
  XK_Down,S32_Key_Down,
  XK_Left,S32_Key_Left,
  XK_Right,S32_Key_Right,
  XK_Home,S32_Key_Home,
  XK_End,S32_Key_End,
  XK_Page_Up,S32_Key_PgUp,
  XK_Page_Down,S32_Key_PgDn,
  XK_Insert,S32_Key_Ins,
  XK_Delete,S32_Key_Del,
  XK_Shift_L,S32_Key_Shift,
  XK_Control_L,S32_Key_Ctrl,
  XK_Alt_L,S32_Key_Alt,
  XK_Meta_L,S32_Key_Feature,
  XK_Escape,S32_Key_Escape,
  0,0
};

#ifdef __USE_LINUX_JOYSTICK
int joyfd = -1;

SpriteEvent __LinuxGameHandler() {
  struct js_event je;
  SpriteEvent se;
  if(joyfd > 0) {
    if(read(joyfd,&je,sizeof(struct js_event)) >= 0) {
      switch(je.type & ~JS_EVENT_INIT) {
      case JS_EVENT_BUTTON:
	if(je.value) {
	  se.type = S32_KeyPressEvent;
	}
	else {
	  se.type = S32_KeyReleaseEvent;
	}
	se.param.code = je.number + 0x3000;
	break;
      case JS_EVENT_AXIS:
	se.type = S32_GameMoveEvent;
	se.x = je.value;
	se.y = je.number;
	break;
      }
    }
    else {
      se.type = S32_NullEvent;
    }
  }
  else {
    se.type = S32_NullEvent;
  }
  return se;
}
#endif

XColor defpalette[256];
  
extern "C" void AsmImageCopy(void *,void *,int,int,int,int,int,int,int,int,int,int);


void BuildSI(SpriteImage *si) {
  si->cx = IMAGE(si->img)->width;
  si->cy = IMAGE(si->img)->height;
  si->depth = IMAGE(si->img)->bits_per_pixel;
  si->scan_length = IMAGE(si->img)->bytes_per_line;
  si->endian = IMAGE(si->img)->byte_order;
  si->bits = IMAGE(si->img)->data;
}

Sprite  *SpriteAppWin::lastSprite() {
  Sprite  *ds = first;
  if(ds == NULL)
    return(NULL);
  else
    while (ds->next != NULL) {
      ds=ds->next;
    }
  return(ds);
}

void SpriteAppWin::addSprite(Sprite  *ds) {
  Sprite  *last;
  if (first == NULL) {
    first = ds;
    ds->prev = NULL;
  }
  else
    {
      last = lastSprite();
      ds->prev = last;
      last->next = ds;
    }
  ds->next = NULL;
}

void SpriteAppWin::deleteSprite(Sprite  *ds) {
  Sprite  *last,*next;
  if(ds->prev == NULL) {
    first = ds->next;
    if(ds->next != NULL)
      (ds->next)->prev = NULL;
  }
  else
    {
      last = ds->prev;
      last->next = ds->next;
      next = ds->next;
      if(next != NULL) {next->prev = last;}
    }
}

void SpriteAppWin::placeSpriteBehind(Sprite  *ds1,Sprite  *ds2) {
  if(ds1 != ds2) {
    Sprite  *p;
    deleteSprite(ds1);
    if(ds2 == NULL) {addSprite(ds1);}
    else
      {
	p = ds2->prev;
	ds1->prev = p;
	ds1->next = ds2;
	ds2->prev = ds1;
	if(p != NULL) {p->next = ds1;} else {first = ds1;}
      }
  }
}
SpriteAppWin::SpriteAppWin(char *_title,int _cx,int _cy) {
  XGCValues gcv;
  XTextProperty tp;
  char *envptr,*envptr2,*envptr3;
  int clr,b,dep;
  display = XOpenDisplay(NULL);
#ifdef __USE_LINUX_JOYSTICK
  if(joyfd < 0) joyfd = open("/dev/js0",O_RDONLY | O_NONBLOCK);
#endif
#ifdef __SPRITE_DGA
  XF86VidModeModeLine modeline;
#else
  XShmSegmentInfo *shminfo;
#endif
  int maj,min,pix;
  screen = DefaultScreen(display);
  cx = _cx;
  cy = _cy;
  extra = NULL;
  title = _title;
  first = NULL;
  iter = NULL; iter2 = NULL;
  rentype = RENDER_IMAGE;
  quitting = 0;
  frameskip = 1;
  dyn_frameskip = 0;
  clock_delay = 30;
//  cmap = DefaultColormap(display,screen);
  root = DefaultRootWindow(display);
  wnd = XCreateSimpleWindow(display,root,0,0,_cx,_cy,1,
			    BlackPixel(display,screen),BlackPixel(display,screen));
  envptr2 = getenv("SPRITE32_FRAME_SKIP");
  envptr3 = getenv("SPRITE32_DYN_FRAME_SKIP");
  dep = DefaultDepth(display,screen);
  b = dep / 8;
  if(dep == 15) b++;
  if(b < 2) {
  cmap = XCreateColormap(display,wnd,DefaultVisual(display,screen),AllocAll);
  for(clr=0;clr<256;clr++) {
    defpalette[clr].pixel = clr;
    defpalette[clr].red = clr & 0xE0 << 8;
    defpalette[clr].green = (clr & 0x1c) << 11;
    defpalette[clr].blue = (clr & 0x03) << 14;
    defpalette[clr].red = clr << 8;
    defpalette[clr].green = clr << 8;
    defpalette[clr].blue = clr << 8;
    defpalette[clr].flags = DoRed | DoGreen | DoBlue;
    XStoreColor(display,cmap,&(defpalette[clr]));
  }
  XInstallColormap(display,cmap);
  XSetWindowColormap(display,wnd,cmap);
  }
  envptr = getenv("SPRITE32_CLOCK_DELAY");
  if(envptr != NULL) {
    clock_delay = atoi(envptr);
  }
  if(envptr2 != NULL) {
    frameskip = atoi(envptr2);
  }
  if(envptr3 != NULL) {
    dyn_frameskip = atoi(envptr3);
  }
  if(!XShmQueryVersion(display,&maj,&min,&pix)) {
    exit(1);
  }
#ifdef __SPRITE_DGA
  dep = DefaultDepth(display,screen);
  b = dep / 8;
  if(dep == 15 || dep == 24) b++;
  envptr = getenv("SPRITE32_CLOCK_DELAY");
  char *bgimg;
  char *bfimg;
  Back.img = XCreateImage(display,DefaultVisual(display,screen),dep,ZPixmap,0,NULL,
			  _cx,_cy,8,0);
  bgimg = (char *)malloc((_cy + 1) * IMAGE(Back.img)->bytes_per_line);
  IMAGE(Back.img)->data = bgimg;
  if(!XF86DGAQueryVersion(display,&maj,&min)) {
    exit(1);
  }
  XF86VidModeGetModeLine(display,screen,&pix,&modeline);
  scx = modeline.hdisplay;
  scy = modeline.vdisplay;
  if(scx < _cx || scy < _cy) {
    fprintf(stderr,"error: screen too small\n");
    exit(1);
  }
  XF86DGAGetVideo(display,screen,&MemBase,&PhysWidth,
		  &BankSize, &Ram);
  XF86DGADirectVideo(display,screen,
		     XF86DGADirectGraphics);
  SecondPageStart = PhysWidth*b*scy;
  BufferOffset = 0;
  if((SecondPageStart * 2) > (Ram * 1024)) {
    hw = 0;
    Buf.img = XCreateImage(display,DefaultVisual(display,screen),dep,ZPixmap,0,NULL,
			   _cx,_cy,8,0);
    SoftwareBuffer = (char *)malloc((_cy + 1) * IMAGE(Buf.img)->bytes_per_line);
    IMAGE(Buf.img)->data = SoftwareBuffer;
  }
  else {
    hw = 1;
    Buf.img = XCreateImage(display,DefaultVisual(display,screen),dep,ZPixmap,0,NULL,
			   PhysWidth,scy,8,0);
    IMAGE(Buf.img)->data = MemBase + SecondPageStart;
  }
  XF86DGASetViewPort(display,screen,0,0);
#else
  extra = malloc(sizeof(XShmSegmentInfo));
  shminfo = (XShmSegmentInfo *)extra;
  dep = DefaultDepth(display,screen);
  b = dep / 8;
  if(dep == 15 || dep == 24) b++;
  char *bgimg;
  char *bfimg;
  Back.img = XCreateImage(display,DefaultVisual(display,screen),dep,ZPixmap,0,NULL,
			  _cx,_cy,8,0);
  Buf.img = XShmCreateImage(display,DefaultVisual(display,screen),dep,ZPixmap,NULL,
			    shminfo,_cx,_cy);
  bgimg = (char *)malloc((_cy + 1) * IMAGE(Back.img)->bytes_per_line);
  IMAGE(Back.img)->data = bgimg;
  shminfo->shmid = shmget(IPC_PRIVATE,IMAGE(Buf.img)->bytes_per_line * (_cy + 1),
			  IPC_CREAT|0777);
  if((int)(shminfo->shmid) == -1) {
    perror("libsprite32: cannot get shared segment");
    exit(-1);
  }
  shminfo->shmaddr = (char *)shmat(shminfo->shmid,0,0);
  if((int)(shminfo->shmaddr) == -1) {
    perror("libsprite32: cannot attach shared memory");
    exit(-1);
  }
  IMAGE(Buf.img)->data = shminfo->shmaddr;
  shminfo->readOnly = false;
  XShmAttach(display,shminfo);
#endif
  BuildSI(&Back);
  BuildSI(&Buf);
  
  XStringListToTextProperty(&title,1,&tp);
  XSetWMName(display,wnd,&tp);
  XStoreName(display,wnd,_title);
  XMapRaised(display,wnd);
  XSelectInput(display,wnd,ExposureMask | KeyPressMask | KeyReleaseMask | ButtonPressMask | ButtonReleaseMask | PointerMotionMask);
  gcv.function = GXcopy;
  gcv.graphics_exposures = False;
}

SpriteAppWin::~SpriteAppWin() {
  quit();
#ifdef __SPRITE_DGA
    XF86DGADirectVideo(display,screen,0);
#else
    if(extra != NULL) {
      XShmSegmentInfo *shminfo = (XShmSegmentInfo *)extra;
      shmid_ds sds;
      shmctl(shminfo->shmid,IPC_RMID,&sds);
      XShmDetach(display,shminfo);
      shmdt(shminfo->shmaddr);
      free(extra);
    }
#endif
  XDestroyWindow(display,wnd);
  XCloseDisplay(display);
}

void SpriteAppWin::quit() {
  quitting = 1;
}


void SpriteAppWin::run() {
  XEvent evt;
  quitting = 0;
#if 0
  while(!quitting) {
    XNextEvent(display,&evt);
    if(evt.xany.window != wnd) continue;
    if((evt.type == Expose)) {
      break;
    }
  }
#endif
  while(!quitting) {
    SpriteEvent se,se2;
    while(QLength(display) > 0) {
      if(XCheckWindowEvent(display,wnd,0xffffffff,&evt)) {
	_massageEvent(&evt);
      }
    }
#ifdef __USE_LINUX_JOYSTICK
    se2.type = S32_NullEvent;
    do {
      se2 = __LinuxGameHandler();
      handleEvent(&se2);
    } while(se2.type != S32_NullEvent);
#endif
    se.type = S32_RefreshEvent;
    handleEvent(&se);
    _spriteLoop();
#ifdef __SPRITE_DGA
      if(hw) {
	HardwareDoubleBuffer(this);
	IMAGE(Buf.img)->data = MemBase + (BufferOffset);
      }
      else {
	SoftwareDoubleBuffer(this);
      }
#else
      XShmPutImage(display,wnd,DefaultGC(display,screen),IMAGE(Buf.img),0,0,0,0,cx,cy,0);
#endif
    XSync(display,False);
  }
}

int SpriteAppWin::loadSpriteImage(char *data,SpriteImage *si) {
  int i;
  int _cx,_cy;
  unsigned char *ptr;
  si->img = XCreateImage(display,DefaultVisual(display,screen),DefaultDepth(display,screen),ZPixmap,0,NULL,
			  0,0,8,0);
  ReadPpmRgbConverted(data,&_cx,&_cy,&ptr,IMAGE(si->img)->bits_per_pixel);
  IMAGE(si->img)->width = _cx;
  IMAGE(si->img)->height = _cy;
  IMAGE(si->img)->data = (char *)ptr;
  IMAGE(si->img)->bytes_per_line = _cx * ((IMAGE(si->img)->bits_per_pixel) >> 3);
  si->cx = IMAGE(si->img)->width;
  si->cy = IMAGE(si->img)->height;
  si->depth = IMAGE(si->img)->bits_per_pixel;
  si->scan_length = IMAGE(si->img)->bytes_per_line;
  si->endian = IMAGE(si->img)->byte_order;
  si->bits = IMAGE(si->img)->data;
  return i;
}

void SpriteAppWin::destroySpriteImage(SpriteImage *si) {
  XDestroyImage(IMAGE(si->img));
}


//Main sprite loop.
void SpriteAppWin::_spriteLoop() {
  struct timeval t,tv1,tv2;
  long tim;
  int q;
  int msec;
  gettimeofday(&tv1,NULL);
  ImageCopy(&Back,&Buf,0,0,0,0,Buf.cx,Buf.cy,0,0);
  /*  for(int i=0;i<IMAGE(Buf.img)->height;i++) {
    memcpy(IMAGE(Buf.img)->data + i * IMAGE(Buf.img)->bytes_per_line,
	   IMAGE(Back.img)->data + i * IMAGE(Back.img)->bytes_per_line,
	   IMAGE(Buf.img)->bytes_per_line);
	   }*/
  //Loop through all the available sprites, move, and render. The iterators are
  //stored in the object so ~Sprite() can massage them out of the SIGSEGV
  //danger zone.
  if(frameskip <= 0) {
    frameskip = 1;
  }
  for(q=0;q<frameskip;q++) {
    for(iter = first;iter != NULL;iter = iter->next) {
      iter->move();
      if(iter->autoColTest()) {
	for(iter2 = first;iter2 != NULL;iter2 = iter2->next) {
	  if(iter2 != iter) {
	    if(iter->collisionTest(iter2)) iter->onCollision(iter2);
	  }
	}
      }
    }
  }
  for(iter = first;iter != NULL;iter = iter->next) {
      iter->render(&Buf);
  }
  gettimeofday(&tv2,NULL);
  tim = (tv2.tv_sec - tv1.tv_sec) * 1000;
  if(tv2.tv_usec < tv1.tv_usec) {
    tim += 1000 - ((tv1.tv_usec - tv2.tv_usec) / 1000);
  }
  else {
    tim += (tv2.tv_usec - tv1.tv_usec) / 1000;
  }
  msec = tim / 1000;
  if(msec < clock_delay) {
    t.tv_sec = 0;
    t.tv_usec = (clock_delay - msec) * 1000;
    //select(0,NULL,NULL,NULL,&t);
    usleep((clock_delay - msec) * 1000);
  }
  else 
    if(dyn_frameskip && clock_delay != 0) {
      frameskip = (msec  / clock_delay);
    }

}

void SpriteAppWin::_massageEvent(void *e) {
  XEvent *evt = (XEvent *)e;
  SpriteEvent se,se2;
  KeySym ks;
  switch(evt->type) {
  case Expose:
    if(rentype == RENDER_IMAGE) {
      for(int i=0;i<IMAGE(Buf.img)->height;i++) {
	memcpy(IMAGE(Buf.img)->data + i * IMAGE(Buf.img)->bytes_per_line,
	       IMAGE(Back.img)->data + i * IMAGE(Back.img)->bytes_per_line,
	       IMAGE(Buf.img)->bytes_per_line);
      }
    }
    break;
  case KeyPress:
    ks = XLookupKeysym((XKeyEvent *)evt,0);
    se.x = 0;
    se.y = 0;
    if(ks >= 0x20 && ks <= 0x7f) {
      se.param.code = ks;
    }
    else {
      for(int i=0;S32_SpecialKeyMap[i] != 0;i += 2) {
	if(ks == S32_SpecialKeyMap[i]) {
	  se.param.code = S32_SpecialKeyMap[i + 1];
	}
      }
    }
    se.type = S32_KeyPressEvent;
    handleEvent(&se);
    break;
  case KeyRelease:
    ks = XLookupKeysym((XKeyEvent *)evt,0);
    se.x = 0;
    se.y = 0;
    if(ks >= 0x20 && ks <= 0x7f) {
      se.param.code = ks;
    }
    else {
      for(int i=0;S32_SpecialKeyMap[i] != 0;i += 2) {
	if(ks == S32_SpecialKeyMap[i]) {
	  se.param.code = S32_SpecialKeyMap[i + 1];
	}
      }
    }
    se.type = S32_KeyReleaseEvent;
    handleEvent(&se);
    break;
  case MotionNotify:
    se.type = S32_MouseMoveEvent;
    se.x = evt->xmotion.x;
    se.y = evt->xmotion.y;
    se.param.code = evt->xmotion.state >> 8;
    handleEvent(&se);
    break;
  case ButtonPress:
    se.type = S32_KeyPressEvent;
    se.x = evt->xbutton.x;
    se.y = evt->xbutton.y;
    se.param.code = evt->xbutton.button + S32_Key_Button1 - Button1;
    handleEvent(&se);
    break;
  case ButtonRelease:
    se.type = S32_KeyReleaseEvent;
    se.x = evt->xbutton.x;
    se.y = evt->xbutton.y;
    se.param.code = evt->xbutton.button + S32_Key_Button1 - Button1;
    handleEvent(&se);
    break;
  }
}

void SpriteAppWin::handleEvent(SpriteEvent *se) {
}


//SpriteImage-based sprite renderer.
void Sprite::render(SpriteImage *si) {
  if(simg != NULL) {
    if(maxframes < 2) {
      ImageCopy(simg,si,0,0,x-hx,y-hy,cx,cy,SIMG_USE_KEY,key);
    }
    else {
      ImageCopy(simg,si,0,cy * frame,x-hx,y-hy,cx,cy,SIMG_USE_KEY,key);
      if(auto_animate) {
	count++; if(count >= delay) {
	  count = 0;
	  frame++; frame %= maxframes;
	}
      }
    }
  }
}

Sprite::Sprite(SpriteAppWin *sw,float _x,float _y,SpriteImage *si,unsigned int mf,unsigned int d) {
  host = sw;
  x = _x;
  y = _y;
  vx = 0; vy = 0;
  hx = 0; hy = 0;
  maxframes = mf;
  frame = 0;
  delay = d;
  count = 0;
  auto_animate = 0;
  unsigned long dummy1;
  int dummy2;
  unsigned int dummy3,_cx,_cy;
  simg = si;
  _cx = simg->cx;
  _cy = simg->cy;
  int b = simg->depth / 8;
  key = 0;
  void *q = &key;
  switch(b) {
  case 1:
    *((char *)q) = *((char *)(si->bits));
    break;
  case 2:
    *((short *)q) = *((short *)(si->bits));
    break;
  case 3:
    *((char *)q) = *((char *)(si->bits));
    *((char *)q+1) = *((char *)(si->bits)+1);
    *((char *)q+2) = *((char *)(si->bits)+2);
    break;
  case 4:
    *((long *)q) = *((long *)(si->bits));
    break;
    }
  cx = (int)_cx;
  cy = (int)_cy;
  calculateSize();
  host->addSprite(this);
  auto_hittest = 0;
}

void Sprite::calculateSize() {
    cy = simg->cy;
    cy /= maxframes;
}

Sprite::~Sprite() {
  host->deleteSprite(this);
  if(host->iter == this) {
    host->iter = nextSprite();
  }
  if(host->iter2 == this) {
    host->iter2 = nextSprite();
  }
  if(host->iter == NULL) {
    host->iter = host->first;
  }
  if(host->iter2 == NULL) {
    host->iter2 = host->first;
  }

}

void Sprite::moveTo(float newx,float newy) {
  x = newx; y = newy;
}

void Sprite::move() {
  x += vx; y += vy;
}


void Sprite::setMotion(float _vx,float _vy) {
  vx = _vx; vy = _vy;
}


void Sprite::changeShape(SpriteImage *si) {
  simg = si;
  if(si == NULL) {
    cx = 0; cy = 0; return;
  }
  cx = (int)(IMAGE(si->img)->width);
  cy = (int)(IMAGE(si->img)->height);
  int b = IMAGE(si->img)->bits_per_pixel / 8;
  key = 0;
  void *q = &key;
  switch(b) {
  case 1:
    *((char *)q) = *((char *)(IMAGE(si->img)->data));
    break;
  case 2:
    *((short *)q) = *((short *)(IMAGE(si->img)->data));
    break;
  case 4:
    *((long *)q) = *((long *)(IMAGE(si->img)->data));
    break;
  }
  calculateSize();
};


//Do some simple collision checking by comparing the rectangles of
//the source and destination sprites for overlap.
int Sprite::collisionTest(Sprite *s) {
  int step1,step2; //Intermediate x and y proximity values.
  int x1 = x - hx;
  int y1 = y - hy;
  int x2 = s->xPos() - s->xHotspot();
  int y2 = s->yPos() - s->yHotspot();
  //Step 1: Check for horizontal proximity.
  if(x1 <= x2) {
    if(x2 - x1 < cx) step1 = True; else step1 = False;
  }
  else {
    if(x2 + s->xSize () > x1) step1 = True; else step1 = False;
  }
  //Step 2: Check for vertical proximity.
  if(y1 <= y2) {
    if(y2 - y1 < cy) step2 = True; else step2 = False;
  }
  else {
    if(y2 + s->ySize() > y1) step2 = True; else step2 = False;
  }
  //Return true if we are overlapping horizontally AND vertically.
  return(step1 && step2);
}

//Default, do-nothing procedure for collision notification.
void Sprite::onCollision(Sprite *spr) {
}

void Sprite::nuke() {
  delete this;
}
