! srend_tile_montage.F90   2016jul09
! --------------------------------------------------------------------
! Copyright (c) 2016 Ted Wetherbee
!
! Permission is hereby granted, free of charge, to any person 
! obtaining a copy of this software and associated documentation files 
! (the "Software"), to deal in the Software without restriction, 
! including without limitation the rights to use, copy, modify, merge,
! publish, distribute, sublicense, and/or sell copies of the Software,
! and to permit persons to whom the Software is furnished to do so, 
! subject to the following conditions:
!
! The above copyright notice and this permission notice shall be
! included in all copies or substantial portions of the Software.
!
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
! EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
! MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
! NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
! BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
! ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
! CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
! SOFTWARE.
! --------------------------------------------------------------------
!
! build: gfortran srend_tile_montage.F90 -O3 -o montage
!
! $./montage filein  fileout [alpha beta]
!    0       1       2        3     4
! --------------------------------------------------------------------
! Option: [alpha beta]
! This command line program converts a spherical lon/lat image to
! perspective through bilinear sampling.  The angle spans for the
! spherical and perspective images are the same, and they must both
! be less than 180 degrees to make sense.
!
! If tiles were rendered in spherical view, then these must be packed
! to 1 image first.  The Imagemagick routine 'montage' would pack PPM
! image tiles, say this situation where 64 tiles form an image with 
! 2 rows and 2 columns
!     tile_001_001.ppm   tile_001_002.ppm
!     tile_002_001.ppm   tile_002_002.ppm
! $ montage tile_*.ppm -tile 2x2 -geometry +0+0 -depth 8 out.ppm
!
! This time_montage code is for use putting image tiles together which
! were created by srend vis code.
!      2016feb25  Ted Wetherbee  ted@fdltcc.edu
! --------------------------------------------------------------------

program tm ! ---------------------------------------------------------
implicit NONE
character*256 :: filein, fileout, alpha_arg, beta_arg
integer*4 :: nargs
real*4 :: alpha, beta
integer*4 :: W, H ! image width and height
character*1,dimension(:,:,:),allocatable :: im ! result image
character*1,dimension(:,:),allocatable :: tileR ! a row of tile data
character*20 :: ppm_header
logical :: fexist
integer*4 :: TR, TC ! tile rows, tile columns
integer*4 :: TW, TH ! tile width and tile height in pixels
integer*4 :: ir, ic, j ! index vars
character*16 :: tileRC ! string to hold _RRR_CCC 
character*256 :: f ! scratch filename
! ---------------- end declarations ----------------------------------

! get command line arguments
nargs = command_argument_count()
if(nargs .ne. 2 .AND. nargs .ne. 4) then
   print *,'-----------------------------------------------------------------------------'
   print *,'srend_tile_montage (stm):                                                          '
   print *,'usage: stm filein fileout [alpha beta]                                        '
   print *,' - compile: gfortran srend_tile_montage.F90 -O3 -o stm                        '
   print *,' - input files must be PPM type, and probably have a .ppm extension          ' 
   print *,' - argument filein is the prefix before: _RRR_CCC.ppm                        '
   print *,' - input PPMs must exactly 20 byte header:  P6 W H 255 achar(20)=LF          '
   print *,' - input PPMs have exactly this form: filename_RRR_CCC.ppm                   '
   print *,' - the output file will be PPM type and should have a .ppm extension         '
   print *,' - alpha and beta are the angular width and height values in degrees of both,'
   print *,'   and both these must be less than 180 degrees.  These should be the same   '
   print *,'   values used rendering with srend.                                         '
   print *,'-----------------------------------------------------------------------------'
   STOP
end if

call get_command_argument(1,filein)
call get_command_argument(2,fileout)
if(nargs == 4) then
   call get_command_argument(3,alpha_arg); read(alpha_arg,*) alpha
   call get_command_argument(4,beta_arg) ; read(beta_arg,*) beta
end if

! find number of tile rows TR
TR = 0 ! tile rows
do ir = 1,999
   write(tileRC,'("_",I3.3,"_",I3.3)') ir, 1
   f = '' ! clear
   f = trim(filein) // trim(tileRC) // '.ppm'
   inquire(FILE=f,EXIST=fexist)
   if(.NOT. fexist) then
     if(ir == 1) then
        print *,'No row 1 ... abort.'
        STOP
     end if
     exit ! loop
   end if     
   TR = ir ! = number of tile rows after loop exit
end do

! find number of tile cols TC
TC = 1 ! tile cols, must have been 1 from above test
do ic = 2,999
   write(tileRC,'("_",I3.3,"_",I3.3)') 1, ic
   f = '' ! clear
   f = trim(filein) // trim(tileRC) // '.ppm'
   inquire(FILE=f,EXIST=fexist)
   if(.NOT. fexist) exit ! loop
   TC = ic ! = number of tile rows after loop exit
end do

! read header from known filein: filein // ' _001_001.ppm
f = '' ! clear
f = trim(filein) // '_001_001.ppm'
open(100,FILE=f,ACCESS="stream")
read(100) ppm_header
close(100)
read(ppm_header(4:20),*) TW, TH ! gets TW=tile width and TH=tile height

W = TC*TW ! image width
H = TR*TH ! image height
allocate( im(1:3,1:W,1:H) )
allocate( tileR(1:3,TW) ) 

! read in tiles and write to im
do ir= 1,TR
   do ic=1,TC
      write(tileRC,'("_",I3.3,"_",I3.3)') ir, ic
      f = ''
      f = trim(filein) // trim(tileRC) // '.ppm'
      open(100+ic,FILE=f,ACCESS="stream")
      read(100+ic) ppm_header ! read past header
   end do
   do j=1,TH
      do ic=1,TC
        read(100+ic) tileR
        im(1:3,1+(ic-1)*TW:ic*TW,j+(ir-1)*TH) = tileR
      end do
   end do
   do ic=1,TC
      close(100+ic)
   end do
end do

! do this if perspective
if(nargs == 4) call s2p_sub(im,W,H,alpha,beta)

! write header and data out
write(ppm_header,'("P6  ",I5," ",I5," 255")') W, H
open(99,FILE=fileout,ACCESS="stream")
write(99) ppm_header(1:19), achar(10)
write(99) im
close(99)

end program tm ! ----------------------------------------------------


! the data array is modified in place according to parameters W,H,alpha,beta
subroutine s2p_sub(im, W, H, Alpha, Beta)
implicit NONE
character*1,dimension(1:3,1:W,1:H) :: im
integer*4 :: W, H
real*4 :: Alpha, Beta
real*4, dimension(1:3,0:W+1,0:H+1) :: ppm_buffer2 ! 2 scratch arrays for calculations
real*4 :: t
real*4 :: dW, dH, dWW, dHH, a_span, b_span, aa, bb, aaa, bbb, ddx0, ddx1, ddy0, ddy1
integer*4 :: i,j,k,ii,jj
real*4 :: PI
! ====================================================================
      PI = 4.0 * atan(1.0)
! ------------- copy im(:,:,:,mhe_nR) section to ppm_buffer2
      do j=1,H
      do i=1,W
      do k=1,3
        ppm_buffer2(k,i,j) = ichar(im(k,i,j)) ! character*1 to real
      end do
      end do
      end do
! pad top and bottom row of pixels in ppm_buffer2
      ppm_buffer2(1:3,1:W,0) = ppm_buffer2(1:3,1:W,1)
      ppm_buffer2(1:3,1:W,H+1) = ppm_buffer2(1:3,1:W,H)
! pad sides, padded array corners are data corners
      ppm_buffer2(1:3,0,0:H+1) = ppm_buffer2(1:3,1,0:H+1)
      ppm_buffer2(1:3,W+1,0:H+1) = ppm_buffer2(1:3,W,0:H+1)
! viewing increments, assume W and H are divisible by 4
      dW = Alpha * PI/180.0 / W
      dH = Beta  * PI/180.0 / H
! convert to radians, no shrink .. have padding
      a_span = Alpha * PI / 180.0
      b_span = Beta  * PI / 180.0

      dWW = 2.0*tan(.5*a_span)/W
      dHH = 2.0*tan(.5*b_span)/H
      do j=1,H
        bb = (H/2.0 - j)*dHH + dHH/2.0 ! angle coords, persp
      do i=1,W
        aa = (W/2.0 - i)*dWW + dWW/2.0 ! angle coords, persp
        aaa = atan(aa)               ! project
        bbb = atan( bb / sqrt(aa*aa + 1.0) )
! interpolation: bilinear
        ddx0 = W/2.0 - (aaa - dW/2.0)/dW ! hit spherical pix
        ii = floor(ddx0)
        ddx0 = ddx0 - ii
        ddx1 = 1.0 - ddx0
         
        ddy0 = H/2.0 - (bbb - dH/2.0)/dH
        jj = floor(ddy0)
        ddy0 = ddy0 - jj
        ddy1 = 1.0 - ddy0
        do k=1,3
           t =  ppm_buffer2(k,ii,jj)    *ddx1*ddy1 +  & ! bilinear interpolation
                ppm_buffer2(k,ii+1,jj)  *ddx0*ddy1 +  &
                ppm_buffer2(k,ii,jj+1)  *ddx1*ddy0 +  &
                ppm_buffer2(k,ii+1,jj+1)*ddx0*ddy0
           t = min(255.0, t) ! safety
           t = max(0.0  , t) ! safety
           im(k,i,j) = char(floor(t))
        end do
      end do
      end do

end subroutine s2p_sub