#!/usr/bin/python

# Copyright (C) 2006-2007 Arnau Sanchez
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation, 
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Standard Python modules
import os, sys, pexpect

##################################
##################################
class SSHClientError(Exception):
	def __init__(self, value):
		self.value = value
	def __str__(self):
		return repr(self.value)
		
##################################
##################################
class SSHClient:
	_timeout = 120
	_ssh_path = "/usr/bin/ssh"
	_prompt = "(%|#|\\$) $"
	_yesno = {False: "no", True: "yes"}
	_ssh_port = 22
	_debug = False
	##################################
	def __init__(self, ssh_path=_ssh_path, timeout=_timeout, prompt=_prompt, \
			debug=_debug):
		self.ssh_path = ssh_path
		self.timeout = timeout
		self.prompt = prompt
		self.debug = debug
		self.name_debug = None

	#######################################
	def pdebug(self, line):
		if not self.debug: return
		if self.name_debug:
			line = "%s: %s"%(self.name_debug, line)
		sys.stderr.write(line+"\n")
		sys.stderr.flush()

	##################################
	def connect(self, command, password=None, passphrase=None):
		self.pdebug("spawning ssh command with pexpect: %s"%command)
		self.child = pexpect.spawn(command)
		while 1:
			responses = ['assword:', 'nter passphrase', r"yes/no", self.prompt]
			self.pdebug("wait for responses: %s"%responses)
			try: 
				index = self.child.expect(responses, timeout=self.timeout)
			except pexpect.TIMEOUT: 
				raise SSHClientError("timeout reached")
			except pexpect.EOF: 
				raise SSHClientError("connect error")
			self.pdebug("matches: '%s'" %(responses[index]))
			if index == 0:
				if password == None: 
					raise SSHClientError("need password, but not given")
				self.pdebug("send password")
				self.child.sendline(password)
			elif index == 1:
				if passphrase == None: 
					raise SSHClientError("need passphrase, but not given")
				self.pdebug("send passphrase")
				self.child.sendline(passphrase)
			elif index == 2:
				self.pdebug("send 'yes'")
				self.child.sendline("yes")
			elif index == 3:
				self.pdebug("prompt received")
				break
		self.pdebug("connected")
		
	##################################
	def command(self, command):
		if not command: return
		self.pdebug("sending command: %s" %command)
		self.child.sendline(command)
		self.child.expect(self.prompt)
		response = self.child.before
		self.pdebug("command response: %s" %response)
		return response

	##################################
	def close(self):
		self.pdebug("closing connection")
		self.child.close()
