/* fcc-search -- searches FCC database entries.
 *
 * Copyright (C) 2002 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <glib.h>
#include <locale.h>
#include <fcc.h>
#include <getopt.h>
#include <guess-where.h>
#include <math.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <utils.h>
#include <values.h>
#include <sys/stat.h>
#include <sys/mman.h>

static const int new_file = O_CREAT | O_WRONLY | O_TRUNC;

static Sinfo *am_sinfo;
static int am_n_sinfo;

static Sinfo *fm_sinfo;
static int fm_n_sinfo;

static Sinfo *tv_sinfo;
static int tv_n_sinfo;

int32_t
rad2rot(double rad)
{
  return rad * ((INT32_MAX + 1.) / M_PI);
}

double
rot2rad(int32_t rot)
{
  return rot * (M_PI / (INT32_MAX + 1.));
}

double
sinfo_get_power(Sinfo *sinfo)
{
  double h, v;
  switch (sinfo->sloc.band)
    {
    case band_am:
      return sinfo->am->power;
    case band_fm:
      h = sinfo->fm->horiz_erp;
      v = sinfo->fm->vert_erp;
      return h > v ? h : v;
    case band_tv:
      return sinfo->tv->erp;
    }
  return -1;
}

double
sinfo_get_ant_rot(Sinfo *sinfo)
{
  switch (sinfo->sloc.band)
    {
    case band_am:
      return 0;
    case band_fm:
      return sinfo->fm->ant_rot;
    case band_tv:
      return sinfo->tv->ant_rot;
    }
  return -1;
}

int
sinfo_get_ant_id(Sinfo *sinfo)
{
  switch (sinfo->sloc.band)
    {
    case band_am:
      return sinfo->am->ant_sys_id;
    case band_fm:
      return sinfo->fm->ant_id;
    case band_tv:
      return sinfo->tv->ant_id;
    }
  return -1;
}

static int
uint_cmp(const void *v1, const void *v2)
{
  const unsigned int *ui1 = v1;
  const unsigned int *ui2 = v2;
  return *ui1 - *ui2;
}

static int
map_file(char *path, char **map, int *fd)
{
  struct stat stat;
  if ((*fd = open(path, O_RDONLY)) < 0)
    {
      warn("can't open %s", path);
      return 0;
    }

  fstat(*fd, &stat);
  *map = mmap(NULL, stat.st_size, PROT_READ, MAP_PRIVATE, *fd, 0);
  if (*map == NULL)
    {
      warn("can't mmap %s", path);
      return 0;
    }
  return 1;
}

/*
 * ant_fill_am_pattern -- computes an AM antenna radiation pattern
 * based on the tower position and phase.  Algorithm courtesy of John
 * Byrns (jbyrns@pyrotechnics.com).
 */
static void
ant_fill_am_pattern(Ant *ant, Sinfo *sinfo)
{
  int i, j, points = 32;
  double refx, refy, imag, real, total_phase, trms;
  Pattern *pat;
  Tower_set *ts = ant->tower_set;

  if (ts == NULL)
    return;

  // printf("\nTower set %d: %d towers, %d relative\n",
  //   ts->id, ts->towers, ts->rel);

  pat = ant->pattern = fmalloc(sizeof(*ant->pattern));
  pat->min = MAXFLOAT;
  pat->max = 0;
  pat->rms = 1;
  pat->n_points = points;
  pat->azimuth = fmalloc(points * sizeof(*pat->azimuth));
  pat->magnitude = fmalloc(points * sizeof(*pat->magnitude));

  refx = refy = 0.0;
  for (i = 0; i < ts->towers; i++)
    {
      Tower *tower = ts->tower[i];
      double orient = tower->orient;

      // printf("Tower %d: ref=%d, ", i, tower->ref_switch);
      if (tower->ref_switch == 1)
	{
	  refx += tower->spacing * sin(orient);
	  refy += tower->spacing * cos(orient);
	  tower->spacing = sqrt(refx * refx + refy * refy);
	  tower->orient = acos(refy / tower->spacing);
	  if (refx < 0.0)
	    tower->orient = 2*M_PI - tower->orient;
	  tower->ref_switch = 0;
          // printf("refx=%+5.3f, refy=%+5.3f; acos(%+5.3f)\n",
          //   refx, refy, refy / tower->spacing);
	}
      else
	{
	  refx = tower->spacing * sin(orient);
	  refy = tower->spacing * cos(orient);
          // printf("refx=%+5.3f, refy=%+5.3f\n", refx, refy);
	}
    }
  trms = 0;
  for (j = 0; j < points; j++)
    {
      pat->azimuth[j] = j * 2*M_PI / points;
      imag = real = 0.0;
      for (i = 0; i < ts->towers; i++)
	{
	  Tower *tower = ts->tower[i];
	  total_phase = tower->phasing +
	    tower->spacing * cos(tower->orient - pat->azimuth[j]);
	  real += tower->field_ratio * cos(total_phase);
	  imag += tower->field_ratio * sin(total_phase);
	}
      pat->magnitude[j] = sqrt(real * real + imag * imag);
      trms += (pat->magnitude[j] * pat->magnitude[j]);
      if (pat->max < pat->magnitude[j])
	pat->max = pat->magnitude[j];
      if (pat->min > pat->magnitude[j])
	pat->min = pat->magnitude[j];
      // printf("%2d: %5.1f %+5.3f\n",
      //   j, rad2deg(pat->azimuth[j]), pat->magnitude[j] / trms);
    }
  trms = sqrt(trms / points);
}

// FIX THIS: The following two routines are similar enough that much
// of the duplicate code could probably be merged.

static Ant *
sinfo_get_am_ant(Sinfo *sinfo)
{
  int fd, id;
  Ant *ant;
  Ts_off *ts_offs;
  FILE *stream;
  size_t stride = sizeof(*ts_offs);
  char *map = NULL;
  struct stat stat;
  static const int tower_set_fields = 28;
  char *field[tower_set_fields], line[1000];

  id = sinfo_get_ant_id(sinfo);
  if (id < 0)
    return NULL;

  if (!map_file(PKGDATADIR "/am_towers.sip", &map, &fd))
    return NULL;

  fstat(fd, &stat);
  ts_offs = bsearch(&id, map, stat.st_size / stride, stride, uint_cmp);
  if (ts_offs == NULL)
    {
      warn("can't find tower set %d in am_towers.sip\n", id);
      close(fd);
      return NULL;
    }

  ant = ant_new();
  ant->id = id;
  ant->ts_off = ts_offs->ts_off;
  close(fd);

  stream = fopen(PKGDATADIR "/am_towers.dat", "r");
  if (stream == NULL)
    {
      warn("can't open " PKGDATADIR "/am_towers.dat");
      return NULL;
    }

  ant->tower_set = fmalloc0(sizeof(*ant->tower_set));
  ant->tower_set->id = id;
  ant->tower_set->ts_off = ts_offs->ts_off;

  fseek(stream, ts_offs->ts_off, SEEK_SET);
  while (1)
    {
      fgets(line, sizeof(line), stream);
      split(field, tower_set_fields, line, '|');
      if (! tower_set_fill_from_fields(ant->tower_set, field))
	break;
    } 
  fclose(stream);

  return ant;
}

static Ant *
sinfo_get_fm_tv_ant(Sinfo *sinfo)
{
  int fd, id;
  Ant *ant;
  Ant_offs *ant_offs;
  FILE *stream;
  size_t stride = sizeof(Ant_offs);
  char *map = NULL;
  struct stat stat;
  static const int make_fields = 8, make_len = 500;
  char *field[make_fields], line[make_len];

  id = sinfo_get_ant_id(sinfo);
  if (id < 0)
    return NULL;

  if (!map_file(PKGDATADIR "/ant_make_pat.sip", &map, &fd))
    return NULL;

  fstat(fd, &stat);
  ant_offs = bsearch(&id, map, stat.st_size / stride, stride, uint_cmp);
  if (ant_offs == NULL)
    {
      warn("can't find antenna %d in am_make_pat.sip\n", id);
      close(fd);
      return NULL;
    }

  ant = ant_new();
  ant->id = id;
  ant->make_off = ant_offs->make_off;
  ant->pattern->pat_off = ant_offs->pat_off;
  close(fd);

  stream = fopen(PKGDATADIR "/ant_pattern.dat", "r");
  if (stream == NULL)
    {
      warn("can't open " PKGDATADIR "/ant_pattern.dat");
      return NULL;
    }

  fseek(stream, ant->pattern->pat_off, SEEK_SET);
  while (1)
    {
      fgets(line, sizeof(line), stream);
      split(field, make_fields, line, '|');
      if (! ant_pattern_fill_from_fields(ant, field))
	break;
    } 
  fclose(stream);

  stream = fopen(PKGDATADIR "/ant_make.dat", "r");
  if (stream == NULL)
    {
      warn("can't open " PKGDATADIR "/ant_make.dat");
      return NULL;
    }
  fseek(stream, ant->make_off, SEEK_SET);
  fgets(line, sizeof(line), stream);
  fclose(stream);
  split(field, make_fields, line, '|');
  ant_make_fill_from_fields(ant, field);

  return ant;
}

Ant *
sinfo_get_ant(Sinfo *sinfo)
{
  Ant *ant;
  switch (sinfo->sloc.band)
    {
    case band_am:
      ant = sinfo_get_am_ant(sinfo);
      ant_fill_am_pattern(ant, sinfo);
      return ant;
    case band_fm:
    case band_tv:
      ant = sinfo_get_fm_tv_ant(sinfo);
      return ant;
    }
  return NULL;
}

static void
sloc_ingest(Sinfo **sinfo_ptr, int *n_sinfo_ptr, char *path)
{
  FILE *fp;
  struct stat stat;
  int n, n_sinfo = 0;
  Sinfo *sinfo = NULL;

  fp = fopen(path, "r");
  if (fp == NULL)
      warn("can't open %s", path);
  else
    {
      fstat(fileno(fp), &stat);
      n = stat.st_size / sizeof(Sloc);
      sinfo = fmalloc0(n * sizeof(Sinfo));

      for (n_sinfo = 0; n_sinfo < n; n_sinfo++)
	if (fread(sinfo + n_sinfo, sizeof(Sloc), 1, fp) <= 0)
	  break;
      fclose(fp);
    }

  *sinfo_ptr = sinfo;
  *n_sinfo_ptr = n_sinfo;
}

/*
 * sinfo_is_near -- each sinfo has a lat, lon, corf, and a set of
 * status flags for old, new, and ancilliary stations.  These are
 * compared against the values requested in the search record.  The
 * sinfo is returned if it meets that search criterion; if not, null
 * is returned.
 */
static Sinfo *
sinfo_is_near(Sinfo *sinfo, Search *search)
{
  double corf;
  Sloc *sloc = &sinfo->sloc;

  if (!search->show_new && sloc->new) return NULL;
  if (!search->show_old && sloc->old) return NULL;
  if (!search->show_anc && sloc->anc) return NULL;

  corf = sloc->corf / 10.0;
  if (corf < search->corf_min || search->corf_max < corf)
    return NULL;

  gc(search->lat, search->lon, rot2rad(sloc->lat), rot2rad(sloc->lon),
      &sinfo->arc, &sinfo->head);
  if (sinfo->arc < search->dist_min || search->dist_max < sinfo->arc)
    return NULL;

  return sinfo;
}

static void
fac_fill(Fac *fac, char *rec)
{
  static const int fac_fields = 28, fac_len = 500;
  char *field[fac_fields], line[fac_len];

  memccpy(line, rec, '\n', fac_len);
  split(field, fac_fields, line, '|');

  fac_fill_from_fields(fac, field);
}

static Fac *
sinfo_fill_fac(Sinfo *sinfo, char *fac_map)
{
  static GHashTable *fac_hash;

  if (fac_hash == NULL)
    fac_hash= g_hash_table_new(g_int_hash, g_int_equal);

  sinfo->fac = g_hash_table_lookup(fac_hash, &sinfo->sloc.fac_off);
  if (sinfo->fac == NULL)
    {
      sinfo->fac = fmalloc0(sizeof(*sinfo->fac));
      g_hash_table_insert(fac_hash, &sinfo->sloc.fac_off, sinfo->fac);
      fac_fill(sinfo->fac, fac_map + sinfo->sloc.fac_off);
      sinfo->fac->fac_off = sinfo->sloc.fac_off;
      sinfo->sloc.anc = service_to_anc(sinfo->fac->service);
      sinfo->sloc.new = callsign_to_new(sinfo->fac->callsign);
    }

  return sinfo->fac;
}

static void
sinfo_split_fill(Sinfo *sinfo, char *rec, void (*filler)(Sinfo *, char **))
{
  static const int fields = 100, len = 1000;
  char *field[fields], line[len];

  memccpy(line, rec, '\n', len);
  split(field, fields, line, '|');

  filler(sinfo, field);
}

static Sinfo *
sinfo_fill_am(Sinfo *sinfo, char *am_map)
{
  static GHashTable *am_hash;

  if (am_hash == NULL)
    am_hash = g_hash_table_new(g_int_hash, g_int_equal);

  sinfo->am = g_hash_table_lookup(am_hash, &sinfo->sloc.eng_off);
  if (sinfo->am == NULL)
    {
      sinfo->am = fmalloc0(sizeof(*sinfo->am));
      g_hash_table_insert(am_hash, &sinfo->sloc.eng_off, sinfo->am);
      sinfo_split_fill(sinfo, 
	am_map + sinfo->sloc.eng_off, sinfo_am_fill_from_fields);
    }

  return sinfo;
}

static Sinfo *
sinfo_fill_fm(Sinfo *sinfo, char *fm_map)
{
  static GHashTable *fm_hash;

  if (fm_hash == NULL)
    fm_hash= g_hash_table_new(g_int_hash, g_int_equal);

  sinfo->fm = g_hash_table_lookup(fm_hash, &sinfo->sloc.eng_off);
  if (sinfo->fm == NULL)
    {
      sinfo->fm = fmalloc0(sizeof(*sinfo->fm));
      g_hash_table_insert(fm_hash, &sinfo->sloc.eng_off, sinfo->fm);
      sinfo_split_fill(sinfo,
	fm_map + sinfo->sloc.eng_off, sinfo_fm_fill_from_fields);
    }

  return sinfo;
}

static Sinfo *
sinfo_fill_tv(Sinfo *sinfo, char *tv_map)
{
  static GHashTable *tv_hash;

  if (tv_hash == NULL)
    tv_hash= g_hash_table_new(g_int_hash, g_int_equal);

  sinfo->tv = g_hash_table_lookup(tv_hash, &sinfo->sloc.eng_off);
  if (sinfo->tv == NULL)
    {
      sinfo->tv = fmalloc0(sizeof(*sinfo->tv));
      g_hash_table_insert(tv_hash, &sinfo->sloc.eng_off, sinfo->tv);
      sinfo_split_fill(sinfo,
	tv_map + sinfo->sloc.eng_off, sinfo_tv_fill_from_fields);
    }

  return sinfo;
}

static Sinfo **
sinfo_near(Search *search, int *n_sinfo_ptr)
{
  char *fac_map, *am_map, *fm_map, *tv_map;
  int fac_fd, am_fd, fm_fd, tv_fd, r, n_sinfo = 0;
  Sinfo **sinfos = NULL;

  if (!map_file(PKGDATADIR "/facility.dat", &fac_map, &fac_fd))
    return NULL;

  if (search->show_am)
    {
      if (am_sinfo == NULL)
	sloc_ingest(&am_sinfo, &am_n_sinfo, PKGDATADIR "/am_ant_sys.si");
      if (map_file(PKGDATADIR "/am_ant_sys.dat", &am_map, &am_fd))
	{
	  for (r = 0; r < am_n_sinfo; r++)
	    {
	      Sinfo *sinfo = am_sinfo + r;
	      if (sinfo_is_near(sinfo, search))
		{
		  sinfos = frealloc(sinfos, (n_sinfo + 1) * sizeof(*sinfos));
		  sinfos[n_sinfo++] = sinfo;
		  sinfo_fill_fac(sinfo, fac_map);
		  sinfo_fill_am(sinfo, am_map);
		}
	    }
	  close(am_fd);
	}
    }

  if (search->show_fm)
    {
      if (fm_sinfo == NULL)
	sloc_ingest(&fm_sinfo, &fm_n_sinfo, PKGDATADIR "/fm_eng_data.si");
      if (map_file(PKGDATADIR "/fm_eng_data.dat", &fm_map, &fm_fd))
	{
	  for (r = 0; r < fm_n_sinfo; r++)
	    {
	      Sinfo *sinfo = fm_sinfo + r;
	      if (sinfo_is_near(sinfo, search))
		{
		  sinfos = frealloc(sinfos, (n_sinfo + 1) * sizeof(*sinfos));
		  sinfos[n_sinfo++] = sinfo;
		  sinfo_fill_fac(sinfo, fac_map);
		  sinfo_fill_fm(sinfo, fm_map);
		}
	    }
	  close(fm_fd);
	}
    }

  if (search->show_tv)
    {
      if (tv_sinfo == NULL)
	sloc_ingest(&tv_sinfo, &tv_n_sinfo, PKGDATADIR "/tv_eng_data.si");
      if (map_file(PKGDATADIR "/tv_eng_data.dat", &tv_map, &tv_fd))
	{
	  for (r = 0; r < tv_n_sinfo; r++)
	    {
	      Sinfo *sinfo = tv_sinfo + r;
	      if (sinfo_is_near(sinfo, search))
		{
		  sinfos = frealloc(sinfos, (n_sinfo + 1) * sizeof(*sinfos));
		  sinfos[n_sinfo++] = sinfo;
		  sinfo_fill_fac(sinfo, fac_map);
		  sinfo_fill_tv(sinfo, tv_map);
		}
	    }
	  close(tv_fd);
	}
    }

  close(fac_fd);
  *n_sinfo_ptr = n_sinfo;
  return sinfos;
}

int
sinfo_compare_freq(const void *a, const void *b)
{
  Sinfo *sa = *(Sinfo **)a;
  Sinfo *sb = *(Sinfo **)b;
  return sa->khz - sb->khz;
}

int
sinfo_compare_dist(const void *a, const void *b)
{
  Sinfo *sa = *(Sinfo **)a;
  Sinfo *sb = *(Sinfo **)b;
  return (sb->arc < sa->arc) - (sa->arc < sb->arc);
}

int
sinfo_compare_power(const void *a, const void *b)
{
  Sinfo *sa = *(Sinfo **)a;
  Sinfo *sb = *(Sinfo **)b;
  float pa = sinfo_get_power(sa);
  float pb = sinfo_get_power(sb);
  return (pb < pa) - (pa < pb);
}

/*
 * local search: an ugly work-around.  The search struct has to be
 * passed to the comparison functions used by the qsort function.  The
 * cleanest way to do this is to nest these comparison functions
 * inside the sort_and_show_sinfo() function, but this only works with
 * gcc.  To accomodate lesser compilers, we'll instead use a file
 * scope local copy of the search struct pointer for the comparison
 * functions to use.  This isn't reentrant, but that shouldn't be a
 * problem here.
 */
static Search *local_search;

static int 
sinfo_compare(const void *a, const void *b)
{
  int c = 0, f = 0;
  while (f < N_SORT_FUNCS && local_search->sort_func[f] && 
    (c = local_search->sort_func[f](a, b)) == 0)
    f++;
  return c * local_search->sort_order[f];
}

static int
keep_compare(const void *a, const void *b)
{
  int freq_order = sinfo_compare_freq(a, b);
  return freq_order ? freq_order : local_search->keep_func(a, b);
}

static void
sort_and_show_sinfo(Search *search, Sinfo **sinfos, int n_sinfos, 
  Sinfos_display sinfos_display)
{
  static Sinfo **kept;
  int s, k, n_kept, last_freq;

  if (sinfos == NULL)
    {
      sinfos_display(search, sinfos, -1, -1);
      return;
    }

  local_search = search;
  if (n_sinfos < search->keep)
    {
      qsort(sinfos, n_sinfos, sizeof(*sinfos), sinfo_compare);
      sinfos_display(search, sinfos, n_sinfos, n_sinfos);
      return;
    }

  qsort(sinfos, n_sinfos, sizeof(*sinfos), keep_compare);
  if (kept)
    g_free(kept);
  kept = NULL;
  
  for (k = s = n_kept = last_freq = 0; s < n_sinfos; s++)
    {
      Sinfo *sinfo = sinfos[s];
      
      if (last_freq != sinfo->khz)
	{
	  last_freq = sinfo->khz;
	  k = 0;
	}

      if (k < search->keep)
	{
	  kept = g_realloc(kept, (n_kept + 1) * sizeof(*kept));
	  kept[n_kept] = sinfo;
	  n_kept++;
	  k++;
	}
    }
  qsort(kept, n_kept, sizeof(*kept), sinfo_compare);
  sinfos_display(search, kept, n_kept, n_sinfos);
}

int
search_sort_and_show_sinfo(Search *search, Sinfos_display sinfos_display)
{
  int n;
  Sinfo **sinfos = sinfo_near(search, &n);
  sort_and_show_sinfo(search, sinfos, n, sinfos_display);
  return n;
}

static void
check_offset(const char *map, int fd, off_t off)
{
  struct stat stat;
  fstat(fd, &stat);
  if (off < 0 || stat.st_size <= off || (off != 0 && map[off - 1] != '\n'))
    fatal("file offset %ld is not a valid start of line", off);
}

static Sinfo *
sinfo_fill_eng(Sinfo *sinfo, char *fn, off_t off,
  Sinfo * (*sinfo_eng_filler)(Sinfo *, char *))
{
  int fd;
  char *map;
  if (map_file(fn, &map, &fd))
    {
      check_offset(map, fd, off);
      sinfo_eng_filler(sinfo, map);
      close(fd);
      return sinfo;
    }
  return NULL;
}

Sinfo *
sinfo_fill(band_enum band, off_t fac_off, off_t eng_off)
{
  int fac_fd;
  char *fac_map;

  if (!map_file(PKGDATADIR "/facility.dat", &fac_map, &fac_fd))
    return NULL;
  check_offset(fac_map, fac_fd, fac_off);

  Sinfo *sinfo = fmalloc0(sizeof(*sinfo));
  sinfo->sloc.fac_off = fac_off;
  sinfo->sloc.eng_off = eng_off;
  sinfo_fill_fac(sinfo, fac_map);
  close(fac_fd);

  switch (band)
    {
    case band_am:
      return sinfo_fill_eng(sinfo,
	PKGDATADIR "/am_ant_sys.dat", eng_off, sinfo_fill_am);

    case band_fm:
      return sinfo_fill_eng(sinfo,
	PKGDATADIR "/fm_eng_data.dat", eng_off, sinfo_fill_fm);

    case band_tv:
      return sinfo_fill_eng(sinfo,
	PKGDATADIR "/tv_eng_data.dat", eng_off, sinfo_fill_tv);
    }

  return NULL;
}
