/* fcc.h -- routines to access information from the FCC database files.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef FCC_H
#define FCC_H

#include <fcntl.h>
#include <stdint.h>
#include <sys/time.h>

typedef struct
{
  int ref_switch;
  float field_ratio, orient, phasing, spacing;
}
Tower;

typedef struct
{
  int id, towers, rel;
  off_t ts_off;
  Tower **tower;
}
Tower_set;
extern Tower_set **tower_set;
extern int tower_set_alloc;

typedef struct
{
  int n_points;
  off_t pat_off;
  float min, max, rms, *azimuth, *magnitude;
}
Pattern;

typedef struct
{
  int id;
  off_t make_off, ts_off;
  char *make, *model, *service, *standard, *updated;
  Pattern *pattern;
  Tower_set *tower_set;
}
Ant;
extern Ant **antenna;
extern int ant_alloc;

Ant *ant_new(void);
void ant_free(Ant *ant);

void ant_make_fill_from_fields(Ant *ant, char **field);
int ant_pattern_fill_from_fields(Ant *ant, char **field);
int tower_set_fill_from_fields(Tower_set *tower_set, char **field);

typedef struct
{
  int fac_id, fac_off, channel;
  double freq;
  char *callsign, *comm_city, *comm_state, *service;
  char *fac_addr1, *fac_addr2, *fac_city, *fac_state, *fac_country, *fac_zip;
}
Fac;

void fac_fill_from_fields(Fac *fac, char **field);

extern int app_alloc, *app2fac;

typedef struct
{
  char ant_type;
  int ant_sys_id, towers;
  float power;
}
Am;

typedef struct
{
  int ant_id;
  char ant_type, ant_pol;
  float ant_rot, horiz_erp, vert_erp, eamsl;
}
Fm;

typedef struct
{
  int ant_id;
  char ant_type, ant_pol;
  float ant_rot, erp, eamsl;
}
Tv;

typedef struct
{
  int id;
  off_t make_off, pat_off;
}
Ant_offs;

typedef struct
{
  int id;
  off_t ts_off;
}
Ts_off;

typedef enum { band_am, band_fm, band_tv } band_enum;

typedef struct
{
  off_t eng_off, fac_off;
  int32_t lat, lon;
  int16_t corf;
  uint8_t new:1, old:1, anc:1, band:2;
}
Sloc;

typedef struct _Sinfo
{
  Sloc sloc;
  double arc, head;
  int khz;
  Fac *fac;
  Am *am;
  Fm *fm;
  Tv *tv;
}
Sinfo;

int32_t rad2rot(double rad);
double rot2rad(int32_t rot);
double sinfo_get_power(Sinfo *sinfo);
double sinfo_get_ant_rot(Sinfo *sinfo);
int service_to_anc(char *service);
int callsign_to_new(char *callsign);
int sinfo_get_ant_id(Sinfo *sinfo);
Ant *sinfo_get_ant(Sinfo *sinfo);

int tv_channel_to_khz(int chan);
void sinfo_am_fill_from_fields(Sinfo *sinfo, char **field);
void sinfo_fm_fill_from_fields(Sinfo *sinfo, char **field);
void sinfo_tv_fill_from_fields(Sinfo *sinfo, char **field);

extern Sinfo **am, **fm, **tv;
extern int am_recs, fm_recs, tv_recs;

#define N_SORT_FUNCS 3
typedef int (* Sort_func)(const void *, const void *);

typedef struct
{
  int show_am, show_fm, show_tv;
  int show_old, show_new, show_anc, keep;
  double lat, lon, dist_min, dist_max, corf_min, corf_max, rad2units;
  char *unit_name, *err_msg;
  const char *location_str, *dist_str, *corf_str, *keep_str;

  Sort_func keep_func;
  Sort_func sort_func[N_SORT_FUNCS];
  int sort_order[N_SORT_FUNCS];

  int interesting_antennas;
}
Search;

void sinfo_init_sort(Search *search);
int sinfo_compare_freq(const void *a, const void *b);
int sinfo_compare_dist(const void *a, const void *b);
int sinfo_compare_power(const void *a, const void *b);

typedef void (*Sinfos_display)(
  Search *search, Sinfo **sinfos, int n_sinfos, int found);
int search_sort_and_show_sinfo(Search *search, Sinfos_display sinfos_display);
void fcc_init(Search *search, int argc, char **argv);
void fcc_ingest(char *share, void (*progress)(char *, int, int, int));

Sinfo *sinfo_fill(band_enum band, off_t fac_off, off_t eng_off);

#endif
