/* station-info-cli -- displays information from the FCC database files.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <guess-where.h>
#include <getopt.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <utils.h>

static void
sinfos_display_fmt(Search *search, Sinfo **sinfos, int n_sinfos,
  char *off_fmt, char *fmt)
{
  int s;
  for (s = 0; s < n_sinfos; s++)
    {
      Sinfo *sinfo = sinfos[s];
      Sloc *sloc = &sinfo->sloc;
      Fac *fac = sinfo->fac;

      if (off_fmt)
	printf(off_fmt, sloc->band, sloc->fac_off, sloc->eng_off);

      printf(fmt,
	sloc->corf / 10.0, fac->service,
	sloc->old["-o"], fac->callsign,
	sinfo_get_power(sinfo),
	sinfo->arc * search->rad2units, search->unit_name,
	rad2point(sinfo->head), rad2deg(sinfo->head),
	fac->comm_state, fac->comm_city);
    }
}

static void
sinfos_display_text(Search *search, Sinfo **sinfos, int n_sinfos, int found)
{
  sinfos_display_fmt(search, sinfos, n_sinfos, NULL,
    "%6.1f %2s %c %-8s %8.1f %5.1f%s %3s %3.0f %s %s\n");
  printf("Listing %d of %d stations found.\n", n_sinfos, found);
}

static void
sinfos_display_tsv(Search *search, Sinfo **sinfos, int n_sinfos, int found)
{
  printf("%d\t%d\n", found, n_sinfos);
  sinfos_display_fmt(search, sinfos, n_sinfos, "%d\t%d\t%d\t",
    "%6.1f\t%2s\t%c\t%-8s\t%8.1f\t%5.1f\t%s\t%3s\t%3.0f\t%s\t%s\n");
}

static void
dump_sinfo(Sinfo *s)
{
  printf("sloc.eng_off %ld\n", s->sloc.eng_off);
  printf("sloc.fac_off %ld\n", s->sloc.fac_off);
  printf("sloc.lat %d\n", s->sloc.lat);
  printf("sloc.lon %d\n", s->sloc.lon);
  printf("sloc.corf %d\n", s->sloc.corf);
  printf("sloc.new %d\n", s->sloc.new);
  printf("sloc.old %d\n", s->sloc.old);
  printf("sloc.anc %d\n", s->sloc.anc);
  printf("sloc.band %d\n", s->sloc.band);

  printf("arc %g\n", s->arc);
  printf("head %g\n", s->head);
  printf("khz %d\n", s->khz);

  printf("fac.fac_id %d\n", s->fac->fac_id);
  printf("fac.channel %d\n", s->fac->channel);
  printf("fac.freq %g\n", s->fac->freq);
  printf("fac.callsign %s\n", s->fac->callsign);
  printf("fac.comm_city %s\n", s->fac->comm_city);
  printf("fac.comm_state %s\n", s->fac->comm_state);
  printf("fac.service %s\n", s->fac->service);
  printf("fac.fac_addr1 %s\n", s->fac->fac_addr1);
  printf("fac.fac_addr2 %s\n", s->fac->fac_addr2);
  printf("fac.fac_city %s\n", s->fac->fac_city);
  printf("fac.fac_state %s\n", s->fac->fac_state);
  printf("fac.fac_country %s\n", s->fac->fac_country);
  printf("fac.fac_zip %s\n", s->fac->fac_zip);

  switch (s->sloc.band)
    {
    case band_am:
      printf("am.ant_type %c\n", s->am->ant_type);
      printf("am.ant_sys_id %d\n", s->am->ant_sys_id);
      printf("am.towers %d\n", s->am->towers);
      printf("am.power %f\n", s->am->power);
      break;

    case band_fm:
      printf("fm.ant_id %d\n", s->fm->ant_id);
      printf("fm.ant_type %d\n", s->fm->ant_type);
      printf("fm.ant_pol %d\n", s->fm->ant_pol);
      printf("fm.ant_rot %f\n", s->fm->ant_rot);
      printf("fm.horiz_erp %f\n", s->fm->horiz_erp);
      printf("fm.vert_erp %f\n", s->fm->vert_erp);
      printf("fm.eamsl %f\n", s->fm->eamsl);
      break;

    case band_tv:
      printf("tv.ant_id %d\n", s->tv->ant_id);
      printf("tv.ant_type %d\n", s->tv->ant_type);
      printf("tv.ant_pol %d\n", s->tv->ant_pol);
      printf("tv.ant_rot %f\n", s->tv->ant_rot);
      printf("tv.erp %f\n", s->tv->erp);
      printf("tv.eamsl %f\n", s->tv->eamsl);
      break;
    }
}

static void
emit_details_tsv(Search *search, band_enum band, off_t fac_off, off_t eng_off)
{
  Ant *ant;
  Sinfo *sinfo = sinfo_fill(band, fac_off, eng_off);

  if (! sinfo)
    printf("failure\n");
  else
    {
      printf("success\n");
      gc(search->lat, search->lon,
	rot2rad(sinfo->sloc.lat), rot2rad(sinfo->sloc.lon),
	&sinfo->arc, &sinfo->head);
      printf("search.lat %+f\n", search->lat);
      printf("search.lon %+f\n", search->lon);
      dump_sinfo(sinfo);

      ant = sinfo_get_ant(sinfo);
      if (ant)
	{
	  Pattern *pat = ant->pattern;
	  int i, n = pat->n_points;
	  printf("ant.pat.min %.3f\n", pat->min);
	  printf("ant.pat.max %.3f\n", pat->max);
	  printf("ant.pat.rms %.3f\n", pat->rms);
	  printf("ant.pat.points %d\n", n);
	  printf("ant.pat.az_mag");
	  for (i = 0; i < n; i++)
	    printf(" %.3f %.3f", pat->azimuth[i], pat->magnitude[i]);
	  printf("\n");
	}
    }
}

/*
 * main -- for the station-info-cli program.
 */
int
main(int argc, char **argv)
{
  int c;
  Search search;
  band_enum band;
  off_t fac_off, eng_off;
  struct timeval t0, tn;

  setlocale(LC_ALL, "");
  fcc_init(&search, argc, argv);
  guess_where(&search.lat, &search.lon);
  search_set_location(&search, NULL);

#if HAVE_GETOPT_LONG
  while ((c = getopt_long(argc, argv, optchars, long_options, NULL)) != -1)
    process_option(&search, c, optarg);
#else
  while ((c = getopt(argc, argv, optchars)) != -1)
    process_option(&search, c, optarg);
#endif

  gettimeofday(&t0, NULL);

  if (emit_details(&band, &fac_off, &eng_off))
    emit_details_tsv(&search, band, fac_off, eng_off);
  else if (!emit_tsv())
    search_sort_and_show_sinfo(&search, sinfos_display_text);
  else
    search_sort_and_show_sinfo(&search, sinfos_display_tsv);

  gettimeofday(&tn, NULL);
  debug("Elapsed time: %.3f seconds", tdiff(&t0, &tn));
  search_dump(&search, stderr);

  return EXIT_SUCCESS;
}
