/* station-info-gen-index -- makes .si and .sip files from FCC data files.
 *
 * Copyright (C) 2002 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <glib.h>
#include <locale.h>
#include <fcc.h>
#include <getopt.h>
#include <guess-where.h>
#include <math.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <utils.h>
#include <sys/stat.h>
#include <sys/mman.h>

static const int new_file = O_CREAT | O_WRONLY | O_TRUNC;

static const char *band_to_str[] = { "am", "fm", "tv" };

static void
munge_tower(char *path)
{
  int i, j, s, fd;

  fd = open(path, new_file, 0666);
  if (fd < 0)
    fatal("munge_tower: can't open %s for output.", path);

  for (i = j = 0; i < tower_set_alloc; i++)
    {
      Ts_off off;
      Tower_set *ts = tower_set[i];

      if (ts == NULL)
	continue;

      if (ts->ts_off < 0)
	fatal("Error generating %s.\n"
	  "The am_towers data file must be uncompressed.\n"
	  "Error code", path);

      off.id = ts->id;
      off.ts_off = ts->ts_off;
      write(fd, &off, sizeof(off));
      j++;
    }
  close(fd);

  s = sizeof(Ts_off);
  printf("munge_tower:      %'d of %'7d recs in %'9d at %d per rec\n",
    j, i, j*s, s);
}

static void
munge_pat(char *path)
{
  int i, j, s, fd;

  fd = open(path, new_file, 0666);
  if (fd < 0)
    fatal("munge_pat: can't open %s for output.", path);

  for (i = j = 0; i < ant_alloc; i++)
    {
      Ant_offs offs;
      Ant *ant = antenna[i];

      if (ant == NULL)
	continue;

      if (ant->make_off < 0 || ant->pattern->pat_off < 0)
	fatal("Error generating %s.\n"
	  "The antenna make and pattern data files must be uncompressed.\n"
	  "Error code", path);

      offs.id = ant->id;
      offs.make_off = ant->make_off;
      offs.pat_off = ant->pattern->pat_off;
      write(fd, &offs, sizeof(offs));
      j++;
    }
  close(fd);

  s = sizeof(Ant_offs);
  printf("munge_ant:        %'d of %'d recs in %'9d at %d per rec\n",
    j, i, j*s, s);
}

static void
munge_eng(Sinfo **sinfos, int n_sinfos, char *path, int band)
{
  int i, j, s, fd;

  fd = open(path, new_file, 0666);
  if (fd < 0)
    fatal("munge_eng: can't open %s for output.", path);

  for (i = j = 0; i < n_sinfos; i++)
    {
      Sinfo *sinfo = sinfos[i];

      if (sinfo->fac == NULL)
	continue;

      if (sinfo->sloc.eng_off < 0 || sinfo->fac->fac_off < 0)
	fatal("Error generating %s.\n"
	  "The facility and engineering data files must be uncompressed.\n"
	  "Error code", path);

      write(fd, &sinfo->sloc, sizeof(sinfo->sloc));
      j++;
    }
  close(fd);

  s = sizeof(Sloc);
  printf("munge_eng(%d): %s: %'d of %'7d recs in %'9d at %d per rec\n",
    band, band_to_str[band], j, i, j*s, s);
}

static void
munge_am_eng(void)
{
  munge_eng(am, am_recs, PKGDATADIR "/am_ant_sys.si", band_am);
}

static void
munge_fm_eng(void)
{
  munge_eng(fm, fm_recs, PKGDATADIR "/fm_eng_data.si", band_fm);
}

static void
munge_tv_eng(void)
{
  munge_eng(tv, tv_recs, PKGDATADIR "/tv_eng_data.si", band_tv);
}

/*
 * main -- for the gen-fcc-bin program.
 */
int
main(int argc, char **argv)
{
  int c;
  struct timeval t0, tn;
  Search *search;

  setlocale(LC_ALL, "");
  search = fmalloc(sizeof(*search));
  fcc_init(search, argc, argv);

  while ((c = getopt_long(argc, argv, optchars, long_options, NULL)) != -1)
    process_option(search, c, optarg);

  gettimeofday(&t0, NULL);
  fcc_ingest(PKGDATADIR, print_ingest_progress);

  munge_pat(PKGDATADIR "/ant_make_pat.sip");
  munge_tower(PKGDATADIR "/am_towers.sip");
  munge_am_eng();
  munge_fm_eng();
  munge_tv_eng();

  gettimeofday(&tn, NULL);
  fprintf(stderr, "munge-gen: %5.3f secs\n", tdiff(&t0, &tn));
    
  return EXIT_SUCCESS;
}
