/* callbacks.c -- callback routines for the station-info GUI.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <utils.h>
#include <callbacks.h>
#include <interface.h>
#include <main.h>
#include <sys/time.h>
#include <support.h>

static int
run_dialog(GtkWidget *dialog, GtkWidget *parent)
{
  gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(parent));
  gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
  return -1;
}

void
fatal_need_for_data(void)
{
  run_dialog(create_need_data_msgbox(), station_info);
  gtk_main_quit();
  exit(EXIT_FAILURE);
}

void
search_set_widget_state(Search *search, GtkWidget *station_info)
{
  gtk_entry_set_text(
    GTK_ENTRY(lookup_widget(station_info, "location_entry")),
    search->location_str);
  gtk_entry_set_text(
    GTK_ENTRY(lookup_widget(station_info, "distance_entry")),
    search->dist_str);
  gtk_entry_set_text(
    GTK_ENTRY(lookup_widget(station_info, "corf_entry")),
    search->corf_str);
  gtk_entry_set_text(
    GTK_ENTRY(lookup_widget(station_info, "keep_entry")),
    search->keep_str);

  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "am_checkbutton")),
    search->show_am);
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "fm_checkbutton")),
    search->show_fm);
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "tv_checkbutton")),
    search->show_tv);

  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "new_checkbutton")),
    search->show_new);
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "old_checkbutton")),
    search->show_old);
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "anc_checkbutton")),
    search->show_anc);
}

static int
search_get_widget_state(Search *search, GtkWidget *station_info)
{
  char *str;
  GtkEntry *entry;

  entry = GTK_ENTRY(lookup_widget(station_info, "location_entry"));
  str = gtk_entry_get_text(entry);
  if (search_set_location(search, str))
    return run_dialog(create_location_msgbox(), station_info);

  entry = GTK_ENTRY(lookup_widget(station_info, "distance_entry"));
  str = gtk_entry_get_text(entry);
  if (search_set_distance(search, str))
    return run_dialog(create_distance_msgbox(), station_info);

  entry = GTK_ENTRY(lookup_widget(station_info, "corf_entry"));
  str = gtk_entry_get_text(entry);
  if (search_set_corf(search, str))
    return run_dialog(create_corf_msgbox(), station_info);

  entry = GTK_ENTRY(lookup_widget(station_info, "keep_entry"));
  str = gtk_entry_get_text(entry);
  if (search_set_keep(search, str))
    return run_dialog(create_keep_msgbox(), station_info);

  search->show_am = gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "am_checkbutton")));
  search->show_fm = gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "fm_checkbutton")));
  search->show_tv = gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "tv_checkbutton")));

  search->show_new = gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "new_checkbutton")));
  search->show_old = gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "old_checkbutton")));
  search->show_anc = gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON(lookup_widget(station_info, "anc_checkbutton")));

  return 0;
}

static void
sinfos_display_clist(Search *search, Sinfo **sinfos, int n_sinfos, int n_found)
{
  int s;
  char found[40], corf_str[20], service_str[20], call_str[20], power_str[20];
  char distance_str[20], heading_str[20], location_str[80];
  char *station_text[] =
  {
    corf_str, service_str, call_str, power_str, 
    distance_str, heading_str, location_str
  };
  printf("sinfos-display-clist: found %d of %d at %p\n",
    n_found, n_sinfos, sinfos);

  if (sinfos == NULL)
    fatal_need_for_data();

  gtk_clist_freeze(clist);
  gtk_clist_clear(clist);

  for (s = 0; s < n_sinfos; s++)
    {
      Sinfo *sinfo = sinfos[s];
      Sloc *sloc = &sinfo->sloc;
      Fac *fac = sinfo->fac;

      sprintf(corf_str, "%6.1f", sloc->corf / 10.0);
      sprintf(service_str, "%s", fac->service);
      sprintf(call_str, "%s", fac->callsign);
      sprintf(power_str, "%8.1f", sinfo_get_power(sinfo));
      sprintf(distance_str, "%.1f %s", 
	sinfo->arc * search->rad2units, search->unit_name);
      sprintf(heading_str, "%s %03.0f", 
	rad2point(sinfo->head), rad2deg(sinfo->head));
      sprintf(location_str, "%s %s", fac->comm_state, fac->comm_city);
      gtk_clist_append(clist, station_text);
    }

  gtk_object_set_data(GTK_OBJECT(clist), "sinfo-array", sinfos);
  gtk_clist_columns_autosize(clist);
  gtk_clist_thaw(clist);

  sprintf(found, "Showing %d of %d stations", n_sinfos, n_found);
  gnome_appbar_set_status(GNOME_APPBAR(appbar), found);
}

void
on_strongest_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
  if (gtk_toggle_button_get_active(togglebutton))
    search.keep_func = sinfo_compare_power;
}

void
on_nearest_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
  if (gtk_toggle_button_get_active(togglebutton))
    search.keep_func = sinfo_compare_dist;
}

void
on_about_activate(GtkMenuItem *menuitem, void *user_data)
{
  run_dialog(create_about(), station_info);
}

void
on_print_activate(GtkMenuItem *menuitem, void *user_data)
{
  run_dialog(create_sorry_msgbox(), station_info);
}

void
on_preferences_activate(GtkMenuItem *menuitem, void *user_data)
{
  run_dialog(create_sorry_msgbox(), station_info);
}

void
on_exit_activate(GtkMenuItem *menuitem, void *user_data)
{
  gtk_main_quit();
}

void
on_station_info_destroy(GtkObject *object, void *user_data)
{
  gtk_main_quit();
}

void
on_help_activate(GtkMenuItem *menuitem, void *user_data)
{
  static GnomeHelpMenuEntry help_desc = { PACKAGE, "index.html" };
  setenv("GNOMEDIR", PREFIX, 0);
  gnome_help_display(NULL, &help_desc);
}

void
on_search_button_clicked(GtkButton *button, void *user_data)
{
  search_get_widget_state(&search, station_info);
  search_dump(&search, stderr);
  search_sort_and_show_sinfo(&search, sinfos_display_clist);
}

void
on_clist_click_column(GtkCList *clist, gint column, void *user_data)
{
  switch (column)
    {
    case 0: sort_add_keys(&search, "~freq"); break;
    case 3: sort_add_keys(&search, "~power"); break;
    case 4: sort_add_keys(&search, "~dist"); break;
    default: return;
    }

  /*
   * FIX THIS: when using sinfo, we're re-searching for the same data
   * that we've already found.  It would be slightly faster to copy
   * the approach used when using sid-data, and separate the search
   * from the sort and display.
   */
  search_sort_and_show_sinfo(&search, sinfos_display_clist);
}

