/* fcc-lib -- manipulate information from the FCC database files.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <fcc.h>
#include <guess-where.h>
#include <utils.h>

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <values.h>

/*
 * open_db_stream -- try to open base, base.dat, base.zip, and
 * base.dat.gz, whichever is found first.
 */
static FILE*
open_db_stream(const char *share, const char *base, char **fn)
{
  FILE *stream;
  char *ext, peek, path[PATH_MAX], cmd[PATH_MAX];

  strcpy(path, share);
  if (path[strlen(path) - 1] != '/')
    strcat(path, "/");
  while (*base && *base == '/')
    base++;
  strcat(path, base);
  ext = path + strlen(path);
  if (strrchr(base, '.'))
    ext = strrchr(path, '.');

  if (fn) *fn = path;
  if ((stream = fopen(path, "r")) == NULL)
    {
      strcpy(ext, ".dat");
      if ((stream = fopen(path, "r")) == NULL)
	{
	  strcpy(cmd, "gzip -dc ");
	  strcpy(ext, ".zip 2>/dev/null");
	  strcat(cmd, path);
	  stream = popen(cmd, "r");
	  if (stream && (peek = getc(stream)) != EOF)
	    ungetc(peek, stream);
	  else
	    {
	      strcpy(cmd, "gzip -dc ");
	      strcpy(ext, ".dat.gz 2>/dev/null");
	      strcat(cmd, path);
	      stream = popen(cmd, "r");
	    }
	}
    }
  return stream;
}

static int
close_db_stream(FILE *stream)
{
  if (lseek(fileno(stream), 0, SEEK_SET) == 0)
    return fclose(stream);
  else
    return pclose(stream);
}

static void *
frealloc0(void *current_mem, size_t chunk, int *current_alloc, int new_alloc)
{
  void *mem = frealloc(current_mem, new_alloc * chunk);
  if (*current_alloc < new_alloc)
    memset(mem + *current_alloc * chunk,
      0, (new_alloc - *current_alloc) * chunk);
  *current_alloc = new_alloc;
  return mem;
}

/*
 * Antenna -- information on stock antennas and their radiation patterns.
 */
Ant **antenna = NULL;
int ant_alloc = 0;
static int ant_types, ant_rec, ant_patterns, pat_rec;
static const int ant_fields = 8;

Ant *
ant_new(void)
{
  Ant *ant = fmalloc0(sizeof(*ant));
  ant->pattern = fmalloc0(sizeof(*ant->pattern));
  ant->pattern->rms = 1;
  ant->pattern->min = MAXFLOAT;
  return ant;
}

void
ant_free(Ant *ant)
{
  if (ant->make) free(ant->make);
  if (ant->model) free(ant->model);
  if (ant->service) free(ant->service);
  if (ant->standard) free(ant->standard);
  if (ant->updated) free(ant->updated);

  if (ant->pattern)
    {
      if (ant->pattern->azimuth) free(ant->pattern->azimuth);
      if (ant->pattern->magnitude) free(ant->pattern->magnitude);
      free(ant->pattern);
    }
  free(ant);
}

void
ant_make_fill_from_fields(Ant *ant, char **field)
{
  ant->make = fstrdup(field[1]);
  ant->model = fstrdup(field[2]);
  ant->service = fstrdup(field[3]);
  ant->standard = fstrdup(field[4]);
  ant->updated = fstrdup(field[5]);
}

static int
ingest_ant_make(char *share, char *base)
{
  int ant_id;
  off_t make_off = 0;
  Ant *ant;
  FILE *stream;
  char *fn, line[2000], *field[ant_fields];

  if ((stream = open_db_stream(share, base, &fn)) == NULL)
    fatal("can't open %s", fn);

  for (ant_rec = 1; fgets(line, sizeof(line), stream); ant_rec++)
    {
      split(field, ant_fields, line, '|');
      ant_id = atoi(field[0]);

      if (ant_id < 0)
	continue;

      if (ant_alloc <= ant_id)
	antenna = frealloc0(antenna,
	  sizeof(*antenna), &ant_alloc, ant_id + 1000);

      ant = ant_new();
      ant->id = ant_id;
      ant->make_off = make_off;
      ant_make_fill_from_fields(ant, field);
      ant_types++;
      antenna[ant_id] = ant;
      make_off = ftell(stream);
    }

  close_db_stream(stream);
  return ant_types;
}

int
ant_pattern_fill_from_fields(Ant *ant, char **field)
{
  double mag;
  Pattern *pat = ant->pattern;
  int n = pat->n_points + 1;
  if (ant->id != atoi(field[0]))
    return 0;

  pat->azimuth = frealloc(pat->azimuth, n * sizeof(*pat->azimuth));
  pat->magnitude = frealloc(pat->magnitude, n * sizeof(*pat->magnitude));

  pat->azimuth[n - 1] = deg2rad(atof(field[1]));
  mag = pat->magnitude[n - 1] = atof(field[2]);

  if (pat->max < mag)
    pat->max = mag;
  if (pat->min > mag)
    pat->min = mag;

  return ++pat->n_points;
}

/*
 * Antenna patterns -- ingest information on FM and TV antenna patterns.
 */
static int
ingest_ant_pattern(char *share, char *base)
{
  int ant_id;
  off_t pat_off = 0;
  FILE *stream;
  char *fn, line[2000], *field[ant_fields];

  if ((stream = open_db_stream(share, base, &fn)) == NULL)
    fatal("can't open %s", fn);

  for (pat_rec = 1; fgets(line, sizeof(line), stream); pat_rec++)
    {
      split(field, ant_fields, line, '|');
      ant_id = atoi(field[0]);
      if (ant_id < 0 || ant_alloc <= ant_id)
	fatal("ant: line %d: ant_id=%d\n", pat_rec, ant_id);
      else
	{
	  Ant *ant = antenna[ant_id];
	  if (ant == NULL)
	    ant = ant_new();
	  if (ant->pattern->n_points == 0)
	    {
	      ant->pattern->pat_off = pat_off;
	      ant_patterns++;
	    }

	  ant_pattern_fill_from_fields(ant, field);
	}
      pat_off = ftell(stream);
    }

  close_db_stream(stream);
  return ant_patterns;
}

int
tower_set_fill_from_fields(Tower_set *tower_set, char **field)
{
  int tn;
  Tower *t;

  if (tower_set->id != atoi(field[0]))
    return 0;

  t = fmalloc0(sizeof(*t));

  t->field_ratio = atof(field[3]);
  t->orient = deg2rad(atof(field[8]));
  t->phasing = deg2rad(atof(field[9]));
  t->spacing = deg2rad(atof(field[11]));
  t->ref_switch = atoi(field[19]);

  tn = tower_set->towers++;
  tower_set->tower = frealloc(tower_set->tower,
    (tn + 1) * sizeof(*tower_set->tower));
  tower_set->tower[tn] = t;
  if (t->ref_switch)
    tower_set->rel++;

  return tower_set->towers;
}

/*
 * Towers -- ingest information on AM antenna towers.
 */
Tower_set **tower_set = NULL;
int tower_set_alloc = 0;
static int tower_sets, tower_rec;
static const int tower_set_fields = 28;

static int
ingest_towers(char *share, char *base)
{
  int ant_id;
  off_t ts_off = 0;
  FILE *stream;
  char *fn, line[2000], *field[tower_set_fields];

  if ((stream = open_db_stream(share, base, &fn)) == NULL)
    fatal("can't open %s", fn);

  for (tower_rec = 1; fgets(line, sizeof(line), stream); tower_rec++)
    {
      split(field, tower_set_fields, line, '|');
      ant_id = atoi(field[0]);

      if (ant_id < 0)
	continue;

      if (tower_set_alloc <= ant_id)
	tower_set = frealloc0(tower_set,
	  sizeof(*tower_set), &tower_set_alloc, ant_id + 1000);

      if (tower_set[ant_id] == NULL)
	{
	  tower_set[ant_id] = fmalloc0(sizeof(*tower_set[ant_id]));
	  tower_set[ant_id]->id = ant_id;
	  tower_set[ant_id]->ts_off = ts_off;
	  tower_sets++;
	}

      tower_set_fill_from_fields(tower_set[ant_id], field);
      ts_off = ftell(stream);
    }

  // printf("ingest towers: %d relative towers\n", rel_towers);
  close_db_stream(stream);
  return tower_sets;
}

/*
 * Application -- The only thing that the application data is used for
 * is to get the facility id associated with a given application id. 
 */
static int app_alloc, *app2fac;
static int app_rec, app_cnt;
static const int app_fields = 5;

static int
ingest_app(char *share, char *base)
{
  int app_id, fac_id;
  FILE *stream;
  char *fn, line[2000], *field[app_fields];

  if ((stream = open_db_stream(share, base, &fn)) == NULL)
    fatal("can't open %s", fn);

  app_cnt = app_alloc = 0;
  app2fac = NULL;
  for (app_rec = 1; fgets(line, sizeof(line), stream); app_rec++)
    {
      split(field, app_fields, line, '|');
      app_id = atoi(field[2]);
      fac_id = atoi(field[3]);

      if (app_id < 0 || fac_id < 0 || app_id == 1000000) // FCC: FIX THIS
	continue;

      if (app_alloc <= app_id)
	app2fac = frealloc0(app2fac,
	  sizeof(*app2fac), &app_alloc, app_id + 1000);

      app_cnt++;
      app2fac[app_id] = fac_id;
    }

  close_db_stream(stream);
  return app_cnt;
}

/*
 * Facility -- the facility record contains much of the general
 * information about a station -- city, state, channel, and so forth. 
 */
static Fac *fac = NULL;
static int fac_rec, facs, fac_alloc = 0;
static const int fac_fields = 70;

void
fac_fill_from_fields(Fac *fac, char **field)
{
  fac->comm_city = capitalize(fstrdup(field[0]));
  fac->comm_state = fstrdup(field[1]);
  fac->fac_addr1 = capitalize(fstrdup(field[3]));
  fac->fac_addr2 = capitalize(fstrdup(field[4]));
  fac->callsign = callsign_normalize(fstrdup(field[5]));
  fac->channel = atoi(field[6]);
  fac->fac_city = capitalize(fstrdup(field[7]));
  fac->fac_country = fstrdup(field[8]);
  fac->freq = atof(field[9]);
  fac->service = fstrdup(field[10]);
  fac->fac_state = fstrdup(field[11]);
  fac->fac_zip = fmalloc(12);
  fac->fac_id = atoi(field[14]);
  sprintf(fac->fac_zip, *field[18] ? "%s-%s" : "%s", field[17], field[18]);
}

static int
ingest_fac(char *share, char *base)
{
  int fac_id;
  off_t fac_off = 0;
  FILE *stream;
  char *fn, line[2000], *field[fac_fields];

  if ((stream = open_db_stream(share, base, &fn)) == NULL)
    fatal("can't open %s", fn);

  for (fac_rec = 1; fgets(line, sizeof(line), stream); fac_rec++)
    {
      split(field, fac_fields, line, '|');
      fac_id = atoi(field[14]);

      if (fac_id < 0)
	continue;

      if (fac_alloc <= fac_id)
	{
	  #if 0
	  int peek = fgetc(stream);
	  ungetc(peek, stream);
	  if (peek == EOF) 
	    continue;
	  else /* not at EOF; we need more core! */
	  #endif
	    fac = frealloc0(fac, sizeof(*fac), &fac_alloc, fac_id + 1000);
	}

      fac_fill_from_fields(fac + fac_id, field);
      fac[fac_id].fac_off = fac_off;
      fac_off = ftell(stream);
      facs++;
    }

  close_db_stream(stream);
  return facs;
}

static double
ddms2rot(char **field)
{
  double deg = atoi(field[0]) + atoi(field[2])/60. + atof(field[3])/3600.;
  if (*field[1] == 'S' || *field[1] == 'W')
    deg *= -1;
  return 2. * 1024 * 1024 * 1024 * (deg / 180);
}

int
service_to_anc(char *service)
{
  return
    !streq("AM", service) &&
    !streq("FM", service) &&
    !streq("LP", service) &&
    !streq("TV", service);
}

int
callsign_to_new(char *callsign)
{
  if (callsign == NULL) return 1;
  if (streq("NEW", callsign)) return 1;
  if (streq("None", callsign)) return 1;
  return 0;
}

/*
 * AM
 */
Sinfo **am;
int am_recs;
static const int am_fields = 38;

void
sinfo_am_fill_from_fields(Sinfo *sinfo, char **field)
{
  sinfo->khz = sinfo->fac->freq;

  sinfo->sloc.band = band_am;
  sinfo->sloc.corf = sinfo->fac->freq * 10 + 0.5;

  sinfo->sloc.lat = ddms2rot(field + 12);
  sinfo->sloc.lon = ddms2rot(field + 12 + 4);
  
  sinfo->sloc.old = *field[27] != 'C';
  sinfo->sloc.anc = service_to_anc(sinfo->fac->service);
  sinfo->sloc.new = callsign_to_new(sinfo->fac->callsign);

  sinfo->am = fmalloc(sizeof(*sinfo->am));
  sinfo->am->ant_sys_id = atoi(field[1]);
  sinfo->am->power = atof(field[22]);
  sinfo->am->towers = atof(field[26]);
  sinfo->am->ant_type = *field[34];
}

/*
 * FM
 */
Sinfo **fm;
int fm_recs;
static const int fm_fields = 70;

void
sinfo_fm_fill_from_fields(Sinfo *sinfo, char **field)
{
  sinfo->khz = sinfo->fac->freq * 1000 + 0.5;

  sinfo->sloc.band = band_fm;
  sinfo->sloc.corf = sinfo->fac->freq * 10 + 0.5;

  sinfo->sloc.lat = ddms2rot(field + 30);
  sinfo->sloc.lon = ddms2rot(field + 30 + 4);
  
  sinfo->sloc.old = *field[19] != 'C';
  sinfo->sloc.anc = service_to_anc(sinfo->fac->service);
  sinfo->sloc.new = callsign_to_new(sinfo->fac->callsign);

  sinfo->fm = fmalloc(sizeof(*sinfo->fm));
  sinfo->fm->ant_pol = *field[2];
  sinfo->fm->ant_rot = deg2rad(atof(field[3]));
  sinfo->fm->ant_id = atoi(field[4]);
  sinfo->fm->ant_type = *field[5];
  sinfo->fm->horiz_erp = atof(field[29]);
  sinfo->fm->vert_erp = atof(field[52]);
  sinfo->fm->eamsl = atof(field[17]);
}

/*
 * TV
 */
Sinfo **tv;
int tv_recs;
static const int tv_fields = 70;

/* The TV station frequencies in the facilities table are frequently
   omitted (or odd looking).  Calculate a nominal frequency based on
   the channel and use that instead. */
int
tv_channel_to_khz(int chan)
{
  if ( 2 <= chan && chan <=  4) return  54000 + 6000 * (chan -  2);
  if ( 5 <= chan && chan <=  6) return  76000 + 6000 * (chan -  5);
  if ( 7 <= chan && chan <= 13) return 174000 + 6000 * (chan -  7);
  if (14 <= chan && chan <= 69) return 470000 + 6000 * (chan - 14);
  return 0;
}

void
sinfo_tv_fill_from_fields(Sinfo *sinfo, char **field)
{
  sinfo->khz = tv_channel_to_khz(sinfo->fac->channel);

  sinfo->sloc.band = band_tv;
  sinfo->sloc.corf = sinfo->fac->channel * 10;

  sinfo->sloc.lat = ddms2rot(field + 28);
  sinfo->sloc.lon = ddms2rot(field + 28 + 4);
  
  sinfo->sloc.old = *field[19] != 'C';
  sinfo->sloc.anc = service_to_anc(sinfo->fac->service);
  sinfo->sloc.new = callsign_to_new(sinfo->fac->callsign);

  sinfo->tv = fmalloc(sizeof(*sinfo->tv));
  sinfo->tv->ant_pol = *field[2];
  sinfo->tv->ant_id = atoi(field[3]);
  sinfo->tv->ant_type = *field[4];
  sinfo->tv->erp = atof(field[15]);
  sinfo->tv->eamsl = atof(field[17]);
  sinfo->tv->ant_rot = deg2rad(atof(field[57]));
}

static int
generic_ingest(char *dir, char *file, int *db_recs,
  Sinfo ***sinfo_addr, int *sinfos_addr, int fields, 
  int app_field, int fac_field, void (*ancillary_ingest)(Sinfo *, char **))
{
  int app_id, fac_id, recno, sinfos = 0;
  Sinfo **sinfo = NULL;
  FILE *stream;
  off_t eng_off = 0;
  char line[2000], *field[200], *fn;

  if ((stream = open_db_stream(dir, file, &fn)) == NULL)
      fatal("can't open %s", fn);
  
  for (recno = 1; fgets(line, sizeof(line), stream); recno++)
    {
      split(field, fields, line, '|');
      app_id = atoi(field[app_field]);
      fac_id = fac_field < 0 ? app2fac[app_id] : atoi(field[fac_field]);
      if (0 < app_id && 0 < fac_id)
	{
	  Fac *f = fac + fac_id;

 	  sinfo = frealloc(sinfo, (sinfos + 1) * sizeof(*sinfo));
 	  sinfo[sinfos] = fmalloc(sizeof(*sinfo[sinfos]));
	  sinfo[sinfos]->fac = f;

 	  sinfo[sinfos]->sloc.eng_off = eng_off;
	  sinfo[sinfos]->sloc.fac_off = f->fac_off;

 	  if (ancillary_ingest)
 	    ancillary_ingest(sinfo[sinfos], field);

	  eng_off = ftell(stream);
	  sinfos++;
	}
    }

  close_db_stream(stream);
  *db_recs = recno;
  *sinfos_addr = sinfos;
  *sinfo_addr = sinfo;
  return sinfos;
}

void
fcc_init(Search *search, int argc, char **argv)
{
  /* Get the name of the program, for use in error messages. */
  prog_name = argv[0];
  if (strrchr(prog_name, '/'))
    prog_name = strrchr(prog_name, '/') + 1;

  /* Establish default values. */
  search->unit_name = "km";
  search->rad2units = 40000 / (2 * M_PI);

  search_set_location(search, "N39,W76");
  search_set_distance(search, "0 - 50 km");
  search_set_corf(search, "0 - 2000");
  search_set_keep(search, "");

  search->show_am = search->show_fm = search->show_tv = 1;
  search->show_old = search->show_new = search->show_anc = 0;
  search->interesting_antennas = 0;

  search->keep_func = sinfo_compare_dist;
  sort_add_keys(search, "freq,dist,power");
}

void
fcc_ingest(char *share, void (*progress)(char *, int, int, int))
{
  int db_recs;
  struct timeval t0, tn;
  gettimeofday(&t0, NULL);
  if (progress)
    progress(NULL, 0, 0, 0);

  ingest_ant_make(share, "/ant_make");
  if (progress)
    progress("ant_make", ant_rec, ant_types, ant_alloc);

  ingest_ant_pattern(share, "/ant_pattern");
  if (progress)
    progress("ant_pattern", pat_rec, ant_patterns, 0);

  ingest_app(share, "/application");
  if (progress)
    progress("application", app_rec, app_cnt, app_alloc);

  ingest_fac(share, "/facility");
  if (progress)
    progress("facility", fac_rec, facs, fac_alloc);

  ingest_towers(share, "/am_towers");
  if (progress)
    progress("am_towers", tower_rec, tower_sets, tower_set_alloc);

  generic_ingest(share, "/am_ant_sys", &db_recs,
    &am, &am_recs, am_fields, 2, -1, sinfo_am_fill_from_fields);
  if (progress)
    progress("am_ant_sys", db_recs, am_recs, 0);

  generic_ingest(share, "/fm_eng_data", &db_recs,
    &fm, &fm_recs, fm_fields, 6, 20, sinfo_fm_fill_from_fields);
  if (progress)
    progress("fm_eng_data", db_recs, fm_recs, 0);

  generic_ingest(share, "/tv_eng_data", &db_recs,
    &tv, &tv_recs, tv_fields, 5, 21, sinfo_tv_fill_from_fields);
  if (progress)
    progress("tv_eng_data", db_recs, tv_recs, 0);

  gettimeofday(&tn, NULL);
  //printf("init: %.3F seconds\n", tdiff(&t0, &tn));
}
