/* utils -- assorted utility routines.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <limits.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <geoloc.h>
#include <utils.h>

int verbose;
char *prog_name;

static void
put_args(int err, char *fmt, va_list args)
{
  fflush(stdout);
  fprintf(stderr, "%s: ", prog_name);
  vfprintf(stderr, fmt, args);
  if (err)
    fprintf(stderr, ",\nError %d: %s", err, strerror(err));
  fprintf(stderr, "\n");
}

void
debug(char *fmt, ...)
{
  if (verbose)
    {
      va_list args;
      va_start(args, fmt);
      put_args(0, fmt, args);
      va_end(args);
    }
}

void
warn(char *fmt, ...)
{
  if (verbose)
    {
      va_list args;
      va_start(args, fmt);
      put_args(errno, fmt, args);
      va_end(args);
    }
}

void
fatal(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  put_args(errno, fmt, args);
  va_end(args);

  exit(EXIT_FAILURE);
}

char *
fstrdup(const char *s)
{
  char *dup = strdup(s);
  if (dup == NULL)
    fatal("can't strdup with length %d\n", strlen(s));
  return dup;
}

void *
fmalloc(size_t len)
{
  void *mem = malloc(len);
  if (mem == NULL)
    fatal("can't malloc %d bytes\n", len);
  return mem;
}

void *
fmalloc0(size_t len)
{
  void *mem = fmalloc(len);
  memset(mem, 0, len);
  return mem;
}

void *
frealloc(void *mem, size_t len)
{
  void *remem = realloc(mem, len);
  if (remem == NULL)
    fatal("can't realloc %d bytes\n", len);
  return remem;
}

int
streq(const char *s1, const char *s2)
{
  if (s2 == NULL)
    return 0;
  return strcasecmp(s1, s2) == 0;
}

char *
capitalize(char *s0)
{
  char *s, bow=1;

  for (s = s0; *s; s++)
    if (!isalpha(*s))
      bow = 1;
    else if (!bow)
      *s = tolower(*s);
    else
      bow = 0;

  return s0;
}

char *
callsign_normalize(char *call)
{
  char *cp;
  call = skipbl(call);
  for (cp = strchr(call, ' '); cp; cp = strchr(cp, ' '))
    *cp = '-';
  return call;
}

int
split(char **field, int f_max, char *line, char sep)
{
  int f = 0;

  f_max--;
  while (line && *line && f < f_max)
    {
      field[f++] = line;

      if (f < f_max)
	{
	  while (*line && *line != sep)
	    line++;

	  if (*line == sep)
	    *line++ = 0;
	}
    }

  field[f] = NULL;
  return f;
}

double
deg2rad(double degrees)
{
  return M_PI * (degrees / 180.);
}

double
rad2deg(double radians)
{
  return 180. * (radians / M_PI);
}

void
gc(double lat0, double lon0, double lat1, double lon1,
  double *arc, double *head)
{
  double dlon = lon1 - lon0;
  double dlat = lat1 - lat0;
  double slat = sin(dlat / 2);
  double slon = sin(dlon / 2);
  double a = slat * slat + slon * slon * cos(lat0) * cos(lat1);
  *arc = 2 * atan2(sqrt(a), sqrt(1 - a));
  *head = acos((sin(lat1) - sin(lat0) * cos(*arc)) / (sin(*arc) * cos(lat0)));
}

/*
 * deg2pt -- translate degrees to a compass point string.
 */
const char *
rad2point(double rad)
{
  int point;
  char *pstr[] =
    {
      "n", "nne", "ne", "ene", "e", "ese", "se", "sse",
      "s", "ssw", "sw", "wsw", "w", "wnw", "nw", "nnw"
    };

  if (rad < 0)
    rad = 2 * M_PI + rad;
  point = (rad + M_PI / 16) / (M_PI / 8);

  return pstr[point % 16];
}

char *
skipbl(char *s)
{
  while (isspace(*s))
    s++;
  return s;
}

double
tdiff(struct timeval *t0, struct timeval *t1)
{
  return t1->tv_sec - t0->tv_sec + (t1->tv_usec - t0->tv_usec) / 1e6;
}

int
search_set_location(Search *search, char *str)
{
  int err = 0;
  char where[100];

  if (str)
    {
      Geoloc *geo = geoloc_new();
      err = !geoloc_from_str(geo, str);
      if (err)
	{
	  search->err_msg = "unrecognized location";
	  geoloc_free(geo);
	  return err;
	}
      search->lat = geo->lat;
      search->lon = geo->lon;
      geoloc_free(geo);
    }

  sprintf(where, "%s %.4f, %s %.4f",
    0 < search->lat ? "N" : "S", rad2deg(fabs(search->lat)),
    0 < search->lon ? "E" : "W", rad2deg(fabs(search->lon)));
  search->location_str = strdup(where);

  return err;
}

static void
set_radius_units(Search *search, double circ, char *name)
{
  search->unit_name = name;
  search->rad2units = circ / (2 * M_PI);
}

static double
str_to_dist(Search *search, const char *s0, char **tail_ptr)
{
  char *s = (char *)s0, *tail;
  double units;

  *tail_ptr = (char *)s0;
  units = strtod(s, &tail);
  if (s == tail)
    return 0;

  *tail_ptr = tail;
  if (!strncasecmp("km", skipbl(tail), 2))
    set_radius_units(search, 40000, "km");
  else if (!strncasecmp("mi", skipbl(tail), 2))
    set_radius_units(search, 24854.848, "mi");
  else if (!strncasecmp("nm", skipbl(tail), 2))
    set_radius_units(search, 21598.272, "nm");
  else
    return units;

  *tail_ptr = skipbl(tail) + 2;
  return units;
}

int
search_set_distance(Search *search, const char *s0)
{
  char *tail;
  const char *s = s0;
  double units, min, max;

  search->err_msg = "invalid distance";
  units = str_to_dist(search, s, &tail);
  if (s == tail)
    return -1;

  min = 0;
  max = units / search->rad2units;
  tail = skipbl(tail);
  if (*tail)
    {
      search->err_msg = "comma or hyphen expected";
      if (*tail != ',' && *tail != '-')
	return -1;

      s = tail + 1;
      units = str_to_dist(search, s, &tail);
      search->err_msg = "invalid maximum distance";
      if (s == tail)
	return -1;
      search->err_msg = "junk after distance";
      if (*skipbl(tail))
	return -1;
      min = max;
      max = units / search->rad2units;
    }

  search->dist_str = s0;
  search->dist_min = min;
  search->dist_max = max;
  return 0;
}

int
search_set_corf(Search *search, const char *s0)
{
  char *tail;
  const char *s = s0;
  double min, max;

  min = max = strtod(s, &tail);
  search->err_msg = "invalid channel or frequency";
  if (s == tail)
    return -1;

  tail = skipbl(tail);
  if (*tail)
    {
      search->err_msg = "comma or hyphen expected";
      if (*tail != ',' && *tail != '-')
	return -1;

      s = tail + 1;
      max = strtod(s, &tail);
      search->err_msg = "invalid upper channel or frequency";
      if (s == tail)
	return -1;
      search->err_msg = "junk after channel or frequency";
      if (*skipbl(tail))
	return -1;
    }

  search->corf_str = s0;
  search->corf_min = min;
  search->corf_max = max;
  return 0;
}

int
search_set_keep(Search *search, const char *s0)
{
  char *tail, *s = (char *)s0;
  int keep;

  if (!*skipbl(s))
    keep = INT_MAX;
  else
    {
      keep = strtoul(s, &tail, 0);
      search->err_msg = "invalid number of stations per channel";
      if (s == tail)
	return -1;

      search->err_msg = "junk after number of stations";
      if (*skipbl(tail))
	return -1;
    }

  search->keep_str = s0;
  search->keep = keep;
  return 0;
}

static char *
sort_func_to_string(Search *search, Sort_func fp)
{
  if (fp == NULL) return "none";
  if (fp == sinfo_compare_freq) return "freq";
  if (fp == sinfo_compare_dist) return "dist";
  if (fp == sinfo_compare_power) return "power";
  return "bogus";
}

static Sort_func
sort_str_to_func(char *str)
{
  if (streq("freq", str))
    return sinfo_compare_freq;
  if (streq("dist", str))
    return sinfo_compare_dist;
  if (streq("power", str))
    return sinfo_compare_power;
  return NULL;
}

static void
sort_add_key_func(Search *search, Sort_func sort_func, int sort_order)
{
  int i, j;

  if (sort_func == NULL)
    memset(search->sort_func, 0, sizeof(search->sort_func));

  /* Find a slot. */
  j = 0;
  while (j < N_SORT_FUNCS - 1 && search->sort_func[j] != sort_func)
    j++;

  if (sort_order == 0 && search->sort_func[j] == sort_func)
    sort_order = -search->sort_order[j];

  /* Shuffle down. */
  for (i = j - 1; 0 <= i; i--, j--)
    {
      search->sort_func[j] = search->sort_func[i];
      search->sort_order[j] = search->sort_order[i];
    }

  /* Fill in new primary search function. */
  search->sort_func[0] = sort_func;
  search->sort_order[0] = sort_order;
}

int
sort_add_keys(Search *search, char *key)
{
  int f, n;

  if (key == NULL)
    sort_add_key_func(search, NULL, 0);
  else
    {
      char *field[5], *key_str = fstrdup(key);
      n = split(field, 5, key_str, ',');
      for (f = n - 1; f >= 0; f--)
	{
	  Sort_func func;
	  int order = *field[f] == '~' ? 0 : (*field[f] == '-' ? -1 : +1);
	  if (*field[f] == '~' || *field[f] == '+' || *field[f] == '-')
	    field[f]++;
	  func = sort_str_to_func(field[f]);
	  if (func)
	    sort_add_key_func(search, func, order);
	  else
	    return f + 1;
	}
      free(key_str);
    }

  return 0;
}

void
search_dump(Search *search, FILE *s)
{
  int i;
  if (!verbose)
    return;
  fprintf(s, "Location: %s: %.3f, %.3f\n", search->location_str,
    rad2deg(search->lat), rad2deg(search->lon));
  fprintf(s, "Distance: %.3f%s to %.3f%s; Frequency: %.3f to %.3f\n",
    search->dist_min * search->rad2units, search->unit_name,
    search->dist_max * search->rad2units, search->unit_name,
    search->corf_min, search->corf_max);
  fprintf(s,
    "Options: am=%d, fm=%d, tv=%d; new=%d, old=%d, anc=%d; keep=%d %s\n",
    search->show_am, search->show_fm, search->show_tv,
    search->show_new, search->show_old, search->show_anc, 
    search->keep, sort_func_to_string(search, search->keep_func));
  fprintf(s, "Sorted: ");
  for (i = 0; i < N_SORT_FUNCS; i++)
    fprintf(s, " %+1d %s,", 
      search->sort_order[i], sort_func_to_string(search,search->sort_func[i]));
  fprintf(s, "\n");
}

const char *optchars = "?HUVSb:d:k:l:o:r:R:s:tu:v";

const struct option 
long_options_array[] =
  {
    { "help",     no_argument,       NULL, 'H' },
    { "help",     no_argument,       NULL, '?' },
    { "usage",    no_argument,       NULL, 'U' },
    { "version",  no_argument,       NULL, 'V' },
    { "verbose",  no_argument,       NULL, 'v' },
    { "bands",    required_argument, NULL, 'b' },
    { "distance", required_argument, NULL, 'd' },
    { "keep",     required_argument, NULL, 'k' },
    { "location", required_argument, NULL, 'l' },
    { "options",  required_argument, NULL, 'o' },
    { "range",    required_argument, NULL, 'r' },
    { "sort",     required_argument, NULL, 's' },
    { "tsv",      no_argument,       NULL, 't' },
    { "details",  required_argument, NULL, 'u' },
    { NULL,       no_argument,       NULL, '\0' }
  };
const struct option *
long_options = (const struct option *)&long_options_array;

/*
 * usage -- summarizes options for station-info-cli program.
 */
static void
usage(void)
{
  printf(
    "Usage: %s [-?] [-H] [--help] [-U] [--usage] [-V] [--version]\n"
    "        [-v] [--verbose]\n"
    "        [-b BANDS] [-d DIST] [-k MAX] [-l LOC] "
    "[-o OPTS]  [-r CORF] [-s SORT] [-t] [--tsv] [-u] [--details]\n",
    prog_name);
}

/*
 * help -- briefly describes options for station-info-cli program.
 */
static void
help(void)
{
  usage();
  printf(
    "\n"
    "Search and display information in the FCC databases.\n"
    "\n"
    "  -H, --help                  Display this help and exit\n"
    "  -U, --usage                 Display brief usage message\n"
    "  -V, --version               Output version information and exit\n"
    "  -v, --verbose               Output some debugging information\n"
    "\n"
    "  -b, --bands=BANDS           Bands: am,fm,tv\n"
    "  -d, --distance=DIST         Distance: [min,] max[km,mi,nm]\n"
    "  -k, --keep=MAX              Max stations per channel: max\n"
    "  -l, --location=LOC          Location: lat,lon\n"
    "  -o, --options=OPTS          Options: new,old,anc\n"
    "  -r, --range=CORF            Channel or Frequency range: [min,] max\n"
    "  -s, --sort=KEYS             Sort keys: freq, dist, power\n"
    "\n"
    "  -t, --tsv                   Emit tab-seperated values.\n"
    "  -u, --details=BAND,FAC,ENG  Emit details.\n"
    );
}

static void
version(void)
{
  printf(
    "%s, version %s\n"
    "Copyright 2001, 2002, John Kodis, <kodis@acm.org>\n"
    "\n"
    "This is free software; see the source for copying conditions.\n"
    "There is NO warranty; not even for MERCHANTABILITY\n"
    "or FITNESS FOR A PARTICULAR PURPOSE.\n"
    , PACKAGE, VERSION);
}

static int tsv;

int emit_tsv(void)
{
  return tsv;
}

static int details_flag;
static band_enum details_band;
static off_t details_fac_off;
static off_t details_eng_off;

int
emit_details(band_enum *band, off_t *fac_off, off_t *eng_off)
{
  if (details_flag)
    {
      *band = details_band;
      *fac_off = details_fac_off;
      *eng_off = details_eng_off;
      return 1;
    }
  return 0;
}

int
process_option(Search *search, int c, char *optarg)
{
  int f, err;
  char *field[10];

  debug("processing option '%c' with argument '%s'", c, optarg);
  errno = 0;
  switch (c)
    {
    case 'V':
      version();
      exit(EXIT_SUCCESS);

    case 'U':
      usage();
      exit(EXIT_SUCCESS);

    case 'H':
    case '?':
      help();
      exit(EXIT_SUCCESS);

    case 'A':
      search->interesting_antennas = 1;
      break;

    case 'b':
      search->show_am = search->show_fm = search->show_tv = 0;
      split(field, 10, fstrdup(optarg), ',');
      if (field[0] == NULL)
	fatal("at least one band must be supplied");
      for (f = 0; field[f]; f++)
	if (streq(field[f], "am")) search->show_am = 1;
	else if (streq(field[f], "fm")) search->show_fm = 1;
	else if (streq(field[f], "tv")) search->show_tv = 1;
	else fatal("unrecognized band: %s", field[f]);
      debug("show: am=%d, fm=%d, tv=%d",
	search->show_am, search->show_fm, search->show_tv);
      break;

    case 'k':
      if (search_set_keep(search, fstrdup(optarg)))
	fatal("can't parse count \"%s\": %s", optarg, search->err_msg);
      debug("keep = %d", search->keep);
      break;

    case 'l':
      if (search_set_location(search, optarg))
	fatal("can't parse location \"%s\": %s", optarg, search->err_msg);
      debug("Location: %s: %.3f, %.3f",
	search->location_str, rad2deg(search->lat), rad2deg(search->lon));
      break;

    case 'd':
      if (search_set_distance(search, fstrdup(optarg)))
	fatal("can't parse distance \"%s\": %s", optarg, search->err_msg);
      debug("dist: %g %s .. %g %s",
	search->dist_min * search->rad2units, search->unit_name,
	search->dist_max * search->rad2units, search->unit_name);
      break;

    case 'o':
      search->show_old = search->show_new = search->show_anc = 0;
      split(field, 5, fstrdup(optarg), ',');
      if (field[0] == NULL)
	fatal("at least one option must be supplied");
      for (f = 0; field[f]; f++)
	debug("show: %d: (%s)", f, field[f]);
      for (f = 0; field[f]; f++)
	if (streq("old", field[f])) search->show_old = 1;
	else if (streq("new", field[f])) search->show_new = 1;
	else if (streq("anc", field[f])) search->show_anc = 1;
	else fatal("unrecognized show option: %s", field[f]);
      debug("show: old=%d, new=%d, anc=%d",
	search->show_old, search->show_new, search->show_anc);
      break;

    case 'r':
      if (search_set_corf(search, fstrdup(optarg)))
	fatal("can't parse channel or frequency \"%s\": %s",
	  optarg, search->err_msg);
      debug("range = %.1f,%.1f", search->corf_min, search->corf_max);
      break;

    case 's':
      sort_add_keys(search, NULL);
      if ((err = sort_add_keys(search, optarg)) != 0)
	fatal("unrecognized sort key %d in %s", err, optarg);
      break;

    case 't':
      tsv = 1;
      break;

    case 'u':
      if (split(field, 5, fstrdup(optarg), ',') != 3)
	fatal("the details option requires exactly three arguments");
      details_band = strtoul(field[0], NULL, 0);
      details_fac_off = strtoul(field[1], NULL, 0);
      details_eng_off = strtoul(field[2], NULL, 0);
      details_flag = 1;
      break;

    case 'v':
      verbose = 1;
      break;

    default:
      usage();
      exit(EXIT_FAILURE);

    case 0:
      break;
    }
  return 0;
}

void
print_ingest_progress(char *fn, int recs, int entries, int alloc)
{
  static struct timeval t0, tn;

  if (fn == NULL)
    gettimeofday(&t0, NULL);
  else
    {
      gettimeofday(&tn, NULL);
      printf("%12s: %3.1f secs, %'7d recs, %'7d entries",
	fn, tdiff(&t0, &tn), recs, entries);
      if (alloc)
	printf(" (%.0f%% of %'7d)", entries * 100. / alloc, alloc); 
      printf("\n");
      t0 = tn;
    }
}
