#!/usr/bin/env perl

#
#   Copyright (C) Dr. Heinz-Josef Claes (2012)
#                 hjclaes@web.de
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.

#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.
#


$main::STOREBACKUPVERSION = undef;


use POSIX;
use strict;
use warnings;



sub libPath
{
    my $file = shift;

    my $dir;

    # Falls Datei selbst ein symlink ist, solange folgen, bis aufgelöst
    if (-f $file)
    {
	while (-l $file)
	{
	    my $link = readlink($file);

	    if (substr($link, 0, 1) ne "/")
	    {
		$file =~ s/[^\/]+$/$link/;
            }
	    else
	    {
		$file = $link;
	    }
	}

	($dir, $file) = &splitFileDir($file);
	$file = "/$file";
    }
    else
    {
	print STDERR "<$file> does not exist!\n";
	exit 1;
    }

    $dir .= "/../lib";           # Pfad zu den Bibliotheken
    my $oldDir = `/bin/pwd`;
    chomp $oldDir;
    if (chdir $dir)
    {
	my $absDir = `/bin/pwd`;
	chop $absDir;
	chdir $oldDir;

	return (&splitFileDir("$absDir$file"));
    }
    else
    {
	print STDERR "<$dir> does not exist, exiting\n";
    }
}
sub splitFileDir
{
    my $name = shift;

    return ('.', $name) unless ($name =~/\//);    # nur einfacher Dateiname

    my ($dir, $file) = $name =~ /^(.*)\/(.*)$/s;
    $dir = '/' if ($dir eq '');                   # gilt, falls z.B. /filename
    return ($dir, $file);
}
my ($req, $prog) = &libPath($0);
(@INC) = ($req, @INC);


require 'checkParam2.pl';
require 'checkObjPar.pl';
require 'prLog.pl';
require 'version.pl';
require 'fileDir.pl';
require 'forkProc.pl';
require 'storeBackupLib.pl';

my $checkSumFile = '.md5CheckSums';


=head1 NAME

storeBackupCheckSource.pl - compares unchaged files in source with their
                            md5 sums in the backup

=head1 DESCRIPTION

The tool is intended to find files in the source that might have changed over 
time without the users interaction or knowledge, for example by bit rot.

If a file is unchanged (same ctime, mtime, size) in the source directory
compared to the backup, it:

- prints an ERROR message if the md5 sum differs

- prints a WARNING if permissions, uid or gid differs

- prints MISSING if file is not in the source directory (option -v)

- prints INFO if file is identical (option -v)

=head1 SYNOPSIS

    storeBackupCheckSource.pl -s sourceDir -b singleBackupDir [-v]

=head1 OPTIONS

=over 8

=item B<--sourceDir>, B<-s>

    source directory of backup when running storeBackup.pl

=item B<--singleBackupDir>, B<-b>

    directory of the backup to compaire sourceDir with
    this must be *one* single backup directory
    (eg. 2012.08.08_02.00.11)

=item B<--verbose>, B<-v>

    also print positive messages (file is identical in source and backup)

=back

=head1 COPYRIGHT

Copyright (c) 2012 by Heinz-Josef Claes (see README).
Published under the GNU General Public License v3 or any later version

=cut

my $Help = join('', grep(!/^\s*$/, `pod2text $0`));
$Help = "cannot find pod2text, see documentation for details\n"
    unless $Help;

&printVersion(\@ARGV, '-V');

my $CheckPar =
    CheckParam->new('-allowLists' => 'no',
		    '-list' => [Option->new('-name' => 'singleBackupDir',
					    '-cl_option' => '-b',
					    '-cl_alias' => '--singleBackupDir',
					    '-param' => 'yes',
					    '-must_be' => 'yes'),
                                Option->new('-name' => 'sourceDir',
					    '-cl_option' => '-s',
					    '-cl_alias' => '--sourceDir',
					    '-param' => 'yes',
					    '-must_be' => 'yes'),
				Option->new('-name' => 'verbose',
					    '-cl_option' => '-v',
					    '-cl_alias' => '--verbose')
				]);

$CheckPar->check('-argv' => \@ARGV,
                 '-help' => $Help
                 );


my $singleBackupDir = $CheckPar->getOptWithPar('singleBackupDir');
my $sourceDir = $CheckPar->getOptWithPar('sourceDir');
my $verbose = $CheckPar->getOptWithoutPar('verbose');

my $prLog;
my ($prLogKind) = ['A:BEGIN',
		   'Z:END',
		   'I:INFO',
		   'W:WARNING',
		   'M:MISSING',
		   'E:ERROR'];
$prLog = printLog->new('-kind' => $prLogKind);


$prLog->print('-kind' => 'E',
	      '-str' => ["cannot access sourceDir <$sourceDir>"],
	      '-exit' => 1)
    unless -d $sourceDir;

$prLog->print('-kind' => 'E',
	      '-str' => ["this is not a full backup",
			 "please run storeBackupUpdateBackup.pl on <$singleBackupDir>"],
	      '-exit' => 1)
    if -e "$singleBackupDir/.storeBackupLinks/linkFile.bz2";



$prLog->print('-kind' => 'A',
	      '-str' => ["comparing <$singleBackupDir> to <$sourceDir>"]);


my $rcsf = readCheckSumFile->new('-checkSumFile' =>
				 "$singleBackupDir/$checkSumFile",
				 '-prLog' => $prLog);

my ($md5sum, $compr, $devInode, $inodeBackup, $ctime, $mtime, $atime,
    $size, $uid, $gid, $mode, $filename);
while ((($md5sum, $compr, $devInode, $inodeBackup, $ctime, $mtime, $atime,
	 $size, $uid, $gid, $mode, $filename) = $rcsf->nextLine()) > 0)
{
    if ($devInode =~ /\ADEVICE/)
    {
	$prLog->print('-kind' => 'I',
		      '-str' => ["ignoring stored device <$filename>"])
	    if $verbose;
	next;
    }

    my $f = "$sourceDir/$filename";
    if (length($md5sum) == 32)   # normal file
    {
	# check if it exists with same time stamp in sourceDir
	if (-e $f)
	{
	    unless (-r $f)
	    {
		$prLog->print('-kind' => 'E',
			      '-str' => ["file <$f> exists but is unreadable"]);
		next;
	    }
	    my ($actMode, $actUid, $actGid, $actCtime, $actMtime,
		$actAtime, $actSize) =
		    (stat($f))[2, 4, 5, 10, 9, 8, 7];
	    $actMode = 0 unless $actMode;
	    $actMode &= 07777;

	    if ($ctime == $actCtime and $mtime == $actMtime and
		$size == $actSize)
	    {
		my $actMd5 = &::calcMD5($f, $prLog);

		unless ($actMd5)
		{
		    $prLog->print('-kind' => 'E',
				  '-str' => ["cannot calculate md5 sum of <$f>"]);
		    next;
		}
		if ($md5sum eq $actMd5)   # looks good
		{
		    $prLog->print('-kind' => 'I',
				  '-str' => ["<$f> identical to backup"])
			if $verbose;

		    $prLog->print('-kind' => 'W',
				  '-str' =>
				  ["<$f> has same md5 sum but different permissions " .
				   "than in backup"])
			if ($mode ne $actMode);
		    $prLog->print('-kind' => 'W',
				  '-str' =>
				  ["<$f> has same md5 sum but different uid " .
				   "than in backup"])
			if ($uid ne $actUid);
		    $prLog->print('-kind' => 'W',
				  '-str' =>
				  ["<$f> has same md5 sum but different gid " .
				   "than in backup"])
			if ($gid ne $actGid);
		}
		else
		{
		    $prLog->print('-kind' => 'E',
				  '-str' =>
				  ["<$f> has same ctime / mtime / size " .
				   "as in backup but different md5 sums: $actMd5 in " .
				   "sourceDir ; $md5sum in backup"]);
		}
	    }
	    else
	    {
		my $actMd5 = ($size == $actSize) ? &::calcMD5($f, $prLog) : undef;
		if ($md5sum eq $actMd5)
		{
		    $prLog->print('-kind' => 'I',
				  '-str' =>
				  ["<$filename> has different ctime / mtime size " .
				   "but same md5 sum"])
			if $verbose;
		}
		else
		{
		    $prLog->print('-kind' => 'I',
				  '-str' => ["<$f> differs from backup"])
			if $verbose
		}
	    }
	}
	else
	{
	    $prLog->print('-kind' => 'M',
			  '-str' =>
			  ["<$filename> is missing in the source directory"])
		if $verbose;
	}
    }
}

$prLog->print('-kind' => 'Z',
	      '-str' => ["comparing <$singleBackupDir> to <$sourceDir>"]);

exit 0;
