cvs_id = "$Id: PollManager.py,v 1.41 2003/11/30 00:49:34 juri Exp $"

import straw
import time
import gtk
import socket

class PollManager:
    def __init__(self):
        self._config = straw.Config.get_instance()
        self._mmgr = straw.MessageManager.get_instance()

        self._config.signal_connect(straw.PollFrequencyChangedSignal,
                                    self.poll_changed)
        self._config.signal_connect(straw.OfflineModeChangedSignal,
                                    self.offline_changed)

        self.pollers = []
        self._is_offline = self._config.offline
        self._poll_frequency = self._config.poll_frequency
        self._last_poll = self._config.last_poll

    def poll_changed(self, signal):
        self._poll_frequency = signal.value

    def offline_changed(self, signal):
        self._is_offline = self._config.offline

    def start_polling_loop(self):
        gtk.timeout_add(straw.NetworkConstants.POLL_INTERVAL, self.maybe_poll)

    def maybe_poll(self):
        time_diff = int(time.time()) - self._last_poll

        if self._poll_frequency > 0 and not self._is_offline and time_diff > self._poll_frequency:
            feeds = straw.FeedList.get_instance().flatten_list()
            self._config.last_poll = self._last_poll = int(time.time())
            self.poll(feeds)

        self.poll_network()
        gtk.timeout_add(straw.NetworkConstants.POLL_INTERVAL, self.maybe_poll)

    def poll(self, obj):

        self._mmgr.post_message(_("Polling feeds..."))

        for f in obj:
            self.pollers.append(f)
            p = Poller(f, self)
            p.poll()

    def poll_network(self):
        try:
            straw.URLFetch.connection_manager.poll(straw.NetworkConstants.POLL_TIMEOUT)
        except socket.error:
            straw.hig_alert.reportError(_("Poll Error"),_("Cannot connect to feed."))

    def poller_finished(self, feed):
        self.pollers.remove(feed)

class Poller:
    def __init__(self, feed, pm):
        self._feed = feed
        self._pm = pm
        self._mmgr = straw.MessageManager.get_instance()

    def poll(self):
        url, user, pw = self._feed.get_access_info()
        parsed = None

        headers = {}
        if self._feed.previous_etag is not None:
            headers['If-None-Match'] = self._feed.previous_etag
        try:
            pc = straw.URLFetch.connection_manager.request(
                url, self, headers, user, pw,
                priority=straw.NetworkConstants.PRIORITY_RSS)
            self._feed.router.start_polling()
        except Exception, e:
            self.http_failed(e)

    def http_results(self, status, header, data):
        success = self.update_feed(status, header, data)
        if success:
            self._mmgr.post_message(_("Parsing data of %s done.") % self._feed.title)
        else:
            self._mmgr.post_message(_("Parsing data of %s failed.") % self._feed.title)
        self._feed.poll_done()
        self._pm.poller_finished(self._feed)

    def update_feed(self, status, header, data):
        self._mmgr.post_message(_("Parsing data for feed %s...") % self._feed.title)

        if status is None:
            self._feed.router.set_error(_("The feed returned no data."))
            return 0
        elif status[1] == 304:
            self._feed.router.route_no_data()
            return 1
        elif status[1] == 410 or status[1] == 404:
            self._feed.router.set_error(_("Straw was unable to find the feed."))
            return 0
        elif status[1] != 200 and status[1] > 299:
            self._feed.router.set_error(_("Polling feed resulted in abnormal status '%s' (code %d)") % (status[2], status[1]))
            return 0

        parsed = ""

        try:
            parsed = straw.SummaryParser.parse(data, self._feed)
        except Exception, e:
            self._feed.router.set_error((_("Unknown exception when parsing RSS: %s") % str(e)))
            return 0

        self._feed.router.route_all(header, parsed)
        return 1

    def http_failed(self, exception):
        if isinstance(exception, socket.error):
            err = exception.args[1]
        else:
            err = str(exception)
        self._feed.router.set_error(err)
        self._mmgr.post_message(_("Polling of feed %s failed") % self._feed.title)
        self._feed.poll_done()
        self._pm.poller_finished(self._feed)

    def http_permanent_redirect(self, location):
        (oldloc, u, p) = self._feed.access_info
        self._feed.access_info = (location, u, p)

pollmanager_instance = None

def get_instance():
    global pollmanager_instance
    if pollmanager_instance is None:
        pollmanager_instance = PollManager()
    return pollmanager_instance
