cvs_id = "$Id: Main.py,v 1.3 2004/04/22 18:41:16 jpakaste Exp $"

import straw

import os
import sys

import pygtk
pygtk.require('2.0')

import gettext
import gnome
import gnome.ui
import gobject
import gtk
import gtkhtml2
import pango
import locale
from gtk import glade
import getopt
from error import log


class Main:
    def __init__(self):
        lname = straw.APPNAME.lower()
        gnome.program_init(lname, straw.VERSION)
        straw.main = self

        try:
            localedir = find_locale_dir()
            gettext.bindtextdomain(lname, localedir)
            gettext.textdomain(lname)
            gettext.install(lname, localedir, unicode=1)
            glade.bindtextdomain(lname, localedir)
        except IOError:
            def broken_gettext_workaround(s):
                return s
            __builtins__.__dict__['_'] = broken_gettext_workaround

        self._setlocale()

        self._libdir = find_lib_dir()
        self._xmlfile = find_glade_file(self._libdir)
        straw.ImageCache.initialize()
        self._image_loader = ImageLoader(self._libdir)

        self._visible_feed = None
        self._visible_item = None
        self._visible_category = None
        self._feed_selection_enabled = 1

        config = straw.Config.get_instance()
        feedlist = straw.FeedList.get_instance()
        feed_categories = straw.FeedCategoryList.get_instance()

        if config.first_time:
            self.import_opml(
                os.path.join(self._libdir, "default_subscriptions.opml"))
        else:
            feedlist.load_data()
            feed_categories.load_data()

        try:
            optlist, args = getopt.getopt(sys.argv[1:], '', ['offline'])
            if '--offline' in [o[0] for o in optlist]:
                config.offline = 1
        except getopt.GetoptError, e:
            log(e.msg)

        straw.main_window = straw.MainWindow(
            glade.XML(self._xmlfile, "straw_main", gettext.textdomain()))
        self._feed_properties_xml = glade.XML(
            self._xmlfile, 'feed_properties_dialog', gettext.textdomain())
        self._feed_properties_dialogs = {}
        self._preferences_dialog = straw.PreferencesDialog(
            glade.XML(self._xmlfile, "preferences_dialog",
                      gettext.textdomain()))
        self._subscribe_dialog = straw.SubscribeDialog(
            glade.XML(self._xmlfile, 'subscribe_dialog',
                      gettext.textdomain()))
        self._find_dialog = straw.FindDialog(
            glade.XML(self._xmlfile, 'find_dialog',
                      gettext.textdomain()))

        straw.main_window.display_categories()
        self.update_feeds(feedlist)

        feedlist.signal_connect(
            straw.FeedsChangedSignal, self.feeds_changed)
        feed_categories.signal_connect(
            straw.FeedCategoriesChangedSignal, self.feed_categories_changed)

        self._itemstore = straw.ItemStore()
        self.restore_feeds(self._itemstore)
        self._itemstore.start()

        straw.main_window.get_window().show()
        self.display_feeds()
        straw.PollManager.get_instance().start_polling_loop()


    def _setlocale(self):
        locale.setlocale(locale.LC_ALL, '')

    def get_icon(self):
        return self._libdir + "/straw.png"


    def restore_feeds(self, itemstore):
        feedlist = straw.FeedList.get_instance()
        for f in feedlist.flatten_list():
            f.restore_items(itemstore.read_feed_items(f))

    # called when FeedsChangedSignal is sent
    def feeds_changed(self, signal):
        self.update_feeds(signal.sender)
        t = isinstance(signal, straw.FeedDeletedSignal)
        if t:
            if signal.feed is self.get_visible_feed():
                category = self.get_visible_category()
                if len(category) > 0:
                    self.display_feed(category[0])
                else:
                    self.display_feed(None)
        return


    def feed_categories_changed(self, signal):
        straw.main_window.display_categories()
        self._preferences_dialog.create_feed_model()

    def update_feeds(self, feeds):
        self._preferences_dialog.create_feed_model()
        straw.main_window.display_feeds(self._visible_category, self._visible_feed)
        return

    def display_feeds(self):
        straw.main_window.display_feeds(self._visible_category, self._visible_feed)
        return

    def get_visible_feed(self):
        return self._visible_feed

    def get_visible_category(self):
        return self._visible_category

    def set_visible_category(self, category):
        self._visible_category = category

    def get_visible_item(self):
        return self._visible_item

    def mark_as_read(self, feed = None):
        if feed is None:
            feed = self._visible_feed
        feed.mark_all_read()

    def display_feed(self, feed):
        if self._feed_selection_enabled and self._visible_feed != feed:
            self._visible_feed = feed
            if feed is None:
                straw.main_window.display_no_feed()
            else:
                if feed.number_of_items == 0:
                    straw.main_window.display_empty_feed(feed)
                else:
                    straw.main_window.display_feed(feed)
                #straw.feed_properties.display_properties(self._visible_feed)
            return 1
        else:
            return 0

    def display_item_unselected(self, item):
        if item and item is not self._visible_item:
            if item.feed is not self._visible_feed:
                straw.main_window.display_feed_unselected(item.feed)
            straw.main_window.display_item_unselected(item)
            straw.DashboardFrontend.get_instance().item_changed(item)

    def display_item(self, item):
        if item and item is not self._visible_item:
            if item.feed is not self._visible_feed:
                straw.main_window.display_feed_unselected(item.feed)
            self._visible_item = item
            item.seen = 1
            straw.main_window.display_item(item)
            straw.DashboardFrontend.get_instance().item_changed(item)

    def get_next_item(self, item = None):
        feeds = self.get_visible_category() #straw.FeedList.get_instance().flatten_list()
        if item is None:
            item = self._visible_item
        if item is None:
            if not len(feeds):
                return None
            feed = feeds[0]
            return feed.get_first_item()
        if item.feed in feeds:
            feed = item.feed
            ni = feed.get_next_item(item)
            if ni is not None:
                return ni
            feed_idx = feeds.index(feed)
        else:
            feed = feeds[0]
            feed_idx = 0

        fi = 1
        while fi <= len(feeds):
            item = feeds[
                (feed_idx + fi) % len(feeds)].get_first_item()
            if item is not None:
                return item
            fi += 1
        return None

    def display_next_item(self, item = None):
        self.display_item_unselected(self.get_next_item(item))

    def get_next_unread_item(self, item = None):
        current = item
        first = None
        while 1:
            item = self.get_next_item(item)
            # so we can check if we have gone around even if the user didn't
            # have anything selected
            if current is None:
                current = item
            if first is None:
                first = item
            elif first is item:
                return None
            if item is None:
                return None
            if not item.seen:
                return item
            if item is current:
                return None

    def display_next_unread_item(self, item = None):
        item = self.get_next_unread_item(
            [item, self._visible_item][item is None])
        if item:
            self.display_item_unselected(item)

    def get_previous_item(self, item = None):
        feeds = self.get_visible_category() #straw.FeedList.get_instance().flatten_list()
        if item is None:
            item = self._visible_item
        if item is None:
            if not len(feeds):
                return None
            feed = feeds[-1]
            return feed.get_last_item()
        if item.feed in feeds:
            feed = item.feed
            ni = feed.get_previous_item(item)
            if ni is not None:
                return ni
            feed_idx = feeds.index(feed)
        else:
            feed = feeds[-1]
            feed_idx = len(feeds)
        fd = 1
        while fd <= len(feeds):
            item = feeds[
                (feed_idx - fd) % len(feeds)].get_last_item()
            if item is not None:
                return item
            fd += 1
        return None

    def display_previous_item(self, item = None):
        self.display_item_unselected(self.get_previous_item(item))

    def get_next_feed(self, item = None):
        feeds = self.get_visible_category() #straw.FeedList.get_instance().flatten_list()
        feed = None
        if item is None:
            item = self._visible_item
        if item is None:
            if len(feeds):
                feed = feeds[0]
        else:
            if item.feed in feeds:
                feed_idx = feeds.index(item.feed)
            else:
                feed_idx = -1
            if feed_idx < len(feeds) - 1:
                feed_idx += 1
            else:
                feed_idx = 0
            feed = feeds[feed_idx]
        return feed

    def display_next_unread_feed(self, item = None):
        feed = self.get_next_feed(item)
        item = feed.get_first_item()
        if item.seen:
            item = self.get_next_unread_item(item)
        if item:
            self.display_item_unselected(item)

    def display_next_feed(self, item = None):
        feed = self.get_next_feed(item)
        item = feed.get_first_item()
        if item:
            self.display_item_unselected(item)

    def get_previous_feed(self, item = None):
        feeds = self.get_visible_category() #straw.FeedList.get_instance().flatten_list()
        feed = None
        if item is None:
            item = self._visible_item
        if item is None:
            if len(feeds):
                feed = feeds[-1]
        else:
            if item.feed in feeds:
                feed_idx = feeds.index(item.feed)
            else:
                feed_idx = len(feeds)
            if feed_idx > 0:
                feed_idx -= 1
            else:
                feed_idx = len(feeds) - 1
            feed = feeds[feed_idx]
        return feed

    def display_previous_feed(self, item = None):
        feed = self.get_previous_feed(item)
        item = feed.get_first_item()
        if item:
            self.display_item_unselected(item)

    def get_next_category(self, category = None):
        if category is None:
            category = self.get_visible_category()
        fclist = straw.FeedCategoryList.get_instance()
        allcats = fclist.user_categories + list(fclist.pseudo_categories)
        if category is None:
            category = allcats[0]
        else:
            index = allcats.index(category)
            if index < len(allcats) - 1:
                index += 1
            else:
                index = 0
            category = allcats[index]
        return category

    def display_next_category(self, category = None):
        category = self.get_next_category(category)
        self.set_visible_category(category)
        straw.main_window.display_feeds(category, self.get_visible_feed())

    def get_previous_category(self, category = None):
        if category is None:
            category = self.get_visible_category()
        fclist = straw.FeedCategoryList.get_instance()
        allcats = fclist.user_categories + list(fclist.pseudo_categories)
        if category is None:
            category = allcats[-1]
        else:
            index = allcats.index(category)
            if index > 0:
                index -= 1
            else:
                index = len(allcats) - 1
            category = allcats[index]
        return category

    def display_previous_category(self, category = None):
        category = self.get_previous_category(category)
        self.set_visible_category(category)
        straw.main_window.display_feeds(category, self.get_visible_feed())
    
    def is_visible_item(self, item):
        return item is self._visible_item

    def show_preferences_dialog(self):
        self._preferences_dialog.show()

    def show_subscribe_dialog(self, url=None):
        self._subscribe_dialog.show(url)

    def show_find_dialog(self):
        self._find_dialog.show()

    def show_feed_properties(self, feed):
        class _dialogdeleter:
            def __init__(self, feed, hash):
                self.feed = feed
                self.hash = hash

            def __call__(self):
                del self.hash[feed]
                
        dialog = self._feed_properties_dialogs.get(feed, None)
        if dialog is not None:
            dialog.show(feed)
        else:
            dialog = straw.FeedPropertiesDialog(
                glade.XML(self._xmlfile, 'feed_properties_dialog',
                          gettext.textdomain()))
            self._feed_properties_dialogs[feed] = dialog
        dialog.show(feed, _dialogdeleter(feed, self._feed_properties_dialogs))

    def enable_feed_selection(self):
        straw.main_window.enable_feed_selection()

    def disable_feed_selection(self):
        straw.main_window.disable_feed_selection()

    def mainloop(self):
        gtk.main()

    def quit(self):
        gtk.mainquit()
        self._itemstore.stop()
        return

    def show_image(self, widget, image):
        self._image_loader.show(widget, image)

    def poll(self, obj):
        manager = straw.PollManager.get_instance()
        offline = straw.Config.get_instance().offline
        if offline:
            response = straw.hig_alert.reportOfflineStatus()

            if response == gtk.RESPONSE_OK:
                straw.Config.get_instance().offline = not offline
            else:
                return

        manager.poll(obj)

    def is_offline(self):
        return straw.Config.get_instance().offline

    def import_opml(self, filename, category=None):
        opml = straw.OPMLImport.read(open(filename))
        if opml is None:
            return
        feeds = straw.FeedList.get_instance()
        current_feeds = {}
        error = 0
        for feed in feeds:
            current_feeds[feed.access_info[0]] = 1

        for b in opml:
            if current_feeds.has_key(b.xmlurl):
                continue
            if b.type == "rss":
                feeds.append( category,
                              straw.create_new_feed(b.title, b.xmlurl))
            else:
                error = 1

        if error:
            straw.hig_alert.reportError(
                _("Warning: Not All Feeds Have Been Added"),
                (_("Some feeds in the file %s were not added because they are not RSS feeds.") % filename))

        return


class ImageLoader:
    def __init__(self, libdir):
        self.libdir = libdir

    def show(self, widget, image):
        fn = os.path.join(self.libdir, image + ".png")
        widget.set_from_file(fn)

def find_lib_dir():
    if os.environ.has_key("STRAW_DATA"):
        return os.environ["STRAW_DATA"]
    h, t = os.path.split(os.path.split(os.path.abspath(sys.argv[0]))[0])
    if t == 'bin':
        fp = os.path.join(h, 'share', 'straw')
        if os.path.isdir(fp):
            return fp
        fp = os.path.join(h, 'lib', 'straw')
        if os.path.isdir(fp):
            return fp
    raise "FileNotFoundError", "couldn't find Straw data directory"

def find_locale_dir():
    h, t = os.path.split(os.path.split(os.path.abspath(sys.argv[0]))[0])
    if t == 'bin':
        fp = os.path.join(h, 'share', 'locale')
        if os.path.isdir(fp):
            return fp
    raise "FileNotFoundError", "couldn't find locale data directory"

def find_glade_file(libdir):
    xmlfile = None
    if os.environ.has_key("STRAW_GLADE"):
        return os.environ["STRAW_GLADE"]
    fp = os.path.join(libdir, 'straw.glade')
    if os.path.isfile(fp):
        return fp
    raise "FileNotFoundError", "couldn't find straw.glade"

