from gtk import glade
import straw

__doc__ = "A MVP framework. Contains base classes for views and presenters."

class WidgetView(object):
    """Base class for widget-based view objects."""
    def __init__(self, widget, model = None, presenter = None):
        self._widget = widget
        self._model = model
        self._presenter = presenter

        if model is not None:
            self._model_set()
        if presenter is not None:
            self._presenter_set()

        self._initialize()

        xml = glade.get_widget_tree(widget)
        nameFuncMap = {}
        for key in dir(self.__class__):
            if key[:4] == '_on_':
                nameFuncMap[key[1:]] = getattr(self, key)
        xml.signal_autoconnect(nameFuncMap)

    def _initialize(self):
        """Called from the constructor after the instance variables have
        been set and the corresponding initialization functions have been
        called, but before connecting gtk signals."""
        pass

    def _model_set(self):
        """Called when the model is set."""
        pass

    def _presenter_set(self):
        """Called when the presenter is set."""
        pass

    def get_model(self):
        return self._model

    def set_model(self, model):
        self._model = model
        self._model_set()
    model = property(get_model, set_model, None, "The model object.")

    def get_presenter(self):
        return self._presenter

    def set_presenter(self, presenter):
        self._presenter = presenter
        self._presenter_set()
    presenter = property(get_presenter, set_presenter, None,
                         "The presenter object.")

class BasicPresenter(object, straw.SignalEmitter):
    """Base class for basic (non-composite) presenters."""
    def __init__(self, model = None, view = None):
        straw.SignalEmitter.__init__(self)

        self._model = model
        self._view = view

        if model is not None:
            self._model_set()
        if view is not None:
            view.model = model
            view.presenter = self
            self._view_set()

        self._initialize()

    def _initialize(self):
        pass

    def _model_set(self):
        pass

    def _view_set(self):
        pass

    def get_view(self):
        return self._view

    def set_view(self, view):
        self._view = view
        view.presenter = self
        view.model = self._model
        self._view_set()

    view = property(get_view, set_view, None,
                    """The view component. Setting this als sets the view
                    object's presenter and model attributes.""")

    def get_model(self):
        return self._model

    def set_model(self, model):
        self._model = model
        if self._view is not None:
            self._view.model = model
        self._model_set()

    model = property(get_model, set_model, None,
                     """The model component. Setting this also sets the view
                     object's (if defined) model attribute.""")
