/*
 * $Id: st-dialog.c,v 1.22 2004/01/27 14:16:42 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Here are the streamtuner dialog functions.
 *
 * They must be callable indifferently:
 *
 *	- from inside the main thread: from outside/inside the GTK+ main loop
 *	- from outside the main thread: from inside the GTK+ main loop
 *
 * WARNING: we must be careful to not lock a mutex recursively.
 */

#include <stdarg.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include "sg-util.h"
#include "sgtk-dialog.h"
#include "sgtk-message-dialog.h"
#include "st-dialog.h"
#include "st-thread.h"
#include "st-shell.h"
#include "st-search-dialog.h"

#define MUST_WRAP	(st_thread_get() != NULL || gtk_main_level() < 1)
	
/*** function declarations ***************************************************/

static int st_dialog_run (GtkDialog *dialog);

/*** API implementation ******************************************************/

void
st_notice (const char *format, ...)
{
  va_list args;
  char *message;

  va_start(args, format);
  message = g_strdup_vprintf(format, args);
  va_end(args);

  g_printerr("streamtuner: %s\n", message);
  g_free(message);
}

/*
 * Deprecated.
 */
void
st_warning (const char *format, ...)
{
  va_list args;
  char *secondary;

  va_start(args, format);
  secondary = g_strdup_vprintf(format, args);
  va_end(args);

  st_error_dialog(NULL, "%s", secondary);
  g_free(secondary);
}

/*
 * Deprecated.
 */
void
st_error (const char *format, ...)
{
  va_list args;
  char *secondary;

  va_start(args, format);
  secondary = g_strdup_vprintf(format, args);
  va_end(args);

  st_error_dialog(NULL, "%s", secondary);
  g_free(secondary);

  exit(1);
}

void
st_error_dialog (const char *primary,
		 const char *format,
		 ...)
{
  char *secondary = NULL;
  GtkWidget *dialog;
  
  if (format)
    {
      va_list args;
      
      va_start(args, format);
      secondary = g_strdup_vprintf(format, args);
      va_end(args);
    }
  
  if (MUST_WRAP)
    GDK_THREADS_ENTER();
  
  dialog = sgtk_message_dialog_new(st_shell_get_transient(),
				   GTK_STOCK_DIALOG_ERROR,
				   primary,
				   secondary);

  gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_OK, GTK_RESPONSE_OK);

  st_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  if (MUST_WRAP)
    GDK_THREADS_LEAVE();

  g_free(secondary);
}

char *
st_search_dialog (void)
{
  GtkWidget *dialog;
  char *str = NULL;

  if (MUST_WRAP)
    GDK_THREADS_ENTER();

  dialog = st_search_dialog_new(st_shell_get_transient());

  if (st_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
    str = g_strdup(st_search_dialog_get_text(ST_SEARCH_DIALOG(dialog)));

  gtk_widget_destroy(dialog);
  
  if (MUST_WRAP)
    GDK_THREADS_LEAVE();

  return str;
}

/*
 * Deprecated.
 */
int
st_question (const char *format, ...)
{
  va_list args;
  char *message;
  GtkWidget *dialog;
  int response;

  va_start(args, format);
  message = g_strdup_vprintf(format, args);
  va_end(args);

  if (MUST_WRAP)
    GDK_THREADS_ENTER();

  dialog = gtk_message_dialog_new(st_shell_get_transient(),
				  GTK_DIALOG_DESTROY_WITH_PARENT,
				  GTK_MESSAGE_QUESTION,
				  GTK_BUTTONS_NONE,
				  message);
  gtk_window_set_title(GTK_WINDOW(dialog), "");
  gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);

  gtk_dialog_add_buttons(GTK_DIALOG(dialog),
			 GTK_STOCK_YES, GTK_RESPONSE_YES,
			 GTK_STOCK_NO, GTK_RESPONSE_NO,
			 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			 NULL);

  response = st_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  if (MUST_WRAP)
    GDK_THREADS_LEAVE();

  g_free(message);

  return response;
}

/*
 * Deprecated.
 */
char
*st_prompt (gboolean hidden,
	    const char *init,
	    const char *format,
	    ...)
{
  va_list args; 
  char *message;
  GtkWidget *dialog;
  GtkWidget *ok;
  GtkWidget *entry;
  char *str = NULL;
  
  va_start(args, format); 
  message = g_strdup_vprintf(format, args);
  va_end(args);

  /* kludge to make old plugins use the new search dialog */
  if (! hidden && ! strncmp("Enter a search string", message,
			    strlen("Enter a search string")))
    {
      g_free(message);

      return st_search_dialog();
    }
  
  if (MUST_WRAP)
    GDK_THREADS_ENTER();

  dialog = gtk_message_dialog_new(st_shell_get_transient(),
				  GTK_DIALOG_DESTROY_WITH_PARENT,
				  GTK_MESSAGE_QUESTION,
				  GTK_BUTTONS_NONE,
				  message);
  gtk_window_set_title(GTK_WINDOW(dialog), "");
  gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);

  gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
  ok = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_OK, GTK_RESPONSE_OK);
  
  gtk_window_set_default(GTK_WINDOW(dialog), ok);

  entry = gtk_entry_new();

  gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);

  if (hidden)
    gtk_entry_set_visibility(GTK_ENTRY(entry), FALSE);
  if (init)
    gtk_entry_set_text(GTK_ENTRY(entry), init);

  gtk_widget_show(entry);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), entry, TRUE, TRUE, 0);

  if (st_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
    str = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
 
  gtk_widget_destroy(dialog);

  if (MUST_WRAP)
    GDK_THREADS_LEAVE();

  g_free(message);
 
  return str;
}

/*** private implementation **************************************************/

static int
st_dialog_run (GtkDialog *dialog)
{
  int response;

  g_return_val_if_fail(GTK_IS_DIALOG(dialog), -1);

  if (st_thread_get())
    response = sgtk_dialog_run_from_thread(dialog);
  else
    response = gtk_dialog_run(dialog);
  
  return response;
}

/*
 * Calls sg_str_capitalize() on STR, then changes all occurrences of
 * ": Xy" and "(Xy" to ": xy" and "(xy".
 *
 * Meant to be used as a pretty-printer for secondary strings.
 */
char *
st_dialog_normalize (const char *str)
{
  char *normalized;
  int i;
  gboolean attn1 = FALSE;
  gboolean attn2 = FALSE;

  normalized = sg_str_capitalize(str);

  for (i = 0; normalized[i]; i++)
    {
      if (attn1 && attn2 && g_ascii_islower(normalized[i + 1]))
	{
	  normalized[i] = g_ascii_tolower(normalized[i]);
	  attn1 = attn2 = FALSE;
	}
      else if (attn1)
	{
	  if (normalized[i] == ' ')
	    attn2 = TRUE;
	}
      else
	{
	  if (normalized[i] == ':')
	    attn1 = TRUE;
	  else if (normalized[i] == '(')
	    attn1 = attn2 = TRUE;
	}
    }

  return normalized;
}
