/*
 * $Id: st-menubar.c,v 1.64 2004/01/30 15:04:36 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-util.h"
#include "sgtk-util.h"
#include "st-stock.h"
#include "st-menubar.h"
#include "st-settings.h"
#include "st-handlers.h"
#include "st-stream-view.h"
#include "st-shell.h"
#include "st-action.h"
#include "st-dialog.h"

/*** function declarations ***************************************************/

static void st_menubar_init			(STMenubar	    *menubar);
static void st_menubar_complete_stream_menu	(STMenubar	    *menubar);
static void st_menubar_init_directories_menu	(STMenubar	    *menubar);
static void st_menubar_complete_directories_menu(STMenubar	    *menubar);
static void st_menubar_directories_toggled_h	(GtkCheckMenuItem   *item,
						 gpointer	    data);
static void st_menubar_init_view_item		(STMenubar	    *menubar,
						 const char	    *path,
						 gboolean	    *var);
static void st_menubar_view_toggled_h		(GtkCheckMenuItem   *item,
						 gpointer	    data);
static void st_menubar_init_toolbar_style_item	(STMenubar	    *menubar,
						 const char	    *path,
						 STToolbarStyle	    style);
static void st_menubar_toolbar_style_toggled_h	(GtkCheckMenuItem   *item,
						 gpointer	    data);
static void st_menubar_init_toolbar_size_item	(STMenubar	    *menubar,
						 const char	    *path,
						 STToolbarSize	    size);
static void st_menubar_toolbar_size_toggled_h	(GtkCheckMenuItem   *item,
						 gpointer	    data);
static void st_menubar_update_directories_menu	(STMenubar	    *menubar,
						 STBrowserTab	    *tab);
static void st_menubar_init_columns_menu	(STMenubar	    *menubar);
static void st_menubar_update_columns_menu	(STMenubar	    *menubar,
						 STBrowserTab	    *tab);

/*** implementation **********************************************************/

GType
st_menubar_get_type (void)
{
  static GType menubar_type = 0;
  
  if (! menubar_type)
    {
      static const GTypeInfo menubar_info = {
	sizeof(STMenubarClass),
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	sizeof(STMenubar),
	0,
	(GInstanceInitFunc) st_menubar_init,
      };
      
      menubar_type = g_type_register_static(GTK_TYPE_MENU_BAR,
					    "STMenubar",
					    &menubar_info,
					    0);
    }

  return menubar_type;
}

static void
st_menubar_init (STMenubar *menubar)
{
  menubar->factory = NULL;

  menubar->stop_item = NULL;
  menubar->refresh_item = NULL;
  menubar->directory_homepage_item = NULL;
  menubar->columns_item = NULL;

  menubar->stream_items = NULL;
}

GtkWidget *
st_menubar_new (GtkAccelGroup *accel_group)
{
  GtkItemFactory *factory;
  STMenubar *menubar;

  static GtkItemFactoryEntry entries[] = {
    {
      N_("/_Stream"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      "/Stream/separator1",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/Stream/_Quit"),			NULL,
      st_shell_quit,				0,
      "<StockItem>",				GTK_STOCK_QUIT
    },
    {
      N_("/_Edit"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/Edit/_Preferences"),			NULL,
      st_shell_present_preferences,		0,
      "<StockItem>",				GTK_STOCK_PREFERENCES
    },
    {
      N_("/_View"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/View/_Toolbar"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      N_("/View/Ta_bs"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      N_("/View/Tab Ic_ons"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      N_("/View/St_atusbar"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      "/View/separator1",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Toolbar Styl_e"),		NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/View/Toolbar Style/_Desktop Default"), NULL,
      NULL,					0,
      "<RadioItem>",				NULL
    },
    {
      "/View/Toolbar Style/separator1",		NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Toolbar Style/I_cons Only"),	NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Style/_Text Only"),	NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Style/Text Belo_w Icons"), NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Style/Text Be_side Icons"), NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Si_ze"),		NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/View/Toolbar Size/_Desktop Default"), NULL,
      NULL,					0,
      "<RadioItem>",				NULL
    },
    {
      "/View/Toolbar Size/separator1",		NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Toolbar Size/_Small"),		NULL,
      NULL,					0,
      N_("/View/Toolbar Size/Desktop Default"),	NULL
    },
    {
      N_("/View/Toolbar Size/_Large"),		NULL,
      NULL,					0,
      N_("/View/Toolbar Size/Desktop Default"),	NULL
    },
    {
      "/View/separator2",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Stream _Columns"),		NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      "/View/separator3",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/_Stop"),			"Escape",
      st_shell_stop,				0,
      "<StockItem>",				GTK_STOCK_STOP
    },
    {
      N_("/View/_Refresh"),			"<Control>R",
      st_shell_refresh,				0,
      "<StockItem>",				GTK_STOCK_REFRESH
    },
    {
      N_("/_Directories"),			NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/Directories/_Previous"),		NULL,
      st_shell_previous_tab,			0,
      "<StockItem>",				GTK_STOCK_GO_BACK
    },
    {
      N_("/Directories/_Next"),			NULL,
      st_shell_next_tab,			0,
      "<StockItem>",				GTK_STOCK_GO_FORWARD
    },
    {
      "/Directories/separator1",		NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/Directories/_Homepage of Selected Directory"), NULL,
      st_shell_visit_directory_homepage,	0,
      "<StockItem>",				GTK_STOCK_HOME
    },
    {
      N_("/_Help"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/Help/_Contents"),			"F1",
      st_shell_help,				0,
      "<StockItem>",				GTK_STOCK_HELP
    },
    {
      N_("/Help/_Homepage"),			NULL,
      st_shell_homepage,			0,
      "<StockItem>",				GTK_STOCK_HOME
    },
    {
      "/Help/separator1",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/Help/_About"),			NULL,
      st_shell_present_about_dialog,		0,
      "<StockItem>",				GTK_STOCK_DIALOG_INFO
    }
  };

  factory = gtk_item_factory_new(ST_TYPE_MENUBAR,
				 "<streamtuner-Browser>",
				 accel_group);

  gtk_item_factory_set_translate_func(factory,
				      sgtk_translate_func,
				      NULL,
				      NULL);

  gtk_item_factory_create_items(factory, G_N_ELEMENTS(entries), entries, NULL);
  menubar = ST_MENUBAR(gtk_item_factory_get_widget(factory, "<streamtuner-Browser>"));

  menubar->accel_group = accel_group;
  menubar->factory = factory;

  menubar->stop_item = gtk_item_factory_get_item(factory, N_("/View/Stop"));
  menubar->refresh_item = gtk_item_factory_get_item(factory, N_("/View/Refresh"));
  menubar->directory_homepage_item = gtk_item_factory_get_item(factory, N_("/Directories/Homepage of Selected Directory"));
  menubar->columns_item = gtk_item_factory_get_item(factory, N_("/View/Stream Columns"));

  st_menubar_complete_stream_menu(menubar);

  st_menubar_init_columns_menu(menubar);

  st_menubar_init_directories_menu(menubar);
  st_menubar_complete_directories_menu(menubar);

  st_menubar_init_view_item(menubar, N_("/View/Toolbar"), &st_settings.view_toolbar);
  st_menubar_init_view_item(menubar, N_("/View/Tabs"), &st_settings.view_tabs);
  st_menubar_init_view_item(menubar, N_("/View/Tab Icons"), &st_settings.view_tab_icons);
  st_menubar_init_view_item(menubar, N_("/View/Statusbar"), &st_settings.view_statusbar);

  st_menubar_init_toolbar_style_item(menubar, N_("/View/Toolbar Style/Desktop Default"), ST_TOOLBAR_STYLE_GTK);
  st_menubar_init_toolbar_style_item(menubar, N_("/View/Toolbar Style/Icons Only"), GTK_TOOLBAR_ICONS);
  st_menubar_init_toolbar_style_item(menubar, N_("/View/Toolbar Style/Text Only"), GTK_TOOLBAR_TEXT);
  st_menubar_init_toolbar_style_item(menubar, N_("/View/Toolbar Style/Text Below Icons"), GTK_TOOLBAR_BOTH);
  st_menubar_init_toolbar_style_item(menubar, N_("/View/Toolbar Style/Text Beside Icons"), GTK_TOOLBAR_BOTH_HORIZ);

  st_menubar_init_toolbar_size_item(menubar, N_("/View/Toolbar Size/Desktop Default"), ST_TOOLBAR_SIZE_GTK);
  st_menubar_init_toolbar_size_item(menubar, N_("/View/Toolbar Size/Small"), GTK_ICON_SIZE_SMALL_TOOLBAR);
  st_menubar_init_toolbar_size_item(menubar, N_("/View/Toolbar Size/Large"), GTK_ICON_SIZE_LARGE_TOOLBAR);

  return GTK_WIDGET(menubar);
}

static void
st_menubar_complete_stream_menu (STMenubar *menubar)
{
  GtkWidget *menu;
  char *accel_path;

  g_return_if_fail(ST_IS_MENUBAR(menubar));

  menu = gtk_item_factory_get_widget(menubar->factory, N_("/Stream"));
  g_return_if_fail(menu != NULL);

  menubar->stream_items = st_stream_menu_items_new(menubar->accel_group);
  st_stream_menu_items_prepend_to_shell(menubar->stream_items, GTK_MENU_SHELL(menu));

  accel_path = g_strdup_printf("<streamtuner-Browser>/Stream/%s", _("Tune in"));
  gtk_accel_map_add_entry(accel_path, GDK_T, GDK_CONTROL_MASK);
  g_free(accel_path);
}

static void
st_menubar_init_directories_menu (STMenubar *menubar)
{
  g_return_if_fail(ST_IS_MENUBAR(menubar));

  if (g_slist_length(st_handlers_list) <= 1)
    {
      GtkWidget *previous;
      GtkWidget *next;

      previous = gtk_item_factory_get_widget(menubar->factory, N_("/Directories/Previous"));
      g_return_if_fail(previous != NULL);

      next = gtk_item_factory_get_widget(menubar->factory, N_("/Directories/Next"));
      g_return_if_fail(next != NULL);

      gtk_widget_set_sensitive(previous, FALSE);
      gtk_widget_set_sensitive(next, FALSE);
    }
}

static void
st_menubar_complete_directories_menu (STMenubar *menubar)
{
  GtkWidget *menu;

  g_return_if_fail(ST_IS_MENUBAR(menubar));

  menu = gtk_item_factory_get_widget(menubar->factory, N_("/Directories"));
  g_return_if_fail(menu != NULL);

  if (st_handlers_list)
    {
      GtkWidget *separator;
      GSList *l;
      GSList *group = NULL;

      separator = gtk_separator_menu_item_new();
      gtk_widget_show(separator);
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), separator);

      SG_LIST_FOREACH(l, st_handlers_list)
	{
	  STHandler *handler = l->data;
	  GtkWidget *item;
	  guint hid;

	  item = gtk_radio_menu_item_new_with_label(group, handler->label);
	  group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));

	  gtk_widget_show(item);
	  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	  hid = g_signal_connect(G_OBJECT(item),
				 "toggled",
				 G_CALLBACK(st_menubar_directories_toggled_h),
				 handler);

	  /* to be used by st_menubar_set_tab() */
	  g_object_set_data(G_OBJECT(item), "handler", handler);
	  g_object_set_data(G_OBJECT(item), "hid", GINT_TO_POINTER(hid));

	  menubar->directories_items = g_slist_append(menubar->directories_items, item);
	}
    }
}

static void
st_menubar_directories_toggled_h (GtkCheckMenuItem *item, gpointer data)
{
  if (gtk_check_menu_item_get_active(item))
    {
      STHandler *handler = data;

      st_shell_set_handler(handler);
    }
}

static void
st_menubar_init_view_item (STMenubar *menubar,
			   const char *path,
			   gboolean *var)
{
  GtkWidget *item;

  g_return_if_fail(ST_IS_MENUBAR(menubar));

  item = gtk_item_factory_get_widget(menubar->factory, path);
  g_return_if_fail(item != NULL);

  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), *var);

  g_signal_connect(G_OBJECT(item),
		   "toggled",
		   G_CALLBACK(st_menubar_view_toggled_h),
		   var);
}

static void
st_menubar_view_toggled_h (GtkCheckMenuItem *item, gpointer data)
{
  gboolean *var = data;

  g_return_if_fail(var != NULL);

  *var = gtk_check_menu_item_get_active(item);
  st_shell_visibility_changed();
}

static void
st_menubar_init_toolbar_style_item (STMenubar *menubar,
				    const char *path,
				    STToolbarStyle style)
{
  GtkWidget *item;

  g_return_if_fail(ST_IS_MENUBAR(menubar));

  item = gtk_item_factory_get_widget(menubar->factory, path);
  g_return_if_fail(item != NULL);

  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item),
				 st_settings.toolbar_style == style);

  g_signal_connect(G_OBJECT(item),
		   "toggled",
		   G_CALLBACK(st_menubar_toolbar_style_toggled_h),
		   GINT_TO_POINTER(style));
}

static void
st_menubar_toolbar_style_toggled_h (GtkCheckMenuItem *item, gpointer data)
{
  if (gtk_check_menu_item_get_active(item))
    {
      STToolbarStyle style = GPOINTER_TO_INT(data);

      st_settings.toolbar_style = style;
      st_shell_toolbar_style_changed();
    }
}

static void
st_menubar_init_toolbar_size_item (STMenubar *menubar,
				   const char *path,
				   STToolbarSize size)
{
  GtkWidget *item;

  g_return_if_fail(ST_IS_MENUBAR(menubar));

  item = gtk_item_factory_get_widget(menubar->factory, path);
  g_return_if_fail(item != NULL);

  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item),
				 st_settings.toolbar_size == size);

  g_signal_connect(G_OBJECT(item),
		   "toggled",
		   G_CALLBACK(st_menubar_toolbar_size_toggled_h),
		   GINT_TO_POINTER(size));
}

static void
st_menubar_toolbar_size_toggled_h (GtkCheckMenuItem *item, gpointer data)
{
  if (gtk_check_menu_item_get_active(item))
    {
      STToolbarSize size = GPOINTER_TO_INT(data);

      st_settings.toolbar_size = size;
      st_shell_toolbar_size_changed();
    }
}

void
st_menubar_update_sensitivity (STMenubar *menubar)
{
  g_return_if_fail(ST_IS_MENUBAR(menubar));

  st_stream_menu_items_update_sensitivity(menubar->stream_items);

  gtk_widget_set_sensitive(menubar->stop_item, st_shell_can_stop());
  gtk_widget_set_sensitive(menubar->refresh_item, st_shell_can_refresh());
  gtk_widget_set_sensitive(menubar->directory_homepage_item, st_shell_can_visit_directory_homepage());
}

void
st_menubar_set_tab (STMenubar *menubar, STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_MENUBAR(menubar));
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  st_menubar_update_directories_menu(menubar, tab);
  st_menubar_update_columns_menu(menubar, tab);
}

static void
st_menubar_init_columns_menu (STMenubar *menubar)
{
  /*
   * The item will be made sensitive later on if a submenu is attached
   * to it. For now we make it unsensitive.
   */
  gtk_widget_set_sensitive(menubar->columns_item, FALSE);
}

static void
st_menubar_update_columns_menu (STMenubar *menubar, STBrowserTab *tab)
{
  GtkWidget *menu;

  g_return_if_fail(ST_IS_MENUBAR(menubar));
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  menu = ST_STREAM_VIEW(tab->stream_view)->columns_menu;
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(menubar->columns_item), menu);
  gtk_widget_set_sensitive(menubar->columns_item, TRUE);
}

static void
st_menubar_update_directories_menu (STMenubar *menubar, STBrowserTab *tab)
{
  GSList *l;

  g_return_if_fail(ST_IS_MENUBAR(menubar));
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  SG_LIST_FOREACH(l, menubar->directories_items)
    {
      GtkWidget *item = l->data;
      STHandler *handler;
      guint hid;
      
      handler = g_object_get_data(G_OBJECT(item), "handler");
      g_return_if_fail(handler != NULL);
      
      hid = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(item), "hid"));
      
      if (handler == tab->handler)
	{
	  g_signal_handler_block(item, hid);
	  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
	  g_signal_handler_unblock(item, hid);
	  
	  break;
	}
    }
}
