/*
 * $Id: st-browser-window.c,v 1.42.2.2 2004/02/07 18:37:12 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include "art/streamtuner.h"
#include "sg-util.h"
#include "st-browser-tab.h"
#include "st-browser-window.h"
#include "st-category-view.h"
#include "st-handler.h"
#include "st-handlers.h"
#include "st-menubar.h"
#include "st-toolbar.h"
#include "st-settings.h"
#include "st-shell.h"
#include "st-statusbar.h"
#include "st-stream-view.h"
#include "st-dialog.h"

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_browser_window_class_init	(STBrowserWindowClass *class);
static void st_browser_window_init		(STBrowserWindow *window);
static void st_browser_window_unrealize		(GtkWidget *widget);

static void st_browser_window_construct_menubar   (STBrowserWindow *window);
static void st_browser_window_construct_toolbar   (STBrowserWindow *window);
static void st_browser_window_construct_notebook  (STBrowserWindow *window);
static void st_browser_window_construct_tabs      (STBrowserWindow *window);
static void st_browser_window_construct_statusbox (STBrowserWindow *window);

static gboolean st_browser_window_delete_event_h  (GtkWidget	*widget,
						   GdkEvent	*event,
						   gpointer	data);
static void	st_browser_window_switch_page_h   (GtkNotebook	*notebook,
						   GtkNotebookPage *page,
						   guint	page_num,
						   gpointer	data);

static void	     st_browser_window_set_tab    (STBrowserWindow  *window,
						   STBrowserTab     *tab);
static STBrowserTab  *st_browser_window_find_tab  (STBrowserWindow  *window,
						   STHandler        *handler);

/*** implementation **********************************************************/

GType
st_browser_window_get_type (void)
{
  static GType browser_window_type = 0;
  
  if (! browser_window_type)
    {
      static const GTypeInfo browser_window_info = {
	sizeof(STBrowserWindowClass),
	NULL,
	NULL,
	(GClassInitFunc) st_browser_window_class_init,
	NULL,
	NULL,
	sizeof(STBrowserWindow),
	0,
	(GInstanceInitFunc) st_browser_window_init,
      };
      
      browser_window_type = g_type_register_static(GTK_TYPE_WINDOW,
						   "STBrowserWindow",
						   &browser_window_info,
						   0);
    }

  return browser_window_type;
}

static void
st_browser_window_class_init (STBrowserWindowClass *class)
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  widget_class->unrealize = st_browser_window_unrealize;
}

static void
st_browser_window_init (STBrowserWindow *window)
{
  GdkPixbuf *icon;
  
  window->tab = NULL;

  window->accel_group = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(window), window->accel_group);

  icon = gdk_pixbuf_new_from_inline(sizeof(art_streamtuner),
				    art_streamtuner,
				    FALSE,
				    NULL);
  g_assert(icon != NULL);
  gtk_window_set_icon(GTK_WINDOW(window), icon);

  window->box = gtk_vbox_new(FALSE, 0);

  g_signal_connect(G_OBJECT(window), "delete-event",
		   G_CALLBACK(st_browser_window_delete_event_h), NULL);

  st_browser_window_construct_menubar(window);
  st_browser_window_construct_toolbar(window);
  st_browser_window_construct_notebook(window);
  st_browser_window_construct_statusbox(window);

  st_browser_window_update_visibility(window);
  st_browser_window_update_toolbar_style(window);
  st_browser_window_update_toolbar_size(window);

  gtk_container_add(GTK_CONTAINER(window), window->box);
  gtk_widget_show(window->box);

  /* restore geometry from settings */
  gtk_window_set_default_size(GTK_WINDOW(window),
			      st_settings.main_window_width,
			      st_settings.main_window_height);
}

static void
st_browser_window_unrealize (GtkWidget *widget)
{
  STBrowserWindow *window = ST_BROWSER_WINDOW(widget);

  if (window->tab)
    {
      g_free(st_settings.selected_handler_name);
      st_settings.selected_handler_name = g_strdup(window->tab->handler->name);
    }
  
  gtk_window_get_size(GTK_WINDOW(window),
		      &st_settings.main_window_width,
		      &st_settings.main_window_height);

  GTK_WIDGET_CLASS(parent_class)->unrealize(widget);
}

static gboolean
st_browser_window_delete_event_h (GtkWidget *widget,
				  GdkEvent *event,
				  gpointer data)
{
  st_shell_quit();

  return TRUE;
}

static void
st_browser_window_construct_menubar (STBrowserWindow *window)
{

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  window->menubar = st_menubar_new(window->accel_group);

  gtk_widget_show(window->menubar);
  gtk_box_pack_start(GTK_BOX(window->box), window->menubar, FALSE, FALSE, 0);
}

static void
st_browser_window_construct_toolbar (STBrowserWindow *window)
{
  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  window->toolbar = st_toolbar_new();

  gtk_widget_show(window->toolbar);
  gtk_box_pack_start(GTK_BOX(window->box), window->toolbar, FALSE, FALSE, 0);
}

static void
st_browser_window_construct_notebook (STBrowserWindow *window)
{
  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  window->notebook = gtk_notebook_new();
  gtk_notebook_popup_enable(GTK_NOTEBOOK(window->notebook));

  st_browser_window_construct_tabs(window);

  window->switch_page_id = 
    g_signal_connect(G_OBJECT(window->notebook), "switch-page",
		     G_CALLBACK(st_browser_window_switch_page_h), window);

  gtk_widget_show(window->notebook);
  gtk_box_pack_start(GTK_BOX(window->box), window->notebook, TRUE, TRUE, 0);
}

static void
st_browser_window_switch_page_h (GtkNotebook *notebook,
				 GtkNotebookPage *page,
				 guint page_num,
				 gpointer data)
{
  STBrowserWindow *window = data;
  GtkWidget *tab;
  
  tab = gtk_notebook_get_nth_page(notebook, page_num);
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  st_browser_window_set_tab(window, ST_BROWSER_TAB(tab));
}

static void
st_browser_window_construct_tabs (STBrowserWindow *window)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  window->tabs = NULL;

  SG_LIST_FOREACH(l, st_handlers_list)
    {
      STHandler *handler = l->data;
      GtkWidget *tab;
      GtkWidget *tab_label;
      GtkWidget *menu_label;

      tab = st_browser_tab_new(GTK_WIDGET(window), handler);
      tab_label = st_browser_tab_get_label(ST_BROWSER_TAB(tab));
      menu_label = st_browser_tab_get_label(ST_BROWSER_TAB(tab));
      
      gtk_widget_show(tab);
      gtk_widget_show(tab_label);
      gtk_widget_show(menu_label);

      window->tabs = g_slist_append(window->tabs, tab);
      gtk_notebook_append_page_menu(GTK_NOTEBOOK(window->notebook),
				    tab,
				    tab_label,
				    menu_label);
    }
}

static void
st_browser_window_construct_statusbox (STBrowserWindow *window)
{
  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  window->statusbox = gtk_event_box_new();

  gtk_widget_show(window->statusbox);
  gtk_box_pack_start(GTK_BOX(window->box), window->statusbox, FALSE, FALSE, 0);
}

void
st_browser_window_update_visibility (STBrowserWindow *window)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  g_object_set(window->toolbar, "visible", st_settings.view_toolbar, NULL);
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(window->notebook), st_settings.view_tabs);
  SG_LIST_FOREACH(l, window->tabs)
    st_browser_tab_set_label_icon_visible(ST_BROWSER_TAB(l->data), st_settings.view_tab_icons);
  g_object_set(window->statusbox, "visible", st_settings.view_statusbar, NULL);
}

void
st_browser_window_update_toolbar_style (STBrowserWindow *window)
{
  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  st_toolbar_set_style(ST_TOOLBAR(window->toolbar), st_settings.toolbar_style);
}

void
st_browser_window_update_toolbar_size (STBrowserWindow *window)
{
  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  st_toolbar_set_size(ST_TOOLBAR(window->toolbar), st_settings.toolbar_size);
}

GtkWidget *
st_browser_window_new (void)
{
  return g_object_new(ST_TYPE_BROWSER_WINDOW,
		      "type", GTK_WINDOW_TOPLEVEL,
		      NULL);
}

void
st_browser_window_update_title (STBrowserWindow *window)
{
  GString *title;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  if (window->tab)
    {
      g_return_if_fail(window->tab->handler->selected_category != NULL);

      title = g_string_new(window->tab->handler->label);
      g_string_append_printf(title, ", %s", window->tab->handler->selected_category->label);
    }
  else
    title = g_string_new("streamtuner");
  
  gtk_window_set_title(GTK_WINDOW(window), title->str);
  g_string_free(title, TRUE);
}

void
st_browser_window_update_sensitivity (STBrowserWindow *window)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  st_toolbar_update_sensitivity(ST_TOOLBAR(window->toolbar));
  st_menubar_update_sensitivity(ST_MENUBAR(window->menubar));

  SG_LIST_FOREACH(l, window->tabs)
    {
      STBrowserTab *tab = l->data;
      st_browser_tab_update_sensitivity(tab);
    }
}

static STBrowserTab *
st_browser_window_find_tab (STBrowserWindow *window, STHandler *handler)
{
  GSList *l;

  SG_LIST_FOREACH(l, window->tabs)
    {
      STBrowserTab *tab = l->data;
      
      if (tab->handler == handler)
	return tab;
    }
  
  return NULL;
}

void
st_browser_window_set_tab_from_handler (STBrowserWindow *window,
					STHandler *handler)
{
  STBrowserTab *tab;
  int num;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));
  g_return_if_fail(handler != NULL);
  
  tab = st_browser_window_find_tab(window, handler);
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  num = gtk_notebook_page_num(GTK_NOTEBOOK(window->notebook),
			      GTK_WIDGET(tab));
  g_return_if_fail(num != -1);

  g_signal_handler_block(window->notebook, window->switch_page_id);
  gtk_notebook_set_page(GTK_NOTEBOOK(window->notebook), num);
  g_signal_handler_unblock(window->notebook, window->switch_page_id);

  st_browser_window_set_tab(window, tab);
}

static void
st_browser_window_set_tab (STBrowserWindow *window, STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  window->tab = tab;

  st_browser_window_update_title(window);
  
  st_toolbar_set_handler(ST_TOOLBAR(window->toolbar), tab->handler);

  if (GTK_BIN(window->statusbox)->child)
    gtk_container_remove(GTK_CONTAINER(window->statusbox), GTK_BIN(window->statusbox)->child);

  gtk_container_add(GTK_CONTAINER(window->statusbox), tab->statusbar);

  /* work-around a bug in the Bluecurve theme engine */
  gtk_widget_realize(ST_STATUSBAR(tab->statusbar)->progress_bar);

  if ((! tab->refreshing) && st_browser_tab_is_empty(tab))
    st_browser_tab_update(tab);
  
  st_menubar_set_tab(ST_MENUBAR(window->menubar), tab);
  st_browser_window_update_sensitivity(window);

  st_shell_tab_selected();
}

void
st_browser_window_previous_tab (STBrowserWindow *window)
{
  gint page;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(window->notebook));

  if (--page < 0)
    {
      gint npages;

      /*
       * We don't use gtk_notebook_get_n_pages() in order to stay
       * compatible with GTK+ 2.0.
       */
      npages = g_list_length(GTK_NOTEBOOK(window->notebook)->children);
      page = npages - 1;
    }

  gtk_notebook_set_current_page(GTK_NOTEBOOK(window->notebook), page);
}

void
st_browser_window_next_tab (STBrowserWindow *window)
{
  gint page;
  gint npages;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(window->notebook));

  /*
   * We don't use gtk_notebook_get_n_pages() in order to stay
   * compatible with GTK+ 2.0.
   */
  npages = g_list_length(GTK_NOTEBOOK(window->notebook)->children);

  if (++page == npages)
    page = 0;

  gtk_notebook_set_current_page(GTK_NOTEBOOK(window->notebook), page);
}

void
st_browser_window_category_selected (STBrowserWindow *window, STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  if (tab == window->tab)
    st_browser_window_update_sensitivity(window);
}

void
st_browser_window_abort_all (STBrowserWindow *window)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_WINDOW(window));

  SG_LIST_FOREACH(l, window->tabs)
    {
      STBrowserTab *tab = l->data;
      
      if (tab->refreshing)
	st_browser_tab_abort(tab);
    }
}
