/*
 * $Id: st-stream-properties-dialog.c,v 1.36 2004/01/28 14:03:17 jylefort Exp $
 *
 * Copyright (c) 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-util.h"
#include "st-stream-properties-dialog.h"
#include "st-handler.h"
#include "st-settings.h"
#include "sgtk-hig.h"
#include "sgtk-handlers.h"

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_stream_properties_dialog_class_init	(STStreamPropertiesDialogClass *class);
static void st_stream_properties_dialog_init		(STStreamPropertiesDialog *dialog);
static void st_stream_properties_dialog_unrealize	(GtkWidget  *widget);

static void st_stream_properties_dialog_construct_table	(STStreamPropertiesDialog *dialog);
static void st_stream_properties_dialog_update_entries	(STStreamPropertiesDialog *dialog);

/*** implementation **********************************************************/

GType
st_stream_properties_dialog_get_type (void)
{
  static GType stream_properties_dialog_type = 0;
  
  if (! stream_properties_dialog_type)
    {
      static const GTypeInfo stream_properties_dialog_info = {
	sizeof(STStreamPropertiesDialogClass),
	NULL,
	NULL,
	(GClassInitFunc) st_stream_properties_dialog_class_init,
	NULL,
	NULL,
	sizeof(STStreamPropertiesDialog),
	0,
	(GInstanceInitFunc) st_stream_properties_dialog_init,
      };
      
      stream_properties_dialog_type = g_type_register_static(SGTK_TYPE_DIALOG,
							     "STStreamPropertiesDialog",
							     &stream_properties_dialog_info,
							     0);
    }

  return stream_properties_dialog_type;
}

static void
st_stream_properties_dialog_class_init (STStreamPropertiesDialogClass *class)
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  widget_class->unrealize = st_stream_properties_dialog_unrealize;
}

static void
st_stream_properties_dialog_init (STStreamPropertiesDialog *dialog)
{
  GtkWidget *close;

  dialog->handler = NULL;
  dialog->stream = NULL;

  dialog->table = NULL;
  dialog->entries = NULL;

  close = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
  gtk_window_set_focus(GTK_WINDOW(dialog), close);

  gtk_window_set_title(GTK_WINDOW(dialog), _("Stream Properties"));

  gtk_window_set_default_size(GTK_WINDOW(dialog),
			      st_settings.stream_properties_window_width,
			      -1);
}

static void
st_stream_properties_dialog_unrealize (GtkWidget *widget)
{
  GtkWindow *window = GTK_WINDOW(widget);
  int height;			/* dummy */

  gtk_window_get_size(window,
		      &st_settings.stream_properties_window_width,
		      &height);

  GTK_WIDGET_CLASS(parent_class)->unrealize(widget);
}

static void
st_stream_properties_dialog_construct_table (STStreamPropertiesDialog *dialog)
{
  GSList *l;
  int fields = 0;
  int vi = 0;
  int width;
  int height;
  GtkWidget *first_entry = NULL;
  GtkWidget *prev_entry = NULL;

  g_return_if_fail(ST_IS_STREAM_PROPERTIES_DIALOG(dialog));
  g_return_if_fail(dialog->handler != NULL);

  if (dialog->table)
    gtk_container_remove(GTK_CONTAINER(SGTK_DIALOG(dialog)->contents), dialog->table);
  
  if (dialog->entries)
    {
      g_slist_free(dialog->entries);
      dialog->entries = NULL;
    }
  
  SG_LIST_FOREACH(l, dialog->handler->fields)
    {
      STHandlerField *field = l->data;
      if (field->visible)
	fields++;
    }

  dialog->table = gtk_table_new(fields, 2, FALSE);
 
  gtk_table_set_row_spacings(GTK_TABLE(dialog->table), SGTK_HIG_CONTROL_SPACING);
  gtk_table_set_col_spacings(GTK_TABLE(dialog->table), SGTK_HIG_CONTROL_LABEL_SPACING);

  SG_LIST_FOREACH(l, dialog->handler->fields)
    {
      STHandlerField *field = l->data;
     
      if (field->visible)
	{
	  GtkWidget *label;
	  GtkWidget *entry;
	  char *str;

	  str = g_strdup_printf("%s:", field->label);
	  label = gtk_label_new(str);
	  g_free(str);

	  gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

	  entry = gtk_entry_new();
	  gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);

	  g_object_set_data(G_OBJECT(entry), "field", field);
	  dialog->entries = g_slist_append(dialog->entries, entry);

	  if (! first_entry)
	    first_entry = entry;

	  if (prev_entry)
	    g_signal_connect(G_OBJECT(prev_entry), "activate",
			     G_CALLBACK(sgtk_focus_entry_h), entry);

	  prev_entry = entry;

	  gtk_widget_show(label);
	  gtk_widget_show(entry);

	  gtk_table_attach(GTK_TABLE(dialog->table),
			   label,
			   0, 1, vi, vi + 1,
			   GTK_FILL,
			   (GtkAttachOptions) 0,
			   0,
			   0);
	  gtk_table_attach(GTK_TABLE(dialog->table),
			   entry,
			   1, 2, vi, vi + 1,
			   GTK_EXPAND | GTK_FILL,
			   (GtkAttachOptions) 0,
			   0,
			   0);
	  
	  vi++;
	}
    }
  
  if (prev_entry)
    g_signal_connect(G_OBJECT(prev_entry), "activate",
		     G_CALLBACK(sgtk_focus_entry_h), first_entry);

  gtk_widget_show(dialog->table);
  gtk_container_add(GTK_CONTAINER(SGTK_DIALOG(dialog)->contents), dialog->table);

  /* force the dialog to recalculate its height */
  gtk_window_get_size(GTK_WINDOW(dialog), &width, &height);
  gtk_window_resize(GTK_WINDOW(dialog), width, 1);
}

static void
st_stream_properties_dialog_update_entries (STStreamPropertiesDialog *dialog)
{
  GSList *l;

  g_return_if_fail(ST_IS_STREAM_PROPERTIES_DIALOG(dialog));
  g_return_if_fail(dialog->handler != NULL);

  SG_LIST_FOREACH(l, dialog->entries)
    {
      GtkWidget *entry = l->data;
      STHandlerField *field;
      GValue value = { 0, };
      char *str = NULL;
      
      field = g_object_get_data(G_OBJECT(entry), "field");
      g_return_if_fail(field != NULL);
      
      if (dialog->stream)
	{
	  gboolean val;

	  st_handler_stream_field_get(dialog->handler,
				      dialog->stream,
				      field,
				      &value);
	  
	  switch (G_VALUE_TYPE(&value))
	    {
	    case G_TYPE_BOOLEAN:
	      val = g_value_get_boolean(&value);
	      str = g_strdup_printf("%s", val ? _("Yes") : _("No"));
	      break;

	    case G_TYPE_INT:
	      str = g_strdup_printf("%i", g_value_get_int(&value));
	      break;
	      
	    case G_TYPE_STRING:
	      str = g_value_dup_string(&value);
	      break;
	      
	    default:
	      g_assert_not_reached();
	    }
	  
	  g_value_unset(&value);
	}
      
      gtk_entry_set_text(GTK_ENTRY(entry), str ? str : "");
      g_free(str);
    }
}

GtkWidget *
st_stream_properties_dialog_new (GtkWindow *parent)
{
  STStreamPropertiesDialog *dialog;

  dialog = g_object_new(ST_TYPE_STREAM_PROPERTIES_DIALOG, NULL);

  if (parent)
    gtk_window_set_transient_for(GTK_WINDOW(dialog), parent);

  return GTK_WIDGET(dialog);
}

void
st_stream_properties_dialog_set_handler (STStreamPropertiesDialog *dialog,
					 STHandler *handler)
{
  g_return_if_fail(ST_IS_STREAM_PROPERTIES_DIALOG(dialog));

  dialog->handler = handler;
  st_stream_properties_dialog_construct_table(dialog);
}

void
st_stream_properties_dialog_set_stream (STStreamPropertiesDialog *dialog,
					STStream *stream)
{
  g_return_if_fail(ST_IS_STREAM_PROPERTIES_DIALOG(dialog));

  dialog->stream = stream;
  st_stream_properties_dialog_update_entries(dialog);
}
