# Copyright (c) 2004 Jean-Yves Lefort
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
#    may be used to endorse or promote products derived from this software
#    without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
# CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
# ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
# RealAudio is a registered trademark of RealNetworks, Inc.

import string, re, datetime, ST
from ST import _

### constants #################################################################

class FIELD:
    DATE, DESCRIPTION, URL = range(3)

BASIC_CH_HOME		= "http://www.basic.ch/"
BASIC_CH_ROOT		= "http://live.basic.ch/"

re_header_charset	= re.compile("^Content-Type: .*charset=(.*)")
re_body_charset		= re.compile('^<meta http-equiv="Content-Type" content=".*charset=(.*)"')
re_dj			= re.compile(r"javascript:openWindow\('(showtl.cfm\?showid=[0-9]+)'.*<b>(<font.*\">)?(.*?)(</font>)?</b>")
re_genre		= re.compile('(sans-serif"><b>|<font color="000000">)(.*?)<')
re_date			= re.compile(r">([0-9][0-9])\.([0-9][0-9])\.([0-9][0-9])<")
re_stream		= re.compile(r'a href="/(ram/.*?)".*?text-decoration:none">(.*)</a>')

### helpers ###################################################################

class struct:
    def __init__(self, **entries):
        self.__dict__.update(entries)

def body_convert (line, info):
    if not hasattr(info, "charset") and line == "</head>":
        # we got no charset info, fallback to ISO8859-1
        info.charset = "ISO8859-1"
    else:
        match = re_body_charset.match(line)
        if match:
            info.charset = match.group(1)

    if hasattr(info, "charset"):
        return unicode(line, info.charset).encode("utf8")
    else:
        return line

# Convert a 2-digits year to a 4-digits year, conforming to the POSIX
# or X/Open standard -- nice y2k69 bug ;)
def yy2yyyy (yy):
    if yy >= 69 and yy <= 99:
        return 1900 + yy
    else:
        return 2000 + yy

### transfer callbacks ########################################################

def header_cb (line, info):
    if not hasattr(info, "charset"):
        match = re_header_charset.match(line)
        if match:
            info.charset = match.group(1)

def categories_line_cb (line, info):
    line = body_convert(line, info)

    match = re_dj.search(line)
    if match is not None:
        if hasattr(info, "category"):
            handler.notice(_("found incomplete category"))

        info.category = ST.Category()
        info.category.name = match.group(1)
        info.category.url_postfix = match.group(1)
        info.category.label = string.capwords(ST.sgml_ref_expand(match.group(3)))
        info.categories.append(ST.CategoryNode(info.category))
    else:
        match = re_genre.search(line)
        if match is not None:
            if not hasattr(info, "category"):
                handler.notice(_("found misplaced genre"))
            else:
                info.category.label = info.category.label + " (" + ST.sgml_ref_expand(match.group(2)) + ")"
                del info.category

def streams_line_cb (line, info):
    line = body_convert(line, info)
    
    match = re_date.search(line)
    if match is not None:
        if hasattr(info, "stream"):
            handler.notice(_("found incomplete stream"))

        date = datetime.date(yy2yyyy(int(match.group(3))),
                             int(match.group(2)),
                             int(match.group(1)))

        info.stream = ST.Stream()
        info.stream.set_field(FIELD.DATE, date.strftime("%x"))
    else:
        match = re_stream.search(line)
        if match is not None:
            if not hasattr(info, "stream"):
                handler.notice(_("found misplaced stream"))
            else:
                info.stream.name = match.group(1)
                info.stream.set_field(FIELD.URL, BASIC_CH_ROOT + info.stream.name)
                info.stream.set_field(FIELD.DESCRIPTION, ST.sgml_ref_expand(match.group(2)))
                info.streams.append(info.stream)
                del info.stream

### handler implementation ####################################################

class BasicChHandler(ST.Handler):
    def __new__(cls):
        self = ST.Handler.__new__(cls, "basic.ch.py")

        self.label = "basic.ch"
        self.copyright = "Copyright \302\251 2004 Jean-Yves Lefort"
        self.description = _("basic.ch Internet Radio - Live and Archived DJ Mixes")
        self.home = BASIC_CH_HOME
        self.icon = ST.find_icon("basic.ch.png")

        category = ST.Category()
        category.name = "__main"
        category.label = _("Live")
        self.stock_categories = ST.CategoryNode(category)

        self.add_field(ST.HandlerField(FIELD.DATE,
                                       _("Date"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE))
        self.add_field(ST.HandlerField(FIELD.DESCRIPTION,
                                       _("Description"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE))
        self.add_field(ST.HandlerField(FIELD.URL,
                                       _("URL"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE
                                       | ST.HANDLER_FIELD_START_HIDDEN))

        return self

    def reload (self, category):
        session = ST.TransferSession()
        
        if not hasattr(self, "categories"):
            info = struct(categories = ST.CategoryNode())
            session.get_by_line(BASIC_CH_ROOT + "downtest.cfm",
                                header_cb = header_cb,
                                header_data = info,
                                body_cb = categories_line_cb,
                                body_data = info)
            self.categories = info.categories

        info = struct(streams = [])
        if category.url_postfix is None:
            # main category
            stream = ST.Stream()
            stream.name = "basic.ram"
            stream.set_field(FIELD.DATE, _("Now"))
            stream.set_field(FIELD.DESCRIPTION, _("Live stream"))
            stream.set_field(FIELD.URL, BASIC_CH_ROOT + stream.name)
            info.streams.append(stream)
        else:
            session.get_by_line(BASIC_CH_ROOT + category.url_postfix,
                                header_cb = header_cb,
                                header_data = info,
                                body_cb = streams_line_cb,
                                body_data = info)

        return (self.categories, info.streams)

    def stream_tune_in (self, stream):
        ST.action_run("play-ra", stream.get_field(FIELD.URL))
    
### initialization ############################################################

def init ():
    global handler

    if not ST.check_api_version(1, 1):
        raise RuntimeError, _("API version mismatch")
    
    ST.action_register("play-ra", _("Listen to a RealAudio%s stream") % ("\302\256"), "realplay %q")

    handler = BasicChHandler()
    ST.handlers_add(handler)

init()
