# Copyright (c) 2003, 2004 Jean-Yves Lefort
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
#    may be used to endorse or promote products derived from this software
#    without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
# CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
# ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

import re, os, ST
from ST import _

### constants #################################################################

class FIELD:
    TITLE, DESCRIPTION, HOMEPAGE = range(3)

GOOGLE_DIRECTORY_ROOT	= "http://directory.google.com"
CATEGORIES_URL_POSTFIX	= "/Top/Arts/Music/Sound_Files/MP3/Streaming/Stations/"

GOOGLE_STATIONS_HOME	= GOOGLE_DIRECTORY_ROOT + CATEGORIES_URL_POSTFIX

re_category	= re.compile('^<a href="(.*)">(<b>)?(.*?)(</b>)?</a>&nbsp;')
re_stream	= re.compile('^<td><font face="arial,sans-serif"><a href="(.*)">(.*)</a>')
re_description	= re.compile('^<br><font size=-1> (.*?)</font>')

### helpers ###################################################################

class struct:
    def __init__(self, **entries):
        self.__dict__.update(entries)

def streamtunerize (str):
    "Convert a string to UTF-8 and expand its SGML character references."

    # Google returns no charset information in the Content-Type HTTP
    # header, so we assume that the data is encoded in ISO8859-1
    return ST.sgml_ref_expand(unicode(str, "iso-8859-1").encode("utf8"))
    
### transfer callbacks ########################################################

def categories_line_cb (line, categories):
    match = re_category.match(line)
    if match is not None:
        category = ST.Category()

        category.name = streamtunerize(match.group(3))
        category.label = category.name

        href = streamtunerize(match.group(1))

        if href[0] == "/":
            category.url_postfix = href
        else:
            category.url_postfix = CATEGORIES_URL_POSTFIX + href
           
        categories.append(ST.CategoryNode(category))

def streams_line_cb (line, info):
    match = re_stream.match(line)
    if match is not None:
        if hasattr(info, "stream"):
            handler.notice(_("found incomplete stream"))

        info.stream = ST.Stream()
        info.stream.name = streamtunerize(match.group(1))
        info.stream.set_field(FIELD.HOMEPAGE, info.stream.name)
        info.stream.set_field(FIELD.TITLE, streamtunerize(match.group(2)))
    else:
        match = re_description.match(line)
        if match is not None:
            if not hasattr(info, "stream"):
                handler.notice(_("found misplaced description"))
            else:
                info.stream.set_field(FIELD.DESCRIPTION, streamtunerize(match.group(1)))
                info.streams.append(info.stream)
                del info.stream

### handler implementation ####################################################

class GoogleStationsHandler(ST.Handler):
    def __new__ (cls):
        self = ST.Handler.__new__(cls, "google-stations.py")

        self.label = _("Google Stations")
        self.copyright = "Copyright \302\251 2003, 2004 Jean-Yves Lefort"
        self.home = GOOGLE_STATIONS_HOME
        self.icon = ST.find_icon("google-stations.png")

        category = ST.Category()
        category.name = "__main"
        category.label = _("Internet radios")
        category.url_postfix = "/Top/Arts/Radio/Internet/"
        self.stock_categories = ST.CategoryNode(category)

        self.add_field(ST.HandlerField(FIELD.TITLE,
                                       _("Title"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE))
        self.add_field(ST.HandlerField(FIELD.DESCRIPTION,
                                       _("Description"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE))
        self.add_field(ST.HandlerField(FIELD.HOMEPAGE,
                                       _("Homepage"),
                                       ST.HANDLER_FIELD_TYPE_STRING,
                                       ST.HANDLER_FIELD_VISIBLE
                                       | ST.HANDLER_FIELD_START_HIDDEN))

        return self
        
    def reload (self, category):
        session = ST.TransferSession()

        if not hasattr(self, "categories"):
            categories = ST.CategoryNode()
            session.get_by_line(GOOGLE_DIRECTORY_ROOT + CATEGORIES_URL_POSTFIX,
                                body_cb = categories_line_cb,
                                body_data = categories)
            self.categories = categories
    
        info = struct(streams = [])
        if category.url_postfix is not None:
            session.get_by_line(GOOGLE_DIRECTORY_ROOT + category.url_postfix,
                                body_cb = streams_line_cb,
                                body_data = info)

        return (self.categories, info.streams)

    def stream_get_stock_field (self, stream, stock_field):
        if stock_field == ST.HANDLER_STOCK_FIELD_NAME:
            return stream.get_field(FIELD.TITLE)

    def stream_browse (self, stream):
    	ST.action_run("view-web", stream.get_field(FIELD.HOMEPAGE))

### initialization ############################################################

def init ():
    global handler
    
    if not ST.check_api_version(1, 1):
        raise RuntimeError, _("API version mismatch")

    ST.action_register("view-web", _("Open a web page"), "epiphany %q")

    handler = GoogleStationsHandler()
    ST.handlers_add(handler)

init()
